/*!
******************************************************************************

	@file	disk.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "disk.h"
#include "fs.h"
#ifndef _VSUN86_PCSIM
#include "mboot.h"
#include "usb.h"
#include "usb/class/msc.h"
#include "fdc.h"
#include "disk/floppy.h"
#else	//_VSUN86_PCSIM
#include "disk/vdisk.h"
#endif	//_VSUN86_PCSIM

static DISK_INFO disk_info[DISK_ID_MAX];
static u32 disk_num;

bool disk_init( void )
{
	memset( disk_info, 0, sizeof(disk_info) );
	disk_num = 0;

#ifndef _VSUN86_PCSIM
	if ( !ata_init() )
		return false;
#endif	//_VSUN86_PCSIM

	return true;
}

bool disk_start( void )
{
#ifndef _VSUN86_PCSIM
	const MBOOT_BOOT_DEVICE_INFO *boot_device = mboot_get_boot_device();
	if ( boot_device->drive == 0x00 )
	{	// 1st FDDからブートした場合、FDC経由でアクセスできるように登録する
		if ( !fdc_init() )
			return false;
		DISK_INFO *disk = &disk_info[0];
		disk->valid			= true;
		disk->ctrl			= DISK_CTRL_FLOPPY;
		disk->int13h_drive	= boot_device->drive;
		disk_num = 1;
		if ( !fs_probe( 0, FS_1ST_SECTOR_VBR ) )
			vmm_printf( VMM_WARNING, "DISK: fs_probe() failed.\n" );
	}

	// ATA接続のデバイスを初期化する
	if ( !ata_start() )
		return false;
#else	//_VSUN86_PCSIM
	if ( !vdisk_start() )
		return false;
#endif	//_VSUN86_PCSIM

	return true;
}

#ifndef _VSUN86_PCSIM
bool disk_register_scsi( SCSI_DEVICE *dev )
{
	if ( disk_num >= DISK_ID_MAX )
		return false;

	const u32 disk_id = disk_num++;
	DISK_INFO *disk = &disk_info[disk_id];
	disk->valid	= true;
	disk->ctrl	= DISK_CTRL_SCSI;
	memcpy( &disk->dev.scsi, dev, sizeof(SCSI_DEVICE) );

	// デバイス情報を取得する
	SCSI_STD_INQUIRY_DATA data;
	if ( !scsi_cmd_inquiry( &disk->dev.scsi, &data ) ) {
		vmm_printf( VMM_ERROR, "DISK(SCSI): INQUIRY failed.\n" );
		return false;
	}
	memcpy( disk->vendor_id,  data.vendor_id,  8  );
	memcpy( disk->product_id, data.product_id, 16 );
	memcpy( disk->revision,   data.revision,   4  );

	// ディスク容量を取得する
	SCSI_DISK_CAPACITY cap;
	if ( !scsi_cmd_read_capacity( &disk->dev.scsi, &cap ) ) {
		vmm_printf( VMM_ERROR, "DISK(SCSI): READ CAPACITY(10) failed.\n" );
		return false;
	}
	disk->last_block = cap.last_block;
	disk->block_len  = cap.block_len;

	// ファイルシステムを選定する
	// ★失敗してもとりあえず処理は続行する
	if ( !fs_probe( disk_id, FS_1ST_SECTOR_MBR ) )
		vmm_printf( VMM_WARNING, "DISK: fs_probe() failed.\n" );

	return true;
}
#endif	//!_VSUN86_PCSIM

#ifndef _VSUN86_PCSIM
bool disk_register_ata( ATA_DEVICE *dev, ATA_IDENTIFY_DEVICE_DATA *data )
{
	if ( disk_num >= DISK_ID_MAX )
		return false;

	const u32 disk_id = disk_num++;
	DISK_INFO *disk = &disk_info[disk_id];
	disk->valid	= true;
	disk->ctrl	= DISK_CTRL_ATA;
	memcpy( &disk->dev.ata, dev, sizeof(ATA_DEVICE) );

	// ディスク容量を取得する
	disk->last_block = *(u32 *)&data->w[100];
	disk->block_len  = 512;
	vmm_printf( VMM_DEBUG, "DISK: last_block = %08x\n", disk->last_block );

	// ファイルシステムを選定する
	// ★失敗してもとりあえず処理は続行する
	if ( !fs_probe( disk_id, FS_1ST_SECTOR_MBR ) )
		vmm_printf( VMM_WARNING, "DISK: fs_probe() failed.\n" );

	return true;
}
#endif	//!_VSUN86_PCSIM

#ifdef	_VSUN86_PCSIM
bool disk_register_vdisk( int vdisk_index, u64 *capacity )
{
	if ( disk_num >= DISK_ID_MAX )
		return false;

	const u32 disk_id = disk_num++;
	DISK_INFO *disk = &disk_info[disk_id];
	disk->valid	= true;
	disk->ctrl	= DISK_CTRL_VDISK;
	disk->dev.vdisk_index = vdisk_index;

	// ディスク容量を取得する
	if ( !vdisk_get_capacity( vdisk_index, capacity ) ) {
		vmm_printf( VMM_ERROR, "vdisk_get_capacity(%d) failed.\n", vdisk_index );
		return false;
	}
	disk->block_len	 = 512;
	disk->last_block = (u32)((*capacity / 512) - 1);

	// ファイルシステムを選定する
	// ★失敗してもとりあえず処理は続行する
	if ( vdisk_index < 2 ) {
		if ( !fs_probe( disk_id, FS_1ST_SECTOR_VBR ) )
			vmm_printf( VMM_WARNING, "DISK: fs_probe() failed.\n" );
	}
	else {
		if ( !fs_probe( disk_id, FS_1ST_SECTOR_MBR ) )
			vmm_printf( VMM_WARNING, "DISK: fs_probe() failed.\n" );
	}

	return true;
}
#endif	//_VSUN86_PCSIM

u32 disk_get_num( void )
{
	return disk_num;
}

bool disk_get_info( u32 disk_id, DISK_INFO *info )
{
	if ( disk_id >= disk_num )
		return false;

	if ( info == NULL )
		return false;

	memcpy( info, &disk_info[disk_id], sizeof(DISK_INFO) );
	return true;
}

bool disk_read( u32 disk_id, u32 block_addr, u16 block_len, void *buf, u32 buf_len )
{
	if ( disk_id >= disk_num )
		return false;

	if ( buf == NULL )
		return false;

	DISK_INFO *disk = &disk_info[disk_id];
	if ( !disk->valid )
		return false;

	if ( block_len * disk->block_len < buf_len )
		return false;

	switch ( disk->ctrl )
	{
#ifndef _VSUN86_PCSIM
	case DISK_CTRL_FLOPPY:
		return floppy_read( disk_id, block_addr, block_len, buf, buf_len );

	case DISK_CTRL_SCSI:
//		return scsi_cmd_read( &disk->dev.scsi, block_addr, block_len, buf, buf_len );
		{
			u8 *p = (u8 *)buf;
			u16 block_off = 0;
			while ( (block_off + 8) <= block_len ) {
				if ( !scsi_cmd_read( &disk->dev.scsi, block_addr + block_off, 8, &p[block_off<<9], 512*8 ) )
					return false;
				block_off += 8;
			}
			while ( (block_off + 4) <= block_len ) {
				if ( !scsi_cmd_read( &disk->dev.scsi, block_addr + block_off, 4, &p[block_off<<9], 512*4 ) )
					return false;
				block_off += 4;
			}
			while ( (block_off + 2) <= block_len ) {
				if ( !scsi_cmd_read( &disk->dev.scsi, block_addr + block_off, 2, &p[block_off<<9], 512*2 ) )
					return false;
				block_off += 2;
			}
			while ( (block_off + 1) <= block_len ) {
				if ( !scsi_cmd_read( &disk->dev.scsi, block_addr + block_off, 1, &p[block_off<<9], 512*1 ) )
					return false;
				block_off += 1;
			}
		}
		break;

	case DISK_CTRL_ATA:
//		return ata_cmd_read_sectors( &disk->dev.ata, block_addr, block_len, buf );
		{
			u8 *p = (u8 *)buf;
			u16 block_off = 0;
			while ( (block_off + 8) <= block_len ) {
				if ( !ata_cmd_read_sectors( &disk->dev.ata, block_addr + block_off, 8, &p[block_off<<9] ) )
					return false;
				block_off += 8;
			}
			while ( (block_off + 4) <= block_len ) {
				if ( !ata_cmd_read_sectors( &disk->dev.ata, block_addr + block_off, 4, &p[block_off<<9] ) )
					return false;
				block_off += 4;
			}
			while ( (block_off + 2) <= block_len ) {
				if ( !ata_cmd_read_sectors( &disk->dev.ata, block_addr + block_off, 2, &p[block_off<<9] ) )
					return false;
				block_off += 2;
			}
			while ( (block_off + 1) <= block_len ) {
				if ( !ata_cmd_read_sectors( &disk->dev.ata, block_addr + block_off, 1, &p[block_off<<9] ) )
					return false;
				block_off += 1;
			}
		}
		break;

#else	//_VSUN86_PCSIM

	case DISK_CTRL_VDISK:
		{
			u8 *p = (u8 *)buf;
			u16 block_off = 0;
			while ( (block_off + 8) <= block_len ) {
				if ( !vdisk_read( disk->dev.vdisk_index, block_addr + block_off, 8, &p[block_off<<9] ) )
					return false;
				block_off += 8;
			}
			while ( (block_off + 4) <= block_len ) {
				if ( !vdisk_read( disk->dev.vdisk_index, block_addr + block_off, 4, &p[block_off<<9] ) )
					return false;
				block_off += 4;
			}
			while ( (block_off + 2) <= block_len ) {
				if ( !vdisk_read( disk->dev.vdisk_index, block_addr + block_off, 2, &p[block_off<<9] ) )
					return false;
				block_off += 2;
			}
			while ( (block_off + 1) <= block_len ) {
				if ( !vdisk_read( disk->dev.vdisk_index, block_addr + block_off, 1, &p[block_off<<9] ) )
					return false;
				block_off += 1;
			}
		}
		break;

#endif	//_VSUN86_PCSIM

	default:
		return false;
	}

	return true;
}
