/*!
******************************************************************************

	@file	ata.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "ata.h"
#include "disk.h"
#include "io.h"
#include "irq.h"
#include "timer.h"
#include "printf.h"

static ATA_IO_PORTS ata_ports[2] =
{
	{
		IO_ATA0_DATA,
		IO_ATA0_ERROR,
		IO_ATA0_FEATURES,
		{ IO_ATA0_SECTOR_CNT },
		{ IO_ATA0_SECTOR_NUM },
		{ IO_ATA0_CYLINDER_L },
		{ IO_ATA0_CYLINDER_H },
		IO_ATA0_DEV_HEAD,
		IO_ATA0_STATUS,
		IO_ATA0_CMD,
		IO_ATA0_ALT_STATUS,
		IO_ATA0_DEV_CTRL,
	},
	{
		IO_ATA1_DATA,
		IO_ATA1_ERROR,
		IO_ATA1_FEATURES,
		{ IO_ATA1_SECTOR_CNT },
		{ IO_ATA1_SECTOR_NUM },
		{ IO_ATA1_CYLINDER_L },
		{ IO_ATA1_CYLINDER_H },
		IO_ATA1_DEV_HEAD,
		IO_ATA1_STATUS,
		IO_ATA1_CMD,
		IO_ATA1_ALT_STATUS,
		IO_ATA1_DEV_CTRL,
	}
};

static ATA_DEVICE ata_dev[4] =
{
	{ &ata_ports[0], 0, IRQ_ATA0, false },
	{ &ata_ports[0], 1, IRQ_ATA0, false },
	{ &ata_ports[1], 2, IRQ_ATA1, false },
	{ &ata_ports[1], 3, IRQ_ATA1, false }
};

static bool ata_irq_handler( u8 irq, void *p );

bool ata_init( void )
{
	return true;
}

bool ata_start( void )
{
	ATA_IDENTIFY_DEVICE_DATA data;
	memset( &data, 0, sizeof(data) );

	vmm_printf( VMM_INFO, "\nInitializing ATA Devices ...\n\n" );

	char model_number[40+1];
	for ( int i=0; i<4; i++ ) {
		ATA_DEVICE *dev = &ata_dev[i];
		u8 ctrl = DISK_CTRL_ATA;
		if ( !ata_cmd_identify_device( dev, &data ) ) {
			vmm_printf( VMM_DEBUG, "ata(%d): IDENTIFY DEVICE failed.\n", i );
			if ( !ata_cmd_identify_packet_device( dev, &data ) ) {
				vmm_printf( VMM_DEBUG, "ata(%d): IDENTIFY PACKET DEVICE failed.\n", i );
				vmm_printf( VMM_INFO, "%d:%d (No Device)\n", i >> 1, i & 0x01 );
				continue;
			}
			ctrl = DISK_CTRL_ATAPI;
		}
		for ( int n=0; n<20; n++ ) {
			model_number[(n<<1)+0] = (char)(data.model_number[n] >> 8);
			model_number[(n<<1)+1] = (char)(data.model_number[n]);
		}
		model_number[40] = '\0';
		vmm_printf( VMM_INFO, "%d:%d %s\n", i >> 1, i & 0x01, model_number );
		disk_register_ata( dev, &data );
	}
	vmm_printf( VMM_INFO, "\n" );

	return true;
}

bool ata_select_device( ATA_DEVICE *dev )
{
//	vmm_printf( VMM_DEBUG, "ata(%d): status=%02x (error=%02x)\n",
//				dev->id, inb( dev->ports->status ), inb( dev->ports->error ) );

	if ( !(inb( dev->ports->status ) & ATA_STATUS_ERR) )
		while ( inb( dev->ports->status ) & (ATA_STATUS_BSY | ATA_STATUS_DRQ) );

	outb( dev->ports->dev_head, (dev->id & 0x01) << ATA_DEV_SHIFT );
	timer_usleep( 1 );	// 400ns以上待つ
//	vmm_printf( VMM_DEBUG, "ata(%d): status=%02x (error=%02x)\n",
//				dev->id, inb( dev->ports->status ), inb( dev->ports->error ) );
	if ( inb( dev->ports->status ) & ATA_STATUS_ERR )
	{	// 機器が接続されていないっぽい？
		if ( inb( dev->ports->error ) == 0xFF )
			return false;
	}
	else
		while ( inb( dev->ports->status ) & (ATA_STATUS_BSY | ATA_STATUS_DRQ) );

	return true;
}

static bool ata_irq_handler( u8 irq, void *p )
{
	(void)irq;

	bool *finish = (bool *)p;
	*finish = true;

	return true;
}

bool ata_pio_read( ATA_DEVICE *dev, u8 cmd, void *buf, size_t buf_len, bool buf_filled )
{
	volatile bool finish = false;

	if ( buf_len & 0x1FF )
		return false;	// バッファは512バイト単位で指定すること

	if ( !irq_register( dev->irq, IRQ_TRIGGER_EDGE, ata_irq_handler, (void *)&finish ) )
		return false;

	outb( dev->ports->cmd, cmd );
//	vmm_printf( VMM_DEBUG, "start polling ... " );
	while ( !finish );	// 割り込みがくるまで待つ
//	vmm_printf( VMM_DEBUG, "end.\n" );
	irq_unregister( dev->irq, ata_irq_handler );

	// 念のためBSY=0か確認する
	while ( inb( dev->ports->status ) & ATA_STATUS_BSY );

	if ( inb( dev->ports->status ) & ATA_STATUS_ERR ) {
		vmm_printf( VMM_DEBUG, "ata(%d): ata_pio_read() failed. (error=%02x)\n",
					dev->id, inb( dev->ports->error ) );
		return false;
	}

	u8 sector_cnt;
	if ( buf_filled )
	{	// バッファを埋めるまでリードする
		if ( buf_len == 0 ) {
//			vmm_printf( VMM_DEBUG, "buf_len = 0\n" );
			return true;
		}
		sector_cnt = buf_len >> 9;
	}
	else
	{	// SectorCountレジスタの値に従ってリードする
		sector_cnt = inb( dev->ports->sector_cnt );
		if ( sector_cnt == 0 ) {
			vmm_printf( VMM_DEBUG, "ata(%d): read sector count == 0\n", dev->id );
			return false;
		}
	}
	u16 *p = (u16 *)buf;
	for ( u8 sector=0; sector < sector_cnt; sector++ ) {
		for ( u32 w=0; w<256; w++ )
			p[w] = inw( dev->ports->data );	// 2バイトずつ読む
		p += 256;	// 512バイト分進める
	}

	inb( dev->ports->alt_status );
	if ( inb( dev->ports->status ) & ATA_STATUS_ERR ) {
		vmm_printf( VMM_DEBUG, "ata(%d): ata_pio_read() failed. (error=%02x)\n",
					dev->id, inb( dev->ports->error ) );
		return false;
	}

//	vmm_printf( VMM_DEBUG, "ata(%d): status=%08x\n", dev->id, inb( dev->ports->status ) );

	return true;
}

bool ata_cmd_read_sectors( ATA_DEVICE *dev, u32 lba, u8 sectors, void *buf )
{
	if ( !ata_select_device( dev ) ) {
		vmm_printf( VMM_DEBUG, "ata_cmd_read_sectors(): ata_select_device() failed.\n" );
		return false;
	}

	if ( !(inb( dev->ports->status ) & ATA_STATUS_DRDY) ) {
		vmm_printf( VMM_DEBUG, "ata_cmd_read_sectors(): ATA_STATUS_DRDY == 0\n" );
		return false;
	}

	const u8 dev_head = inb( dev->ports->dev_head ) | ATA_DEV_LBA | ((lba>>24) & 0x0F);
	outb( dev->ports->dev_head, dev_head );
	outb( dev->ports->lba_h, (u8)(lba>>16) );
	outb( dev->ports->lba_m, (u8)(lba>> 8) );
	outb( dev->ports->lba_l, (u8)(lba    ) );
	outb( dev->ports->sector_cnt, sectors );

	if ( !ata_pio_read( dev, ATA_CMD_READ_SECTORS, buf, sectors << 9, true ) )
	{	// リードエラー
		vmm_printf( VMM_DEBUG, "ata(%d): READ SECTOR(S) failed. (lba=%08x, sectors=%02x, error=%02x)\n",
					dev->id, lba, sectors, inb( dev->ports->error ) );
		return false;
	}

	return true;
}

bool ata_cmd_identify_device( ATA_DEVICE *dev, ATA_IDENTIFY_DEVICE_DATA *data )
{
	if ( !ata_select_device( dev ) )
		return false;

	if ( !(inb( dev->ports->status ) & ATA_STATUS_DRDY) )
		return false;

	return ata_pio_read( dev, ATA_CMD_IDENTIFY_DEVICE, data, sizeof(ATA_IDENTIFY_DEVICE_DATA), true );
}

bool ata_cmd_identify_packet_device( ATA_DEVICE *dev, ATA_IDENTIFY_DEVICE_DATA *data )
{
	if ( !ata_select_device( dev ) )
		return false;

	if ( !(inb( dev->ports->status ) & ATA_STATUS_DRDY) )
		return false;

	return ata_pio_read( dev, ATA_CMD_IDENTIFY_PACKET_DEVICE, data, sizeof(ATA_IDENTIFY_DEVICE_DATA), true );
}
