/*!
******************************************************************************

	@file	disp.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "disp.h"
#include "font.h"
#include "io.h"
#include "irq.h"
#include "printf.h"
#include "syscall.h"

static bool		disp_enable = false;
static DISPLAY	disp;

#define DISP_VRAMBUF		disp_vrambuf
#define DISP_VRAMBUF_SIZE	0x00200000		// 2MB
static u8 disp_vrambuf[DISP_VRAMBUF_SIZE];

bool disp_init( void )
{
	// フォントを準備する
	if ( !font_init() )
		return false;

	// 画面モードを変更する
	memset( &disp, 0, sizeof(disp) );
	disp.video = video_init( 640, 480, 32 );
	if ( disp.video == NULL )
		return false;

	disp_enable		 = true;
	disp.enable_text = true;
	disp.page_bytes	 = disp.video->xbytes * disp.video->height;

	return true;
}

void disp_enable_text( void )
{
	disp.enable_text = true;
}

void disp_disable_text( void )
{
	disp.enable_text = false;
}

void * disp_lock( void )
{
	if ( !disp_enable )
		return NULL;

#ifndef _VSUN86_PCSIM
	return disp_vrambuf;
#else	//_VSUN86_PCSIM
	if ( 0 != SDL_LockSurface( disp.video->screen ) )
		return NULL;
	return disp.video->screen->pixels;
#endif	//_VSUN86_PCSIM
}

void disp_unlock( void )
{
	if ( !disp_enable )
		return;

#ifdef	_VSUN86_PCSIM
	SDL_UnlockSurface( disp.video->screen );
#endif	//_VSUN86_PCSIM
}

u32 disp_get_vram_size( void )
{
	if ( !disp_enable )
		return 0;

	return disp.video->fbsize << 16;
}

int disp_write_char( int x, int y, int c, RGB32 color )
{
	if ( !disp.enable_text )
		return -1;

	const int screen_w = disp.video->width;
	const int screen_h = disp.video->height;

	int glyph_w, glyph_h;
	const u8 *glyph = (u8 *)font_get_glyph( c, &glyph_w, &glyph_h );
	if ( glyph == NULL )
		return -1;
	if ( (x + glyph_w > screen_w) || (y + glyph_h > screen_h) )
		return -1;

	u32 *vram = (u32 *)disp_lock();
	if ( vram == NULL )
		return -1;

	for ( int n=0; n<(glyph_w>>3); n++ ) {
		for ( int glyph_y=0; glyph_y<glyph_h; glyph_y++ ) {
			u8 b = glyph[n*glyph_h+glyph_y];
			u32 *fb = &vram[(y+glyph_y) * (disp.video->xbytes >> 2) + x];
			int x_cnt = 0;
			for ( int glyph_x=n<<3; glyph_x<glyph_w; glyph_x++ ) {
				fb[x_cnt++] = (b & 0x80)? color : COLOR_NONE;
				if ( x_cnt >= 8 )
					break;
				b <<= 1;
			}
		}
	}

	disp_unlock();
	disp_update( x, y, glyph_w, glyph_h );

	return 0;
}

int disp_draw_text( int x, int y, const char *buf, size_t buf_len, RGB32 color )
{
	if ( !disp.enable_text )
		return -1;

	const int screen_w = disp.video->width;
	const int screen_h = disp.video->height;

	int x_off = 0;
	int max_glyph_h = 0;
	for ( u32 off=0; off<buf_len; off++ ) {
		if ( buf[off] == '\0' )
			break;

		int glyph_w, glyph_h;
		const u8 *glyph = (u8 *)font_get_glyph( buf[off], &glyph_w, &glyph_h );
		if ( glyph == NULL )
			return -1;
		if ( (x + x_off + glyph_w > screen_w) || (y + glyph_h > screen_h) )
			return -1;
		if ( max_glyph_h < glyph_h )
			max_glyph_h = glyph_h;

		u32 *vram = (u32 *)disp_lock();
		if ( vram == NULL )
			return -1;
		for ( int n=0; n<(glyph_w>>3); n++ ) {
			for ( int glyph_y=0; glyph_y<glyph_h; glyph_y++ ) {
				u8 b = glyph[n*glyph_h+glyph_y];
				u32 *fb = &vram[(y+glyph_y) * (disp.video->xbytes >> 2) + (x+x_off)];
				int x_cnt = 0;
				for ( int glyph_x=n<<3; glyph_x<glyph_w; glyph_x++ ) {
					fb[x_cnt++] = (b & 0x80)? color : COLOR_NONE;
					if ( x_cnt >= 8 )
						break;
					b <<= 1;
				}
			}
		}
		disp_unlock();
		x_off += glyph_w;
	}

	disp_update( x, y, x_off, max_glyph_h );

	return 0;
}

bool disp_get_video_info( void *p )
{
	if ( !disp_enable )
		return false;

	VSUN86_VIDEO_INFO *info = (VSUN86_VIDEO_INFO *)p;
	if ( info == NULL )
		return false;

	info->width	 = disp.video->width;
	info->height = disp.video->height;
	info->bpp	 = disp.video->bpp;
	info->vram	 = DISP_VRAMBUF;

	return true;
}

void disp_bitblt( void *p )
{
	if ( !disp_enable )
		return;

#ifndef _VSUN86_PCSIM
	const int width  = disp.video->width;
	const int height = disp.video->height;
	const int bpp    = disp.video->bpp;

	u32 *src = (u32 *)p;
	u32 *dst = (u32 *)disp.video->fb;
//	vmm_printf( VMM_DEBUG, "disp_bitblt(): dst=%08x, src=%08x, %dx%dx%d\n",
//				dst, src, width, height, bpp );

//	irq_lock();

//	const u64 start = Vsun86_GetTickCount();
	if ( bpp == 32 ) {
		const u32 pages = (width * height) >> 10;
		u32 off = 0;
		for ( u32 i=0; i<pages; i++ )
		{	// 4KB単位でデータを転送する
//			const u32 pte = get_pte( &src[off] );
//			if ( pte & PTE_DIRTY )
			{	// ページが更新されている
				for ( int n=0; n<1024; n++ )
					dst[off + n] = src[off + n];
//				set_pte( &src[off], pte & ~PTE_DIRTY );
			}
			off += 1024;
		}
	}
	else {
		memcpy( dst, src, width * height * (bpp >> 3) );
	}
//	const u64 end = Vsun86_GetTickCount();
//	vmm_printf( VMM_DEBUG, "disp_bitblt(): %lld ms\n", end - start );

//	irq_unlock();

#else	//_VSUN86_PCSIM
	(void)p;
#endif	//_VSUN86_PCSIM
}

void disp_update( int x, int y, int w, int h )
{
	if ( !disp_enable )
		return;

#ifndef _VSUN86_PCSIM
	(void)x;
	(void)y;
	(void)w;
	(void)h;
	disp_bitblt( disp_vrambuf );
#else	//_VSUN86_PCSIM
	SDL_UpdateRect( disp.video->screen, x, y, w, h );
#endif	//_VSUN86_PCSIM
}

void disp_vertical_scroll( int top, int bottom, int scroll_y, RGB32 bgcolor )
{
	const int screen_width  = disp.video->width;
	const int screen_height = disp.video->height;

	if ( scroll_y == 0 )
		return;

	if ( (top < 0) || (bottom < 0) || (top >= bottom) )
		return;
	if ( (top >= screen_height) || (bottom >= screen_height) )
		return;
	if ( (top + scroll_y < 0) || (bottom + scroll_y >= screen_height) )
		return;

	u32 *vram = (u32 *)disp_lock();
	if ( vram == NULL )
		return;

	if ( scroll_y < 0 )
	{	// 上方向にスクロールする
		for ( int y=top; y<=bottom; y++ ) {
			for ( int x=0; x<screen_width; x++ )
				vram[x+(y+scroll_y)*screen_width] = vram[x+y*screen_width];
		}
		int i = (bottom+1+scroll_y)*screen_width;
		while ( i < (bottom+1)*screen_width )
			vram[i++] = bgcolor;
	}
	else
	{	// 下方向にスクロールする
		for ( int y=bottom; y>=top; y-- ) {
			for ( int x=0; x<screen_width; x++ )
				vram[x+(y+scroll_y)*screen_width] = vram[x+y*screen_width];
		}
		int i = (top+scroll_y)*screen_width;
		while ( i < top*screen_width )
			vram[i++] = bgcolor;
	}

	disp_unlock();
	disp_update( 0, 0, disp.video->width, disp.video->height );
}
