/*!
******************************************************************************

	@file	fdc.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "fdc.h"
#include "atomic.h"
#include "io.h"
#include "irq.h"
#include "timer.h"
#include "printf.h"

static bool fdc_send_cmd( u8, const u8 *, size_t, bool, u8 *, size_t );
static bool fdc_is_busy( u8 drive );
static bool fdc_is_input_ready( void );
static bool fdc_is_output_ready( void );
static bool fdc_irq_handler( u8 irq, void *args );

bool fdc_init( void )
{
	outb( IO_FDC_CCR, 0x00 );
	return fdc_specify( FDC_SRT_4MS | FDC_HUT_16MS, 16 );
}

bool fdc_prepare( u8 drive )
{
	static bool motor_enable[4] = { false, false, false, false };

	if ( drive == FDC_DRIVE_IGNORE )
		return true;

	if ( drive > 0 )
		return false;

	drive &= 0x03;
	if ( !motor_enable[drive] )
	{	// モーターを回転させる
		const u8 flags = FDC_DOR_MOTOR0_ENABLE
					   | FDC_DOR_DMA_INT_ENABLE
					   | FDC_DOR_CONTROLLER_RESET;
		outb( IO_FDC_DOR, drive | flags );
		motor_enable[drive] = true;

		// モーターの回転が安定するまで待つ
		timer_usleep( 1000000 );

		// RECALIBRATEコマンドを投げる
		if ( !fdc_recalibrate( drive ) )
		{	// 再試行する
			if ( !fdc_recalibrate( drive ) )
				return false;
		}

		// ★どこかのタイミングでモーターの回転を止めるようにしたい！
	}

	return true;
}

bool fdc_specify( u8 srt_hut, u8 hlt )
{
	const u8 cmd[] = {
		FDC_CMD_SPECIFY, srt_hut, hlt << 1,
	};
	return fdc_send_cmd( FDC_DRIVE_IGNORE, cmd, sizeof(cmd), false, NULL, 0 );
}

bool fdc_read_data( u8 drive, u8 c, u8 h, u8 s )
{
	const u8 cmd[] = {
		FDC_CMD_READ_DATA | FDC_CMDOPT_MT | FDC_CMDOPT_MFM | FDC_CMDOPT_SK,
		(h<<2) | (drive & 0x03), c, h, s, FDC_N_512, 18, 1, 0xFF
	};
	u8 result[7];	// ST0,ST1,ST2,C,H,R,N

	if ( !fdc_send_cmd( drive, cmd, sizeof(cmd), true, result, sizeof(result) ) )
		return false;

	if ( (result[0] & 0xC0) != 0x00 ) {
		vmm_printf( VMM_DEBUG, "fdc: [READ DATA] ST0=%02x, ST1=%02x, ST2=%02x, C:H:R:N=%02x:%02x:%02x:%02x\n",
					result[0], result[1], result[2],
					result[3], result[4], result[5], result[6] );
		return false;
	}

	return true;
}

bool fdc_recalibrate( u8 drive )
{
	const u8 cmd[] = {
		FDC_CMD_RECALIBRATE, (drive & 0x03)
	};

	return fdc_send_cmd( drive, cmd, sizeof(cmd), true, NULL, 0 );
}

bool fdc_sense_int_status( u8 *buf, size_t buf_len )
{
	const u8 cmd[] = {
		FDC_CMD_SENSE_INT_STATUS
	};

	if ( (buf == NULL) || (buf_len < 2) )
		return false;

	return fdc_send_cmd( FDC_DRIVE_IGNORE, cmd, sizeof(cmd), false, buf, 2 );
}

bool fdc_seek( u8 drive, u8 c, u8 h )
{
	const u8 cmd[] = {
		FDC_CMD_SEEK, (h<<2) | (drive & 0x03), c
	};

	return fdc_send_cmd( drive, cmd, sizeof(cmd), true, NULL, 0 );
}

static bool fdc_send_cmd( u8 drive, const u8 *cmd, size_t cmd_len, bool wait_interrupt,
						  u8 *result, size_t result_len )
{
	static volatile int processing = 0;
	volatile bool cmd_end = false;

	if ( !try_lock( &processing ) )
		return false;

	// コマンドが送れるようになるまで待つ
	while ( fdc_is_busy( drive ) );

	if ( wait_interrupt )
	{	// 割り込みハンドラを登録する
		if ( !irq_register( IRQ_FDC, IRQ_TRIGGER_EDGE, fdc_irq_handler, (void *)&cmd_end ) ) {
			unlock( &processing );
			return false;
		}
	}

	// コマンドを送る
	for ( u32 i=0; i<cmd_len; i++ ) {
		while ( !fdc_is_output_ready() );
		outb( IO_FDC_FIFO, cmd[i] );
	}

	if ( wait_interrupt )
	{	// 割り込みが入るまで待つ
		while ( !cmd_end );
		// 割り込みハンドラの登録を解除する
		irq_unregister( IRQ_FDC, fdc_irq_handler );
	}

	// 実行結果を取得する
	for ( u32 i=0; i<result_len; i++ ) {
		while ( !fdc_is_input_ready() );
		result[i] = inb( IO_FDC_FIFO );
	}

	unlock( &processing );
	return true;
}

static bool fdc_is_busy( u8 drive )
{
	u8 mask = FDC_MSR_CMD_BUSY;

	if ( drive != FDC_DRIVE_IGNORE ) {
		if ( drive & 0x01 )
			mask |= FDC_MSR_DRIVE1_BUSY;
		else
			mask |= FDC_MSR_DRIVE0_BUSY;
	}

	const u8 status = inb( IO_FDC_MSR );
	if ( status & mask )
		return true;

	return false;
}

static bool fdc_is_input_ready( void )
{
	const u8 status = inb( IO_FDC_MSR );

	if ( (status & (FDC_MSR_DIO|FDC_MSR_REQ_MASTER)) != (FDC_MSR_DIO_IN|FDC_MSR_REQ_MASTER) )
		return false;

	return true;
}

static bool fdc_is_output_ready( void )
{
	const u8 status = inb( IO_FDC_MSR );

	if ( (status & (FDC_MSR_DIO|FDC_MSR_REQ_MASTER)) != (FDC_MSR_DIO_OUT|FDC_MSR_REQ_MASTER) )
		return false;

	return true;
}

static bool fdc_irq_handler( u8 irq, void *args )
{
	(void)irq;

	bool *cmd_end = (bool *)args;
	*cmd_end = true;

	return true;
}
