/*!
******************************************************************************

	@file	fs.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "disk.h"
#include "fs.h"
#include "fs/fat.h"
#include "printf.h"

static FS_OPS	fs_ops[FS_OPS_MAX];
static FS_DRIVE	fs_drive[FS_DRIVE_MAX];
static u32		fs_drive_num;
static FS_FILE	fs_file_desc[FS_FILE_DESC_MAX];
static bool		fs_file_desc_used[FS_FILE_DESC_MAX];

#define FS_CACHE			fs_cache
#define FS_CACHE_SIZE		0x00200000		// 2MB
static u8 fs_cache[FS_CACHE_SIZE];

bool fs_init( void )
{
	memset( fs_ops,   0, sizeof(fs_ops) );
	memset( fs_drive, 0, sizeof(fs_drive) );
	fs_drive_num = 0;

	memset( fs_file_desc, 0, sizeof(fs_file_desc) );
	memset( fs_file_desc_used, 0, sizeof(fs_file_desc_used) );

	if ( !fat_init( &fs_ops[FS_OPS_FAT] ) )
		return false;

	return true;
}

bool fs_probe( u32 disk_id, u8 first_sector_type )
{
	const u32 cache_size = FS_CACHE_SIZE / FS_DRIVE_MAX;
	bool probe_ok = false;

	FS_OPS *ops;
	FS_DRIVE *drive;

	if ( first_sector_type == FS_1ST_SECTOR_VBR )
	{	// VBRを読み出す
		VBR vbr;
		if ( fs_drive_num >= FS_DRIVE_MAX )
			return false;
		if ( !disk_read( disk_id, 0, 1, &vbr, sizeof(vbr) ) ) {
			vmm_printf( VMM_ERROR, "FS: disk_read() failed.\n" );
			return false;
		}
		if ( (vbr.sig[0] != 0x55) || (vbr.sig[1] != 0xAA) ) {
			vmm_printf( VMM_ERROR, "FS: MBR read error.\n" );
			return false;
		}
		// ★FAT前提で処理する
		ops = &fs_ops[FS_OPS_FAT];
		drive = &fs_drive[fs_drive_num];
		drive->disk_id	  = disk_id;
		drive->fs_ops_id  = ops->fs_id;
		drive->first_sec  = 0;
		drive->cache	  = &FS_CACHE[cache_size * fs_drive_num];
		drive->cache_size = cache_size;
		if ( !ops->probe( drive ) ) {
			vmm_printf( VMM_ERROR, "FS: ops->probe() failed.\n" );
			return false;
		}
		drive->valid = true;
		fs_drive_num++;
		probe_ok = true;
	}
	else if ( first_sector_type == FS_1ST_SECTOR_MBR )
	{	// MBRを読み出す
		MBR mbr;
		memset( &mbr, 0, sizeof(mbr) );
		if ( !disk_read( disk_id, 0, 1, &mbr, sizeof(mbr) ) ) {
			vmm_printf( VMM_ERROR, "FS: disk_read() failed.\n" );
			return false;
		}
		if ( (mbr.sig[0] != 0x55) || (mbr.sig[1] != 0xAA) ) {
			vmm_printf( VMM_ERROR, "FS: MBR read error.\n" );
			return false;
		}

		for ( int i=0; i<4; i++ )
		{	// ファイルシステムタイプを判定する
			if ( fs_drive_num >= FS_DRIVE_MAX )
				break;
			switch ( mbr.partition[i].fs_desc )
			{
			case FS_FAT16:
				ops = &fs_ops[FS_OPS_FAT];
				break;
			default:
				ops = NULL;
				break;
			}
			if ( ops == NULL )
				continue;	// ファイルシステムが不明
			drive = &fs_drive[fs_drive_num];
			drive->disk_id	  = disk_id;
			drive->fs_ops_id  = ops->fs_id;
			drive->first_sec  = mbr.partition[i].first_sec;
			drive->cache	  = &FS_CACHE[cache_size * fs_drive_num];
			drive->cache_size = cache_size;
			if ( ops->probe( drive ) ) {
				vmm_printf( VMM_DEBUG, "drive(%d): disk_id=%08x, fs_ops_id=%d, first_sec=%08x\n",
							fs_drive_num, drive->disk_id, drive->fs_ops_id, drive->first_sec );
				drive->valid = true;
				fs_drive_num++;
				probe_ok = true;
			}
		}
	}

	return probe_ok;
}

// ★ファイル名は「//(disk_id)/(path)」の形でよろしく！
int fs_open( const char *filename )
{
	if ( filename == NULL )
		return FS_FILE_DESC_INVALID;

	int fd;
	for ( fd=0; fd<FS_FILE_DESC_MAX; fd++ ) {
		if ( !fs_file_desc_used[fd] )
			break;
	}
	if ( fd >= FS_FILE_DESC_MAX )
		return FS_FILE_DESC_INVALID;

	// ファイル名をチェックする
	if ( (filename[0] != '/') || (filename[1] != '/') ||
		 (filename[2] <  '0') || (filename[2] >  '9') ||
		 (filename[3] != '/') || (filename[4] == '\0') )
	{	// ファイル名は「//?/...」の形式になっていないとダメ
		vmm_printf( VMM_ERROR, "fs_open( \"%s\" ) failed.\n", filename );
		return FS_FILE_DESC_INVALID;
	}
	const u32 drive_id = filename[2] - '0';
	if ( drive_id >= fs_drive_num )
	{	// ドライブ番号が不正
		vmm_printf( VMM_ERROR, "fs_open( \"%s\" ) failed.\n", filename );
		return FS_FILE_DESC_INVALID;
	}

	FS_FILE *desc = &fs_file_desc[fd];
	memset( desc, 0, sizeof(FS_FILE) );
	desc->drive_id = drive_id;
	strcpy( desc->name, &filename[4] );

	FS_DRIVE *drive = &fs_drive[drive_id];
	FS_OPS *ops = &fs_ops[drive->fs_ops_id];
	if ( !ops->open( drive, desc ) )
		return FS_FILE_DESC_INVALID;

	fs_file_desc_used[fd] = true;
	return fd;
}

bool fs_close( int fd )
{
	if ( !fs_file_desc_used[fd] )
		return false;

	fs_file_desc_used[fd] = false;
	return true;
}

bool fs_read( int fd, void *buf, size_t size )
{
	if ( !fs_file_desc_used[fd] )
		return false;

	FS_FILE *desc = &fs_file_desc[fd];
	FS_DRIVE *drive = &fs_drive[desc->drive_id];
	FS_OPS *ops = &fs_ops[drive->fs_ops_id];

	return ops->read( drive, desc, buf, size );
}

long fs_seek( int fd, long ptr, int method )
{
	if ( !fs_file_desc_used[fd] )
		return -1;

	FS_FILE *desc = &fs_file_desc[fd];
	FS_DRIVE *drive = &fs_drive[desc->drive_id];
	FS_OPS *ops = &fs_ops[drive->fs_ops_id];

	if ( !ops->seek( drive, desc, ptr, method ) )
		return -1;

	return desc->rp;
}

bool fs_get_file_size( int fd, size_t *file_size )
{
	if ( !fs_file_desc_used[fd] )
		return false;

	*file_size = fs_file_desc[fd].size;
	return true;
}
