/*!
******************************************************************************

	@file	keyboard.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "keyboard.h"
#include "irq.h"
#include "io.h"
#include "task.h"
#include "printf.h"
#include "timer.h"

static bool keyboard_valid = false;
static KEYBOARD keyboard;

#ifndef _VSUN86_PCSIM
static void keyboard_check_ps2kbd_status( void );
static void keyboard_check_usbkbd_status( void );
static bool keyboard_irq_handler( u8 irq, void *args );
static u32  keyboard_get_scan_code( void );
#else	//_VSUN86_PCSIM
#include "SDL.h"
static void keyboard_check_sdlkbd_status( KEYBOARD_EVENT, SDLKey, SDLMod );
#endif	//_VSUN86_PCSIM

bool keyboard_register( void *dev, KEYBOARD_OPS *ops )
{
	if ( keyboard_valid )
		return false;	// キーボードを登録できない

	keyboard_valid	  = true;
	keyboard.dev	  = dev;
	keyboard.ops.read = ops->read;

	return true;
}

void keyboard_task( void *args )
{
	(void)args;

#ifndef _VSUN86_PCSIM
	// 割り込みハンドラを登録する(PS/2キーボード用)
	while ( 0 != keyboard_get_scan_code() );	// キーバッファをクリアする
	if ( !irq_register( IRQ_KEYBOARD, IRQ_TRIGGER_EDGE, keyboard_irq_handler, NULL ) )
		abort();

	// USBキーボードの状態を監視するためのタイマをセットする
	u32 timer_id_scan_usb_kbd = timer_add( 20, NULL, NULL, false, TASK_ID_KEYBOARD );

	TASK_EVENT ev;
	while ( task_wait_event( &ev ) )
	{
		switch ( ev.msg )
		{
		case MSG_KEY_IRQ:
			{	// PS/2キーボードの状態を確認する
				keyboard_check_ps2kbd_status();
			}
			break;

		case MSG_TIMER_EXPIRED:
			if ( ev.arg1 == timer_id_scan_usb_kbd )
			{	// USBキーボードの状態を確認する
				keyboard_check_usbkbd_status();
				timer_id_scan_usb_kbd = timer_add( 20, NULL, NULL, false, TASK_ID_KEYBOARD );
			}
			break;
		}
	}
#else	//_VSUN86_PCSIM
	TASK_EVENT ev;
	while ( task_wait_event( &ev ) )
	{
		switch ( ev.msg )
		{
		case MSG_VKEY_DOWN:
			keyboard_check_sdlkbd_status( MSG_KEY_DOWN, (SDLKey)ev.arg1, (SDLMod)ev.arg2 );
			break;

		case MSG_VKEY_UP:
			keyboard_check_sdlkbd_status( MSG_KEY_UP, (SDLKey)ev.arg1, (SDLMod)ev.arg2 );
			break;
		}
	}
#endif	//_VSUN86_PCSIM
}

#ifndef _VSUN86_PCSIM
static void keyboard_check_ps2kbd_status( void )
{
	u32 code;
	static u8 mod = 0;
	static u32 pending_code = 0;

	// キースキャンコードから OADG109A のキー番号に変換
	static const u8 keyboard_oadg_key[128] =
	{	//	 +0  +1  +2  +3  +4  +5  +6  +7  +8  +9  +A  +B  +C  +D  +E  +F
			  0,110,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 15, 16,
			 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 43, 58, 31, 32,
			 33, 34, 35, 36, 37, 38, 39, 40, 41,  1, 44, 42, 46, 47, 48, 49,
			 50, 51, 52, 53, 54, 55, 57,100, 60, 61, 30,112,113,114,115,116,
			117,118,119,120,121, 90,125, 91, 96,101,105, 92, 97,102,106, 93,
			 98,103, 99,104,  0,  0, 45,122,123,  0,  0,  0,  0,  0,  0,  0,
			  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
			133,  0,  0, 56,  0,  0,  0,  0,  0,132,  0,131,  0, 14,  0,  0
	};

	while ( 0 != (code = keyboard_get_scan_code()) ) {
		if ( pending_code ) {
			code |= pending_code << 8;
			pending_code = 0;
		}
		switch ( code )
		{
		case 0xE0:		pending_code = 0xE0;	break;
		case 0xE1:		pending_code = 0xE1;	break;
		case 0x1D:		mod |= KEY_MOD_LCTRL;	break;
		case 0x9D:		mod &= ~KEY_MOD_LCTRL;	break;
		case 0x2A:		mod |= KEY_MOD_LSHIFT;	break;
		case 0xAA:		mod &= ~KEY_MOD_LSHIFT;	break;
		case 0x36:		mod |= KEY_MOD_RSHIFT;	break;
		case 0xB6:		mod &= ~KEY_MOD_RSHIFT;	break;
		case 0x38:		mod |= KEY_MOD_LALT;	break;
		case 0xB8:		mod &= ~KEY_MOD_LALT;	break;
		case 0xE01D:	mod |= KEY_MOD_RCTRL;	break;
		case 0xE09D:	mod &= ~KEY_MOD_RCTRL;	break;
		case 0xE038:	mod |= KEY_MOD_RALT;	break;
		case 0xE0B8:	mod &= ~KEY_MOD_RALT;	break;
		case 0xE05B:	mod |= KEY_MOD_LWIN;	break;
		case 0xE0DB:	mod &= ~KEY_MOD_LWIN;	break;
		case 0xE05C:	mod |= KEY_MOD_RWIN;	break;
		case 0xE0DC:	mod &= ~KEY_MOD_RWIN;	break;
		case 0x54:		// [ALT]+[PrintScreen]
			task_send_event( TASK_ID_SHELL, MSG_KEY_DOWN, KEY_MOD_LALT | KEY_MOD_RALT | mod, 124 );
			break;
		case 0xD4:		// [ALT]+[PrintScreen]
			task_send_event( TASK_ID_SHELL, MSG_KEY_UP, KEY_MOD_LALT | KEY_MOD_RALT | mod, 124 );
			break;
		default:
			if ( code < 0xE000 ) {
				if ( code & 0x80 )
					task_send_event( TASK_ID_SHELL, MSG_KEY_UP, mod, keyboard_oadg_key[code & 0x7F] );
				else
					task_send_event( TASK_ID_SHELL, MSG_KEY_DOWN, mod, keyboard_oadg_key[code] );
			}
			else if ( code <= 0xE0FF ) {
				u8 key;
				switch ( code & 0x7F ) {
				case 0x52:	key = 75;	break;
				case 0x53:	key = 76;	break;
				case 0x4B:	key = 79;	break;
				case 0x47:	key = 80;	break;
				case 0x4F:	key = 81;	break;
				case 0x48:	key = 83;	break;
				case 0x50:	key = 84;	break;
				case 0x49:	key = 85;	break;
				case 0x51:	key = 86;	break;
				case 0x4D:	key = 89;	break;
				case 0x35:	key = 95;	break;
				case 0x1C:	key = 108;	break;
				default:
					vmm_printf( VMM_DEBUG, "unknown key code = %08x\n", code );
					key = 0;
					break;
				}
				if ( key != 0 ) {
					if ( code & 0x80 )
						task_send_event( TASK_ID_SHELL, MSG_KEY_UP, mod, key );
					else
						task_send_event( TASK_ID_SHELL, MSG_KEY_DOWN, mod, key );
				}
			}
			else if ( code <= 0xE1FF ) {
				pending_code = code;
			}
			else if ( code <= 0xE1FFFF ) {
				vmm_printf( VMM_DEBUG, "unknown key code = %08x\n", code );
			}
			else {
				vmm_printf( VMM_DEBUG, "unknown key code = %08x\n", code );
			}
			break;
		}
	}
}

static void keyboard_check_usbkbd_status( void )
{
	if ( !keyboard_valid )
		return;

	KEY_CODES codes;
	if ( !keyboard.ops.read( keyboard.dev, &codes ) )
		return;

	for ( u8 i=0; i<codes.num; i++ ) {
		switch ( keyboard.key[codes.key[i]] )
		{
		case 0:
			{	// キーが押された
				task_send_event( TASK_ID_SHELL, MSG_KEY_DOWN, codes.mod, codes.key[i] );
				keyboard.key[codes.key[i]] = 5;
			}
			break;
		case 1:
			{	// キーは前回スキャン時に押されている
				keyboard.key[codes.key[i]] = 3;
				if ( i+1 == codes.num )
				{	// 最後のキーはリピート対象
					task_send_event( TASK_ID_SHELL, MSG_KEY_DOWN, codes.mod, codes.key[i] );
				}
			}
			break;
		}
	}
	for ( u32 i=0; i<256; i++ ) {
		if ( keyboard.key[i] > 0 ) {
			keyboard.key[i]--;
			if ( keyboard.key[i] == 0 )
			{	// キーが離された
				task_send_event( TASK_ID_SHELL, MSG_KEY_UP, codes.mod, i );
			}
		}
	}
}

static bool keyboard_irq_handler( u8 irq, void *args )
{
	(void)irq;
	(void)args;

	task_send_event( TASK_ID_KEYBOARD, MSG_KEY_IRQ, 0, 0 );
	return true;
}

static u32 keyboard_get_scan_code( void )
{
	u32 code = 0;

	if ( inb( IO_KBC_STATUS ) & KBC_STATUS_OBF ) {
		code = inb( IO_KBC_DATA );
		if ( code == 0xE0 )
		{	// E0拡張コード (E0 xx)
			if ( inb( IO_KBC_STATUS ) & KBC_STATUS_OBF ) {
				code <<= 8;
				code |= inb( IO_KBC_DATA );
			}
		}
		else if ( code == 0xE1 )
		{	// E1拡張コード (E1 xx xx)
			if ( inb( IO_KBC_STATUS ) & KBC_STATUS_OBF ) {
				code <<= 8;
				code |= inb( IO_KBC_DATA );
			}
			if ( inb( IO_KBC_STATUS ) & KBC_STATUS_OBF ) {
				code <<= 8;
				code |= inb( IO_KBC_DATA );
			}
		}
	}

	return code;
}
#else	//_VSUN86_PCSIM
static void keyboard_check_sdlkbd_status( KEYBOARD_EVENT msg, SDLKey sdl_key, SDLMod sdl_mod )
{
	u8 mod = 0;
	if ( sdl_mod & KMOD_LSHIFT )	mod |= KEY_MOD_LSHIFT;
	if ( sdl_mod & KMOD_RSHIFT )	mod |= KEY_MOD_RSHIFT;
	if ( sdl_mod & KMOD_LCTRL  )	mod |= KEY_MOD_LCTRL;
	if ( sdl_mod & KMOD_RCTRL  )	mod |= KEY_MOD_RCTRL;
	if ( sdl_mod & KMOD_LALT   )	mod |= KEY_MOD_LALT;
	if ( sdl_mod & KMOD_RALT   )	mod |= KEY_MOD_RALT;

	u8 key = 0xFF;

	switch ( sdl_key )
	{
	case SDLK_1:			key = 2;	break;
	case SDLK_2:			key = 3;	break;
	case SDLK_3:			key = 4;	break;
	case SDLK_4:			key = 5;	break;
	case SDLK_5:			key = 6;	break;
	case SDLK_6:			key = 7;	break;
	case SDLK_7:			key = 8;	break;
	case SDLK_8:			key = 9;	break;
	case SDLK_9:			key = 10;	break;
	case SDLK_0:			key = 11;	break;
	case SDLK_BACKSPACE:	key = 15;	break;
	case SDLK_q:			key = 17;	break;
	case SDLK_w:			key = 18;	break;
	case SDLK_e:			key = 19;	break;
	case SDLK_r:			key = 20;	break;
	case SDLK_t:			key = 21;	break;
	case SDLK_y:			key = 22;	break;
	case SDLK_u:			key = 23;	break;
	case SDLK_i:			key = 24;	break;
	case SDLK_o:			key = 25;	break;
	case SDLK_p:			key = 26;	break;
	case SDLK_a:			key = 31;	break;
	case SDLK_s:			key = 32;	break;
	case SDLK_d:			key = 33;	break;
	case SDLK_f:			key = 34;	break;
	case SDLK_g:			key = 35;	break;
	case SDLK_h:			key = 36;	break;
	case SDLK_j:			key = 37;	break;
	case SDLK_k:			key = 38;	break;
	case SDLK_l:			key = 39;	break;
	case SDLK_RETURN:		key = 43;	break;
	case SDLK_z:			key = 46;	break;
	case SDLK_x:			key = 47;	break;
	case SDLK_c:			key = 48;	break;
	case SDLK_v:			key = 49;	break;
	case SDLK_b:			key = 50;	break;
	case SDLK_n:			key = 51;	break;
	case SDLK_m:			key = 52;	break;
	case SDLK_COMMA:		key = 53;	break;
	case SDLK_PERIOD:		key = 54;	break;
	case SDLK_SLASH:		key = 55;	break;
	default:
		if ( msg == MSG_KEY_DOWN )
			fprintf( stderr, "SDL_KEYDOWN: sym=%d, mod=0x%04x\n", sdl_key, sdl_mod );
		else
			fprintf( stderr, "SDL_KEYUP  : sym=%d, mod=0x%04x\n", sdl_key, sdl_mod );
		break;
	}

	if ( key != 0xFF )
		task_send_event( TASK_ID_SHELL, msg, mod, key );
}
#endif	//_VSUN86_PCSIM
