/*!
******************************************************************************

	@file	syscall.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/
/*-*- mode: C++; Encoding: utf8n -*-*/

#include "vsun86.h"
#include "cpu.h"
#include "syscall.h"
#include "task.h"
#include "atomic.h"
#include "timer.h"
#include "keyboard.h"
#include "disp.h"
#include "fs.h"
#include "heap.h"
#include "printf.h"
#include "uart.h"
#include "pfemu.h"
#include "pfemu/vcpu.h"
#include "shell.h"

typedef struct {
	u32		edi, esi, ebp, esp, ebx, edx, ecx, eax;
} SYSCALL_PARAMS;

EXTERN_C int syscall_entry_point( u8 vector, SYSCALL_PARAMS *p )
{
	static volatile int processing = 0;

	if ( vector != SYSCALL_VECTOR )
		return p->eax;

//	lock( &processing );
	if ( !try_lock( &processing ) ) {
		vmm_printf( VMM_DEBUG, "system call failed. (currently processing)\n" );
		abort();
	}

	// システムコール実行中はHW割り込みを有効にする
//	cpu_enable_interrupt();

	switch ( p->eax )
	{
	case SYSCALL_OUTPUT_DBGSTR:
		{	// void Vsun86_OutputDebugString( const char *str )
			uart_send( (const char *)p->ecx );
		}
		break;

	case SYSCALL_GET_TICK_COUNT:
		{	// u64 Vsun86_GetTickCount( void )
			const u64 tick = timer_get_tick_count();
			p->edx = (u32)(tick >> 32);
			p->eax = (u32)(tick);
		}
		break;

	case SYSCALL_GET_SYSTEM_TIME:
		{	// int Vsun86_GetSystemTime( VSUN86_SYSTEM_TIME *p )
			p->eax = timer_get_system_time( (VSUN86_SYSTEM_TIME *)(p->ecx) )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_GET_LOCAL_TIME:
		{	// int Vsun86_GetLocalTime( VSUN86_SYSTEM_TIME *p )
			p->eax = timer_get_local_time( (VSUN86_SYSTEM_TIME *)(p->ecx) )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_GET_VIDEO_INFO:
		{	// int Vsun86_GetVideoInfo( VSUN86_VIDEO_INFO *p )
			p->eax = disp_get_video_info( (void *)p->ecx )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_PUT_STRING:
		{	// void Vsun86_PutString( const char *str )
			shell_put_string( (const char *)p->ecx, CONSOLE_FGCOLOR );
		}
		break;

	case SYSCALL_PUT_CHAR:
		{	// void Vsun86_PutChar( int c )
			shell_put_char( (char)p->ecx, CONSOLE_FGCOLOR );
		}
		break;

	case SYSCALL_BIT_BLT:
		{	// void Vsun86_BltBlt( void *src )
			disp_bitblt( (void *)p->ecx );
		}
		break;

	case SYSCALL_OPEN_FILE:
		{	// int Vsun86_OpenFile( const char *filename )
			int fd = fs_open( (const char *)p->ecx );
			if ( fd == FS_FILE_DESC_INVALID )
				p->eax = (u32)FILE_DESC_INVALID;
			else
				p->eax = fd;
		}
		break;

	case SYSCALL_CLOSE_FILE:
		{	// void Vsun86_CloseFile( int fd )
			fs_close( (int)p->ecx );
		}
		break;

	case SYSCALL_READ_FILE:
		{	// int Vsun86_ReadFile( int fd, void *buf, size_t size )
			p->eax = fs_read( (int)p->ecx, (void *)p->edx, (size_t)p->ebx )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_SET_FILE_PTR:
		{	// long Vsun86_SetFilePointer( int fd, long ptr, int method )
			p->eax = fs_seek( (int)p->ecx, (long)p->edx, (int)p->ebx );
		}
		break;

	case SYSCALL_GET_FILE_SIZE:
		{	// int Vsun86_GetFileSize( int fd, size_t *size )
			p->eax = fs_get_file_size( (int)p->ecx, (size_t *)p->edx );
		}
		break;

	case SYSCALL_HEAP_ALLOC:
		{	// void * Vsun86_HeapAlloc( size_t size )
			p->eax = (u32)heap_alloc( (size_t)p->ecx );
		}
		break;

	case SYSCALL_HEAP_FREE:
		{	// int Vsun86_HeapFree( void *ptr )
			p->eax = (u32)heap_free( (void *)p->ecx );
		}
		break;

	case SYSCALL_GET_VCPU:
		{	// VCPU * Vsun86_GetVCPU( void )
			p->eax = (u32)pfemu_get_vcpu();
		}
		break;

	case SYSCALL_RUN_VCPU:
		{	// int Vsun86_RunVCPU( VCPU_RESULT *result )
			p->eax = pfemu_run_vcpu( (void *)p->ecx )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_SET_VCPU_EVENT:
		{	// int Vsun86_SetVCPUEvent( VCPU_EVENT *e )
			p->eax = pfemu_set_vcpu_event( (void *)p->ecx )? 0:0xFFFFFFFF;
		}
		break;

	case SYSCALL_SET_VCPU_MMIO:
		{	// int Vsun86_SetVCPU_MMIO( u32 start_page, u32 pages, VCPU_MMIO_PROCS *procs )
			p->eax = pfemu_set_vcpu_mmio( p->ecx, p->edx, (void *)p->ebx );
		}
		break;

	case SYSCALL_SET_VM_EVENTCB:
		{	// int Vsun86_SetVMEventCallback( VM_EVENT_CALLBACK proc )
			shell_set_vm_eventcb( (void *)p->ecx );
		}
		break;

	default:
		vmm_printf( VMM_DEBUG, "syscall(%08x) failed.\n", p->eax );
		break;
	}

	unlock( &processing );

	return p->eax;
}
