/*!
******************************************************************************

	@file	msc.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "usb.h"
#include "msc.h"
#include "printf.h"
#include "disk.h"

#define USB_MSC_CMD_TIMEOUT		5000	// [ms]

static bool usb_msc_bulk_xfer( USB_DEVICE *dev, void *cmd, u32 cmd_len, void *buf, u32 buf_len, bool dir_in );

bool usb_msc_init( USB_DEVICE *dev )
{
	if ( dev->iface_protocol != USB_MSC_BULK_ONLY_TRANSPORT )
	{	// Bulk Only Transport のみ対応
		vmm_printf( VMM_DEBUG, "usb_msc: Protocol %02x is not supported.\n", dev->iface_protocol );
		return false;
	}

	if ( dev->iface_subclass != USB_MSC_SUBCLASS_SCSI )
	{	// SCSIのみ対応 (USBメモリの大半はSCSIでいけるらしい？)
		vmm_printf( VMM_DEBUG, "usb_msc: Subclass %02x is not supported.\n", dev->iface_subclass );
		return false;
	}

	int bulk_in  = -1;
	int bulk_out = -1;
	for ( int i=1; i<USB_ENDPOINT_MAX; i++ )
	{	// Bulk-In / Bulk-Out エンドポイントを取得する
		if ( !dev->ep[i].valid )
			continue;
		if ( USB_ENDPT_XFER_TYPE( dev->ep[i].attr ) != USB_ENDPT_BULK )
		{	// Bulk Only のはずなのに…
			vmm_printf( VMM_DEBUG, "usb_msc: Endpoint check error. (type=%d)\n", USB_ENDPT_XFER_TYPE( dev->ep[i].attr ) );
			return false;
		}
		if ( dev->ep[i].addr & 0x80 )
		{	// Bulk-In
			if ( bulk_in < 0 )
				bulk_in = i;
		}
		else
		{	// Bulk-Out
			if ( bulk_out < 0 )
				bulk_out = i;
		}
	}
	if ( (bulk_in < 0) || (bulk_out < 0) )
	{	// Bulk-In / Bulk-Out は両方必要
		vmm_printf( VMM_DEBUG, "usb_msc: Endpoint check error. (in=%d,out=%d)\n", bulk_in, bulk_out );
		return false;
	}
	dev->msc.bulk_in	= (u8)bulk_in;
	dev->msc.bulk_out	= (u8)bulk_out;
	dev->msc.bulk_xfer	= usb_msc_bulk_xfer;

	SCSI_DEVICE scsi_dev;
	scsi_dev.bus_dev	= dev;
	scsi_dev.send_cmd	= (SCSI_SEND_CMD)usb_msc_bulk_xfer;
	if ( !disk_register_scsi( &scsi_dev ) ) {
		vmm_printf( VMM_DEBUG, "usb_msc: disk_register_scsi() failed.\n" );
		return false;
	}

	return true;
}

static bool usb_msc_bulk_xfer( USB_DEVICE *dev, void *cmd, u32 cmd_len, void *buf, u32 buf_len, bool dir_in )
{
	USB_MSC_CBW cbw;
	USB_MSC_CSW csw;

	const u8 out_ep = dev->msc.bulk_out;
	const u8 in_ep  = dev->msc.bulk_in;

	memset( &cbw, 0, sizeof(USB_MSC_CBW) );
	memset( &csw, 0, sizeof(USB_MSC_CSW) );

	// CBWを準備する
	cbw.dCBWSignature			= USB_MSC_CBW_SIG;
	cbw.dCBWTag					= 0;
	cbw.dCBWDataTransferLength	= buf_len;
	cbw.bmCBWFlags				= dir_in? USB_MSC_CBW_DIR_IN : USB_MSC_CBW_DIR_OUT;
	cbw.bCBWLUN					= 0;
	cbw.bCBWCBLength			= cmd_len;
	memcpy( cbw.CBWCB, cmd, cmd_len );

	// CBWを転送する
	if ( !usb_bulk_write( dev, out_ep, &cbw, sizeof(USB_MSC_CBW), USB_MSC_CMD_TIMEOUT ) ) {
		vmm_printf( VMM_DEBUG, "usb_msc: send CBW failed.\n" );
		return false;
	}

	if ( dir_in )
	{	// データが来るのを待つ
		if ( !usb_bulk_read( dev, in_ep, buf, buf_len, USB_MSC_CMD_TIMEOUT ) ) {
			vmm_printf( VMM_DEBUG, "usb_msc: recv data failed.\n" );
			return false;
		}
	}
	else
	{	// データを転送する
		if ( !usb_bulk_write( dev, out_ep, buf, buf_len, USB_MSC_CMD_TIMEOUT ) ) {
			vmm_printf( VMM_DEBUG, "usb_msc: send data failed.\n" );
			return false;
		}
	}

	// CSWが来るのを待つ
	if ( !usb_bulk_read( dev, in_ep, &csw, sizeof(USB_MSC_CSW), USB_MSC_CMD_TIMEOUT ) ) {
		vmm_printf( VMM_DEBUG, "usb_msc: recv CSW failed.\n" );
		return false;
	}

	if ( csw.dCSWSignature != USB_MSC_CSW_SIG )
		return false;

	if ( csw.bCSWStatus != 0x00 )
		return false;

	return true;
}
