/*!
******************************************************************************

	@file	uhci.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "pci.h"
#include "usb.h"
#include "uhci.h"
#include "printf.h"
#include "io.h"
#include "timer.h"
#include "task.h"

static u8 uhci_num;
static UHCI_HCD uhci_hcd[UHCI_HOST_MAX];

#define IO_UHCI_USBCMD		( hcd->io_base + 0x00 )
#define IO_UHCI_USBSTS		( hcd->io_base + 0x02 )
#define IO_UHCI_USBINTR		( hcd->io_base + 0x04 )
#define IO_UHCI_FRNUM		( hcd->io_base + 0x06 )
#define IO_UHCI_FLBASEADD	( hcd->io_base + 0x08 )
#define IO_UHCI_SOFMOD		( hcd->io_base + 0x0C )
#define IO_UHCI_PORTSC1		( hcd->io_base + 0x10 )
#define IO_UHCI_PORTSC_MAX	( hcd->io_base + 0x100 )

#define PORT_RESET			0x0200
#define PORT_CLEAR_ENABLE	0x0008
#define PORT_ENABLE			0x0004
#define PORT_CLEAR_CONNECT	0x0002

#define USBCMD_MAXP			0x0080	// Max Packet
#define USBCMD_CF			0x0040	// Configure Flag
#define USBCMD_SWDBG		0x0020	// Software Debug
#define USBCMD_FGR			0x0010	// Force Global Resume
#define USBCMD_EGSM			0x0008	// Enter Global Suspend Mode
#define USBCMD_GRESET		0x0004	// Global Reset
#define USBCMD_HCRESET		0x0002	// Host Controller Reset
#define USBCMD_RS			0x0001	// Run / Stop

#define USBINTR_TIMEOUT		0x0001
#define USBINTR_RESUME		0x0002
#define USBINTR_IOC			0x0004
#define USBINTR_SHORT_PKT	0x0008
#define USBINTR_ALL			0x000F

static void uhci_task( void *args );
static bool uhci_irq_handler( u8 irq, void *args );
//static void uhci_write_command( UHCI_HCD *hcd, u16 cmd );
//static u16  uhci_read_status( UHCI_HCD *hcd );
//static void uhci_clear_status( UHCI_HCD *hcd );
//static u16  uhci_read_port( UHCI_HCD *hcd, u8 port );
//static void uhci_write_port( UHCI_HCD *hcd, u8 port, u16 data );
static void uhci_run( UHCI_HCD *hcd );
static void uhci_stop( UHCI_HCD *hcd );
static void uhci_enable_interrupt( UHCI_HCD *hcd, u16 intr );
static void uhci_set_fl_base_addr( UHCI_HCD *hcd, u32 addr );
static void uhci_set_frame_number( UHCI_HCD *hcd, u16 index );
//static u16  uhci_get_frame_number( UHCI_HCD *hcd );
static void uhci_check_status( UHCI_HCD *hcd );

bool uhci_init( void )
{
	uhci_num = 0;
	memset( uhci_hcd, 0, sizeof(uhci_hcd) );

	return true;
}

bool uhci_probe( PCI_DEVICE *dev )
{
	const u8 index = uhci_num;
	if ( index > 0 )
		return false;	// とりあえず１つだけ

	UHCI_HCD *hcd = &uhci_hcd[index];
	memset( hcd, 0, sizeof(UHCI_HCD) );

	hcd->id = index;
	hcd->io_base = dev->base_addr[4];
	if ( PCI_ADDR_IS_MEM( hcd->io_base & 0x01 ) )
		return false;	// I/O空間のアドレスでないとダメ
	hcd->io_base &= 0xFFFC;

	if ( !pci_irq_register( dev, uhci_irq_handler, hcd ) )
		return false;	// IRQハンドラが登録できない

	hcd->ports = 0;
	for ( u16 port=IO_UHCI_PORTSC1; port<IO_UHCI_PORTSC_MAX; port+=2 ) {
		const u16 port_status = inw( port );
		if ( !(port_status & 0x80) || (port_status == 0xFFFF) )
			break;	// ポートのステータスが読めない
		hcd->ports++;
	}

	vmm_printf( VMM_DEBUG, "usb_uhci(%d): io_base=%04x, irq=%02x, ports=%d\n",
				index, hcd->io_base, dev->interrupt_line, hcd->ports );

	if ( !uhci_queue_init( hcd ) )
		return false;

	uhci_set_fl_base_addr( hcd, virt_to_phys(hcd->frame_list) );
	uhci_set_frame_number( hcd, 0 );
	uhci_enable_interrupt( hcd, USBINTR_ALL );
	/*
	for ( int i=0; i<hcd->ports; i++ )
		uhci_write_port( hcd, i, PORT_RESET | PORT_ENABLE | PORT_CLEAR_CONNECT );
	timer_sleep( 50 );
	uhci_write_command( hcd, USBCMD_MAXP | USBCMD_FGR | USBCMD_GRESET );
	for ( int i=0; i<hcd->ports; i++ )
		uhci_write_port( hcd, i, PORT_CLEAR_ENABLE | PORT_ENABLE | PORT_CLEAR_CONNECT );
	*/

	if ( index == 0 ) {
		if ( !task_create( TASK_ID_USB_UHCI, "USB-UHCI", uhci_task, NULL, 4096 ) )
			return false;
	}

	uhci_num++;
	return true;
}

bool uhci_start( void )
{
	return true;
}

static void uhci_task( void *args )
{
	(void)args;

	static u8 hcd_num = uhci_num;

	if ( timer_add( 1000, NULL, NULL, false, TASK_ID_USB_UHCI ) == TIMER_ID_INVALID ) {
		vmm_printf( VMM_DEBUG, "usb_uhci: timer_add() failed.\n" );
		abort();
	}

	for ( u8 i=0; i<hcd_num; i++ )
		uhci_run( &uhci_hcd[i] );

	TASK_EVENT e;
	while ( task_wait_event( &e ) )
	{
		switch ( e.msg )
		{
		case MSG_TIMER_EXPIRED:
			{	// USB機器の挿抜を確認する
				for ( u8 i=0; i<hcd_num; i++ )
					uhci_check_status( &uhci_hcd[i] );
				// タイマを再設定する
				if ( timer_add( 1000, NULL, NULL, false, TASK_ID_USB_UHCI ) == TIMER_ID_INVALID ) {
					vmm_printf( VMM_DEBUG, "usb_uhci: timer_add() failed.\n" );
					abort();
				}
			}
			break;
		}
	}

	for ( u8 i=0; i<hcd_num; i++ )
		uhci_stop( &uhci_hcd[i] );
}

static bool uhci_irq_handler( u8 irq, void *args )
{
	UHCI_HCD *hcd = (UHCI_HCD *)args;

	vmm_printf( VMM_DEBUG, "usb_uhci(%d): IRQ %02x\n", hcd->id, irq );
	return false;
}

/*
static void uhci_write_command( UHCI_HCD *hcd, u16 cmd )
{
	outw( IO_UHCI_USBCMD, cmd );
}

static u16 uhci_read_status( UHCI_HCD *hcd )
{
	return inw( IO_UHCI_USBSTS );
}

static void uhci_clear_status( UHCI_HCD *hcd )
{
	outw( IO_UHCI_USBSTS, uhci_read_status( hcd ) );
}

static u16 uhci_read_port( UHCI_HCD *hcd, u8 port )
{
	if ( port >= hcd->ports )
		return 0xFFFF;

	return inw( IO_UHCI_PORTSC1 + (port << 1) );
}

static void uhci_write_port( UHCI_HCD *hcd, u8 port, u16 data )
{
	if ( port >= hcd->ports )
		return;

	outw( IO_UHCI_PORTSC1 + (port << 1), data );
}
*/

static void uhci_run( UHCI_HCD *hcd )
{
	u16 cmd = inw( IO_UHCI_USBCMD );
	outw( IO_UHCI_USBCMD, cmd | USBCMD_RS );	// Run/Stop = 1 (Run)
}

static void uhci_stop( UHCI_HCD *hcd )
{
	u16 cmd = inw( IO_UHCI_USBCMD );
	outw( IO_UHCI_USBCMD, cmd & ~USBCMD_RS );	// Run/Stop = 0 (Stop)
}
static void uhci_enable_interrupt( UHCI_HCD *hcd, u16 intr )
{
	outw( IO_UHCI_USBINTR, intr );
}

static void uhci_set_fl_base_addr( UHCI_HCD *hcd, u32 addr )
{
	outd( IO_UHCI_FLBASEADD, addr );
}

static void uhci_set_frame_number( UHCI_HCD *hcd, u16 index )
{
	outw( IO_UHCI_FRNUM, index );
}

/*
static u16 uhci_get_frame_number( UHCI_HCD *hcd )
{
	return inw( IO_UHCI_FRNUM );
}
*/

static void uhci_check_status( UHCI_HCD *hcd )
{
	/*
	vmm_printf( VMM_DEBUG, "usb_uhci(%d): status=%04x:%04x",
				hcd->id, inw( IO_UHCI_USBCMD ), uhci_read_status( hcd ) );
	uhci_clear_status( hcd );
	for ( u8 i=0; i<hcd->ports; i++ ) {
		vmm_printf( VMM_DEBUG, ":%04x", uhci_read_port( hcd, i ) );
		uhci_write_port( hcd, i, uhci_read_port( hcd, i ) );
	}
	vmm_printf( VMM_DEBUG, ":%04x\n", uhci_get_frame_number( hcd ) );
	*/
	(void)hcd;
}
