﻿Public Class Board
    Public Const WHITE As Integer = -1
    Public Const BLACK As Integer = 1
    Public Const EMPTY As Integer = 0

    Public Const SIZE_X As Integer = 8
    Public Const SIZE_Y As Integer = 8

    Private _arrangement(SIZE_X * SIZE_Y - 1) As Integer

    Public Property IsGameEnd As Boolean = False

    Private _turn As Integer
    Public ReadOnly Property Turn As Integer
        Get
            Return _turn
        End Get
    End Property

    Public Sub TurnChange()
        Debug.Assert(_turn = WHITE OrElse _
                     _turn = BLACK)
        _turn = -_turn
    End Sub

    Public Function StoneCount(ByVal color As Integer) As Integer
        Debug.Assert(color = WHITE OrElse
                     color = BLACK OrElse
                     color = EMPTY)
        Dim count As Integer = 0
        For Each stone As Integer In _arrangement
            If stone = color Then
                count += 1
            End If
        Next
        Return count
    End Function

    Public Property CellStone(ByVal x As Integer, ByVal y As Integer) As Integer
        Get
            Debug.Assert(ValidPosition(x, y))
            Return _arrangement(y * SIZE_X + x)
        End Get
        Set(ByVal value As Integer)
            Debug.Assert(ValidPosition(x, y))
            Debug.Assert(value = WHITE OrElse _
                         value = BLACK OrElse _
                         value = EMPTY)
            _arrangement(y * SIZE_X + x) = value
        End Set
    End Property

    Public Shared Function ValidPosition(ByVal x As Integer, ByVal y As Integer) As Boolean
        Return (0 <= x AndAlso x < SIZE_X) AndAlso
               (0 <= y AndAlso y < SIZE_Y)
    End Function

    Public Sub New()
        For Each cell In _arrangement
            cell = EMPTY
        Next
        CellStone(3, 3) = WHITE
        CellStone(3, 4) = BLACK
        CellStone(4, 3) = BLACK
        CellStone(4, 4) = WHITE
        _turn = BLACK
    End Sub

    Public Sub New(ByVal source As Board)
        Debug.Assert(source IsNot Nothing)
        For i As Integer = 0 To _arrangement.Length - 1
            Me._arrangement(i) = source._arrangement(i)
        Next
        Me._turn = source._turn
        Me.IsGameEnd = source.IsGameEnd
    End Sub

    Public Function CanPutAny() As Boolean
        For x As Integer = 0 To SIZE_X - 1
            For y As Integer = 0 To SIZE_Y - 1
                If CanPut(x, y) Then
                    Return True
                End If
            Next
        Next
        Return False
    End Function

    Public Sub Put(ByVal x As Integer, ByVal y As Integer)
        Debug.Assert(ValidPosition(x, y))
        Debug.Assert(CanPut(x, y))
        CellStone(x, y) = Turn
        If CanReverse(x, y, -1, -1) Then
            Reverse(x, y, -1, -1)
        End If
        If CanReverse(x, y, -1, 0) Then
            Reverse(x, y, -1, 0)
        End If
        If CanReverse(x, y, -1, 1) Then
            Reverse(x, y, -1, 1)
        End If
        If CanReverse(x, y, 0, -1) Then
            Reverse(x, y, 0, -1)
        End If
        If CanReverse(x, y, 0, 1) Then
            Reverse(x, y, 0, 1)
        End If
        If CanReverse(x, y, 1, -1) Then
            Reverse(x, y, 1, -1)
        End If
        If CanReverse(x, y, 1, 0) Then
            Reverse(x, y, 1, 0)
        End If
        If CanReverse(x, y, 1, 1) Then
            Reverse(x, y, 1, 1)
        End If
    End Sub

    Private Sub Reverse(ByVal sourceX As Integer, ByVal sourceY As Integer, ByVal directionX As Integer, ByVal directionY As Integer)
        Debug.Assert(-1 <= directionX AndAlso directionX <= 1)
        Debug.Assert(-1 <= directionY AndAlso directionY <= 1)
        Debug.Assert(Not (directionX = 0 AndAlso directionY = 0))
        Dim x As Integer = sourceX + directionX
        Dim y As Integer = sourceY + directionY

        While ValidPosition(x, y)
            If CellStone(x, y) = EMPTY OrElse
               CellStone(x, y) = Turn Then
                Return
            End If
            CellStone(x, y) = Turn
            x += directionX
            y += directionY
        End While
    End Sub

    Public Function CanPut(ByVal x As Integer, ByVal y As Integer) As Boolean
        Debug.Assert(ValidPosition(x, y))
        If CellStone(x, y) <> EMPTY Then
            Return False
        End If

        If CanReverse(x, y, -1, -1) Then
            Return True
        End If
        If CanReverse(x, y, -1, 0) Then
            Return True
        End If
        If CanReverse(x, y, -1, 1) Then
            Return True
        End If
        If CanReverse(x, y, 0, -1) Then
            Return True
        End If
        If CanReverse(x, y, 0, 1) Then
            Return True
        End If
        If CanReverse(x, y, 1, -1) Then
            Return True
        End If
        If CanReverse(x, y, 1, 0) Then
            Return True
        End If
        If CanReverse(x, y, 1, 1) Then
            Return True
        End If

        Return False
    End Function

    Private Function CanReverse(ByVal sourceX As Integer, ByVal sourceY As Integer, ByVal directionX As Integer, ByVal directionY As Integer) As Boolean
        Debug.Assert(-1 <= directionX AndAlso directionX <= 1)
        Debug.Assert(-1 <= directionY AndAlso directionY <= 1)
        Debug.Assert(Not (directionX = 0 AndAlso directionY = 0))
        Dim x As Integer = sourceX + directionX
        Dim y As Integer = sourceY + directionY

        If Not ValidPosition(x, y) Then
            Return False
        End If

        If CellStone(x, y) = EMPTY Then
            Return False
        End If

        If CellStone(x, y) = Turn Then
            Return False
        End If

        x += directionX
        y += directionY

        While ValidPosition(x, y)
            If CellStone(x, y) = EMPTY Then
                Return False
            End If
            If CellStone(x, y) = Turn Then
                Return True
            End If
            x += directionX
            y += directionY
        End While

        Return False

    End Function

End Class
