using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using System.Xml;
using nft.util;

namespace nft.framework.plugin {
    /// <summary>
    /// Represents a loaded plug-in
    /// </summary>
    public class Plugin : IHasNameAndID, IAddable {
        public const string PluginFileName = "plugin.xml";
        #region IHasNameAndID o
        
        private string _id;
        // plugin 'id' is equals to the subdir name where "plugin.xml" is placed.
        public string ID {
            get { return _id; }
        }

        public string Name {
            get { return _title; }
        }

        #endregion

        private string _title;
        public string Title { get { return _title; } }

        private string _author;
        public string author { get { return _author; } }

        private string _homepage;
        public string homepage { get { return _homepage; } }

        internal protected DateTime _lastModified; // settled from PluginManager
        public DateTime lastModifiedTime { get { return _lastModified; } }

        public string getInfoText(string linecode) {
            string templ = "Title:{0}" + linecode + "Author:{1}" + linecode + "HomePage:{2}";
            return string.Format(templ, Title, author, homepage);
        }

        /// <summary>
        /// Location uri of this plug-in
        /// </summary>
        public readonly Uri Uri;

        /// <summary>
        /// ParamsReader is given from PluginManager and will set null after loading complete.
        /// </summary>
        internal protected ParamsReader _reader;

        private string data_dir = null;

        #region IAddable o
        internal protected InstallationState _state = InstallationState.Uninitialized;
        [NonSerialized]
        internal protected AttachChangeEvent onDetach;
        [NonSerialized]
        internal protected AttachChangeEvent onAttach;

        private int detachCount = 0;
        private int totalDetachables = 0;
        private bool detachable = false;
        private bool attached = true;
        public InstallationState State { get { return _state; } }

        public bool IsDetachable { get { return detachable & (_state >= InstallationState.PartialError); } }

        public bool QueryDetach() { return true; }

        public void Detach() {
            Debug.WriteLine("dt" + totalDetachables);
            foreach (Contribution c in Contributions)
                if (c.QueryDetach())
                    c.Detach();
            if (onDetach != null)
                onDetach(this);
            attached = false;
        }

        public void Attach() {
            Debug.WriteLine("at" + totalDetachables);
            foreach (Contribution c in Contributions)
                c.Attach();
            if (onAttach != null)
                onAttach(this);
            attached = true;
        }

        public AttachChangeEvent OnDetach { get { return onDetach; } set { onDetach = value; } }
        public AttachChangeEvent OnAttach { get { return onAttach; } set { onAttach = value; } }
        public virtual bool IsAttached { get { return attached; } }
        public virtual bool IsPartiallyDetached { get { return detachCount != 0 && detachCount != totalDetachables; } }

        void AttachChangeEventHandler(IAddable sender) {
            if (sender.IsAttached) {
                detachCount--;
                attached = true;
                if (onAttach != null)
                    onAttach(this);
            } else {
                detachCount++;
                attached = (detachCount < totalDetachables);
                if (!attached && onDetach != null)
                    onDetach(this);
            }
        }

        internal void AssociateAddableContribution(Contribution c){
            if (c.IsDetachable) {
                totalDetachables++;
                detachable = true;
                c.OnAttach += new AttachChangeEvent(this.AttachChangeEventHandler);
                c.OnDetach += new AttachChangeEvent(this.AttachChangeEventHandler);
            }
        }
        #endregion

        /// <summary>
        /// Loads a plug-in with ParamsReader".
        /// directory name of URI is used as identifier.
        /// </summary>
        /// <param name="path">URI indicates file source path. folder name is used as plugin ID.</param>
        public Plugin(Uri path, DateTime lastModified) {
            this.Uri = path;
            _lastModified = lastModified;
            _id = Path.GetFileName(path.LocalPath.Replace("/", "\\"));
        }

        internal protected void LoadParams(ParamsReader reader){
            _reader = reader;
            _title = reader["title"].InnerText;
            _homepage = reader["homepage"].InnerTextOr("N/A");
            _author = reader["author"].InnerTextOr("<unknown>");
        }

        /// <summary>
        /// Base directory of this plug-in
        /// </summary>
        [Obsolete]
        public string PathName {
            get { return Uri.LocalPath; }
        }


        /// <summary>
        /// Can be used for Cached data storage of this plug-in. 
        /// Each plug-in has it's isolated sub-directory under the DataDir.
        /// </summary>
        public string DataDirectory {
            get {
                if (data_dir == null) {
                    data_dir = Directories.MakeDataDirNameFor(this);
                }
                return data_dir;
            }
        }


        /// <summary>
        /// All the contributions in this plug-in
        /// </summary>
        public IList Contributions {
            get {
                IList l = new ArrayList();
                foreach (Contribution c in PluginManager.theInstance.Contributions) {
                    if (c.Parent == this)
                        l.Add(c);
                }
                return l;
            }
        }

        /// <summary>
        /// All the contribution definers and binary modules in this plug-in
        /// </summary>
        [Obsolete]
        public IList Primitives {
            get {
                return new ArrayList();
            }
        }

        /*
        /// <summary>
        /// Get all the dependent plug-ins.
        /// called from PluginManager before initialize this plugin.
        /// </summary>
        public Plugin[] getDependencies() {
            ArrayList a = new ArrayList();
            if (!this.ID.Equals("system"))
                a.Add(PluginManager.theInstance.GetPlugin("system"));

            foreach (ParamsReader depend in _reader.EnumChildren("depend")) {
                string name = depend["on"].InnerText;
                Plugin p = PluginManager.theInstance.GetPlugin(name);
                if (p == null) {
                    string templ = Main.resources["plugin.dependency_not_found"].stringValue;
                    throw new Exception(String.Format(templ, this.ID, name));
                }
                a.Add(p);
            }
            return (Plugin[])a.ToArray(typeof(Plugin));
        }

        /// <summary>
        /// Loads class type contributions from this plug-in
        /// </summary>
        internal void loadBinaries() {
            // locate contribution factories first,
            // because we'll need them to load contributions.
            foreach (ParamsReader contrib in _reader.EnumChildren("contribution")) {
                string type = contrib["type"].InnerText;
                try {
                    Contribution cb = null;
                    // load a contribution factory
                    if ("factory".Equals(type)) {
                        PluginManager.theInstance.NotifyStartParse(this, contrib);
                        ParamsReader nd = contrib["declare"];
                        string regtype;
                        if (!nd.IsNull) {
                            regtype = nd["type"].InnerText;
                        } else {
                            // use name property for alternative.
                            regtype = contrib["name"].InnerText;
                        }
                        object[] args = new object[] { contrib };
                        // create and register contribution factory
                        IContributionFactory factory =
                            PluginUtil.createCtbFactory(this, contrib);
                        PluginManager.theInstance.AddContributionFactory( regtype, factory);
                        cb = factory as Contribution;
                        // Register dummy contribution (in order to list on Dialog).
                        if (cb == null) {
                            cb = new CtbContributionDefiner(this, contrib);
                        }    
                    } else if ("binary".Equals(type)) {
                        PluginManager.theInstance.NotifyStartParse(this, contrib);
                        cb = PluginUtil.createContributionObject(this, contrib) as BinaryModule;
                        if (cb == null) {
                            // create dummy module for list Dialog
                            cb = new BinaryModule(this, contrib);
                        }
                    } else {
                        continue;
                    }
                    if (Contribution.IsPrimitiveContribution(cb.GetType())) {
                        Primitives.Add(cb);
                    } else {
                        Contributions.Add(cb);
                    }
                    cb.Attach();
                    cb._state = InstallationState.Ready;
                } catch (Exception e) {
                    _state = InstallationState.FatalError;
                    string msg = MakeContribExceptionMessage(contrib);
                    throw new Exception(msg, e);
                }
            }
        }

        /// <summary>
        /// Loads contributions from this plug-in
        /// </summary>
        internal void loadContributions() {
            Contribution c = null;
            int count = 0;
            int errors = 0;
            // load contributions
            foreach (ParamsReader contrib in _reader.EnumChildren("contribution")) {
                try {
                    count++;
                    string type = contrib["type"].InnerText;
                    if ("factory".Equals(type) || "binary".Equals(type)) continue;	// ignore

                    PluginManager.theInstance.NotifyStartParse(this, contrib);
                    IContributionFactory factory = PluginManager.theInstance.GetContributionFactory(type);
                    c = factory.load(this, contrib);
                    Contributions.Add(c);
                    c.Attach();
                    detachable |= c.IsDetachable;
                    if (c.IsDetachable) {
                        totalDetachables++;
                        c.OnAttach += new AttachChangeEvent(this.AttachChangeEventHandler);
                        c.OnDetach += new AttachChangeEvent(this.AttachChangeEventHandler);
                    }
                    PluginManager.theInstance.AddContribution(c);
                    c._state = InstallationState.Ready;
                } catch (Exception e) {
                    errors++;
                    Debug.WriteLine(e.Message);
                    Debug.WriteLine(e.StackTrace);
                    if (e.InnerException != null) {
                        Debug.WriteLine(e.InnerException.Message);
                        Debug.WriteLine(e.InnerException.StackTrace);
                    }
                    if (c != null)
                        c._state = InstallationState.FatalError;
                    string msg = MakeContribExceptionMessage(contrib);
                    if (_state != InstallationState.FatalError)
                        _state = InstallationState.PartialError;
                    PluginManager.theInstance.ReportError(msg, e);
                }
            }
            if (_state != InstallationState.FatalError && errors == count) {
                _state = InstallationState.FatalError;
            } else if (_state == InstallationState.Uninitialized) {
                _state = InstallationState.Ready;
            }
        }

        private string MakeContribExceptionMessage(ParamsReader pr) {
            string templ = Main.resources["plugin.contrib_load_error"].stringValue;
            string _id = pr["id"].InnerTextOr("unknown");
            string _name = pr["name"].InnerTextOr("unknown");
            return string.Format(templ, Uri.AbsoluteUri, _name, _id);
        }
        */
    }
}
