/*
 * JHPdf Free PDF Library : HPdfReadStream.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.io;

import java.nio.charset.Charset;
import java.util.EnumSet;

import net.sf.jhpdf.encrypt.HPdfEncrypt;

/**
 * Stream for reading interface.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public interface HPdfReadStream extends HPdfStream {
    
    /**
     * mode for seek operation.
     * corresponds 'whence' arguments of C xseek().
     */
    public enum SeekMode {
        SET,
        CUR,
        END,
        ;
    }

    /**
     * read sequence of byte into buffer.<br>
     * this method should behave same as calling {@link #read(buf, 0, siz)}
     * @param buf buffer for storing read values.
     * @param siz number of required values.
     * @return number of really read values or -1 when EOF.
     */
    public int read(byte[] buf, int siz);
    
    /**
     * read sequence of byte into buffer. 
     * @param buf buffer for storing read values.
     * @param bufOffset offset from head of buffer when storing values.
     * @param siz number of required values.
     * @return number of really read values or -1 when EOF.
     */
    public int read(byte[] buf, int bufOffset, int siz);
    
    /**
     * readLn() method determines stream encoding by this method return value.
     * @return stream encoding charset (considered only in readLn())
     */
    public Charset getCharsetForReadLn();
    
    /**
     * read a 'line' from stream.<br>
     * end of 'line' is byte-value 0x0a (perhaps followed 0x0d) or EOF.<br>
     * <strong>byte-sequence is considered as encoded with charset
     *  getCharsetForReadLn() returns.</strong>
     * @return one 'line' String.
     */
    public String readLn();
    
    /**
     * move position to start reading.
     * @param offset offset from position indicated by mode.
     * @param mode indicates seek behavior.<br>
     * SET means absolute position from head of stream.<br>
     * CUR means relative to current reading position.<br>
     * END means absolute position from end of stream.<br>
     * seek(0, END) means 'set position to end of stream',
     * so following read() will return -1 (and no data retrieved).
     */
    public void seek(long offset, SeekMode mode);
    
    /**
     * get current position to start reading.
     * @return current position to start reading.
     */
    public long tell();
    
    /**
     * write entire this stream to another stream.
     * @param dst stream to write this stream content.
     * @param filters set of filters used when writing.
     * @param e encrypt worker used when writing. (can be null when no encrypt needed)
     */
    public void writeToStream(HPdfWriteStream dst, EnumSet<HPdfFilterFlag> filters, HPdfEncrypt e);
    
    /**
     * convenience method of {@link #writeToStream(HPdfWriteStream, EnumSet<HPdfFilterFlag>, HPdfEncrypt)}
     * when using only single or no filter.
     * @param dst stream to write this stream content.
     * @param filter filter used when writing.
     * @param e encrypt worker used when writing. (can be null when no encrypt needed)
     */
    public void writeToStream(HPdfWriteStream dst, HPdfFilterFlag filter, HPdfEncrypt e);
}
