/*
 * JHPdf Free PDF Library : HPdfDocument.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import java.util.EnumSet;
import java.util.List;
import java.util.ArrayList;

import net.sf.jhpdf.JHPdf;
import net.sf.jhpdf.HPdfPdfVer;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encoder.HPdfEncoder;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.encrypt.HPdfEncrypt.HPdfEncryptMode;
import net.sf.jhpdf.encrypt.HPdfEncrypt.HPdfPdfPermission;
import net.sf.jhpdf.encrypt.HPdfEncryptDict;
import net.sf.jhpdf.graphics.HPdfColorSpace;
import net.sf.jhpdf.io.HPdfStream;
import net.sf.jhpdf.io.HPdfWriteStream;
import net.sf.jhpdf.io.HPdfFileWriteStream;
import net.sf.jhpdf.io.HPdfMemStream;
import net.sf.jhpdf.pdfobject.HPdfCatalog.HPdfPageLayout;
import net.sf.jhpdf.pdfobject.HPdfCatalog.HPdfPageMode;
import net.sf.jhpdf.pdfobject.HPdfCatalog.HPdfViewerPreference;
import net.sf.jhpdf.pdfobject.HPdfInfo.HPdfInfoType;
import net.sf.jhpdf.pdfobject.HPdfPageLabel.HPdfPageNumStyle;
import net.sf.jhpdf.pdfobject.font.HPdfFontDef;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class represents PDF Document.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public final class HPdfDocument {

    private static final Logger logger = LoggerFactory.getLogger(HPdfDocument.class);
    
    public enum HPdfCompressionMode {
        TEXT,
        IMAGE,
        METADATA,
        ALL,
    };
    
    private static final long HPDF_SIG_BYTES = 0x41504446L;
    
    private long sigBytes;

    /**
     * ctor.
     */
    public HPdfDocument() {
        logger.trace("HPdfDocument#ctor");
        
        this.sigBytes = HPDF_SIG_BYTES;
        
        setPdfVersion(HPdfPdfVer.HPDF_VER_13);
        
        newDoc();
    }
    
    public void newDoc() {
        logger.trace("HPdfDocument#newDoc");
        
        if (!this.validate()) {
            throw new HPdfException(HPdfErrorCode.HPDF_DOC_INVALID_OBJECT, 0);
        }
        
        this.freeDoc();
        
        this.setXref(new HPdfXref());
        this.setTrailer(this.getXref().getTrailer());
        
        this.clearFontMgr();
        
        if (this.getFontDefList() == null) {
            this.createFontDefList();
        }
        
        if (this.getEncoderList() == null) {
            this.createEncoderList();
        }
        
        this.setCatalog(new HPdfCatalog(this.getXref()));
        this.setRootPages(this.getCatalog().getRoot());
        
        this.createPageList();
        
        this.setCurPages(this.getRootPages());
        
        this.setInfoAttr(HPdfInfoType.PRODUCER, 
            "JHPDF Free PDF Library " + JHPdf.getVersion());
    }
    
    public void freeDoc() {
        logger.trace("HPdfDocument#freeDoc");
        
        if (this.validate()) {
            this.setXref(null);
            
            this.clearFontMgr();
            
            if (this.getFontDefList() != null) {
                this.cleanupFontDefList();
            }
            
            this.clearTTFontTag();
            
            setPdfVersion(HPdfPdfVer.HPDF_VER_13);
            this.setOutlines(null);
            this.setCatalog(null);
            this.setRootPages(null);
            this.setCurPages(null);
            this.setCurrentPage(null);
            this.setEncryptOn(false);
            this.setCurPageNum(0);
            this.setCurrentEncoder(null);
            this.setDefEncoder(null);
            this.setPagePerPages(0);
            
            this.clearPageList();
            
            this.setEncryptDict(null);
            this.setInfo(null);
            
            this.disposeStream();
        }
    }
    
    public void freeDocAll() {
        logger.trace("HPdfDocument#freeDocAll");
        
        if (this.validate()) {
            this.freeDoc();
            
            if (this.getFontDefList() != null) {
                freeFontDefList();
            }
            
            if (this.getEncoderList() != null) {
                freeEncoderList();
            }
            
            clearCompressionMode();
        }
    }
    
    public boolean hasDoc() {
        if (!this.validate()) {
            return false;
        }
        if (this.getCatalog() == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DOCUMENT, 0);
        }
        return true;
    }
    
    public boolean validate() {
        logger.trace("HPdfDocument#validate");
        return this.sigBytes == HPDF_SIG_BYTES;
    }
    
    public HPdfPage addPage() {
        logger.trace("HPdfDocument#addPage");
        
        if (!this.hasDoc()) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DOCUMENT, 0);
        }
        
        if (this.getPagePerPages() != 0) {
            if (this.getPagePerPages() <= this.getCurPageNum()) {
                this.setCurPages(this.addPagesTo(this.getRootPages()));
                this.setCurPageNum(0);
            }
        }
        
        HPdfPage page = new HPdfPage(this.getXref());
        
        this.getCurPages().addKids(page);
        
        this.getPageList().add(page);
        
        this.setCurrentPage(page);
        
        if (this.getCompressionMode().contains(HPdfCompressionMode.TEXT)) {
            page.setFilter(HPdfStream.HPdfFilterFlag.FLATE_DECODE);
        }
        
        this.setCurPageNum(this.getCurPageNum() + 1);
        
        return page;
    }
    
    public void insertPage(HPdfPage page) {
        // TODO only stub
    }
    
    private HPdfPages addPagesTo(HPdfPages parent) {
        logger.trace("HPdfDocument#addPagesTo");
        
        if (!this.hasDoc()) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DOCUMENT, 0);
        }
        
        // MEMO: original code performs HPDF_Pages_Validate (parent)
        // but validation is almost covered by type check.
        // only null check is required.
        if (parent == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_PAGES, 0);
        }
        
        HPdfPages pages = new HPdfPages(parent, this.getXref());
        this.setCurPages(pages);
        
        return pages;
    }
    
    public void addPageLabel(int pageNum, HPdfPageNumStyle style, int firstPage, String prefix) {
        // TODO only stub.
    }

    public void setOpenAction(HPdfDestination openAction) {
        // TODO ꂽ\bhEX^u
        
    }

    public void attachFile(String file) {
        // TODO ꂽ\bhEX^u
        
    }
    
    public void saveToStream() {
        // TODO only stub.
    }
    
    public byte[] getContents() {
        // TODO only stub.
        return null;
    }
    
    public long getStreamSize() {
        // TODO only stub.
        return 0;
    }
    
    public byte[] readFromStream() {
        // TODO only stub.
        return null;
    }
    
    public void resetStream() {
        // TODO only stub.
    }
    
    public void saveToFile(String filePath) throws HPdfException {
        logger.trace("HPdfDocument#saveToFile");
        
        if (!this.hasDoc()) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DOCUMENT, 0);
        }
        HPdfFileWriteStream stream = null;
        try {
            stream = new HPdfFileWriteStream(filePath);
            internalSaveToStream(stream);
        } finally {
            if (stream != null) {
                stream.close();
            }
        }
    }

    public HPdfEncoder getEncoder(String encodingName) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public void setPassword(String ownerPasswd, String userPasswd) {
        // TODO ꂽ\bhEX^u
        
    }

    public HPdfOutline createOutline(HPdfOutline parent, String title,
            HPdfEncoder encoder) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfExtGState createExtGState() {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public String loadType1FontFromFile(String afmFileName, String dataFileName) {
        // TODO ꂽ\bhEX^u
        return null;
    }
    
    public HPdfFontDef getTTFontDefFromFile(String fileName, boolean embedding){
        // TODO ꂽ\bhEX^u
        return null;
    }

    public String loadTTFontFromFile(String fileName, boolean embedding) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public String loadTTFontFromFile2(String fileName, int index, boolean embedding) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public void useJPEncodings() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useJPFonts() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useKREncodings() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useKRFonts() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useCNSEncodings() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useCNSFonts() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useCNTEncodings() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useCNTFonts() {
        // TODO ꂽ\bhEX^u
        
    }

    public void useUTFEncodings() {
        // TODO ꂽ\bhEX^u
        
    }

    public HPdfImage loadPngImageFromMem(byte[] buffer, int size) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadPngImageFromFile(String fileName) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadPngImageFromFile2(String fileName) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadJpegImageFromFile(String fileName) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadJpegImageFromMem(byte[] buffer, int size) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadU3DFromFile(String fileName) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadRaw1BitImageFromMem(byte[] buffer, int width, int height, int lineWidth, boolean blackIs1, boolean topIsFirst) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadRawImageFromFile(String filename, int width, int height, HPdfColorSpace colorSpace) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    public HPdfImage loadRawImageFromMem(byte[] buffer, int width, int height, HPdfColorSpace colorSpace, int bitsPerComponent) {
        // TODO ꂽ\bhEX^u
        return null;
    }
    
    public HPdfOutputIntent loadIccFromMem(HPdfStream iccData, HPdfXref xref, int numComponent) {
        // TODO only stub.
        return null;
    }
    
    public HPdfOutputIntent loadIccFromFile(String iccFileName, int numComponent) {
        // TODO only stub.
        return null;
    }
    
    public HPdfPage getPageByIndex(int index) {
        // TODO only stub.
        return null;
    }
    
    private HPdfPdfVer pdfVersion;
    
    private HPdfPdfVer getPdfVersion() {
        return this.pdfVersion;
    }
    
    private void setPdfVersion(HPdfPdfVer v) {
        this.pdfVersion = v;
    }
    
    private HPdfPageLayout pageLayout;
    
    /**
     * @return pageLayout
     */
    public HPdfPageLayout getPageLayout() {
        return pageLayout;
    }

    /**
     * @param pageLayout ݒ肷 pageLayout
     */
    public void setPageLayout(HPdfPageLayout pageLayout) {
        this.pageLayout = pageLayout;
    }
    
    private HPdfCatalog catalog;
    
    private HPdfCatalog getCatalog() {
        return this.catalog;
    }
    
    private void setCatalog(HPdfCatalog catalog) {
        this.catalog = catalog;
    }
    
    private HPdfOutline outlines;
    
    private HPdfOutline getOutlines() {
        return this.outlines;
    }
    
    private void setOutlines(HPdfOutline outlines) {
        this.outlines = outlines;
    }
    
    private HPdfXref xref;
    
    private HPdfXref getXref() {
        return this.xref;
    }
    
    private void setXref(HPdfXref xref) {
        this.xref = xref;
    }
    
    private HPdfPages rootPages = null;
    
    private HPdfPages getRootPages() {
        return this.rootPages;
    }
    
    private void setRootPages(HPdfPages pages) {
        this.rootPages = pages;
    }
    
    private HPdfPages curPages;
    
    private HPdfPages getCurPages() {
        return this.curPages;
    }
    
    private void setCurPages(HPdfPages pages) {
        this.curPages = pages;
    }
    
    private HPdfPage curPage;
    
    public HPdfPage getCurrentPage() {
        return this.curPage;
    }
    
    private void setCurrentPage(HPdfPage page) {
        this.curPage = page;
    }
    
    private List<HPdfPage> pageList = null;
    
    private List<HPdfPage> getPageList() {
        return this.pageList;
    }
    
    private void createPageList() {
        this.pageList = new ArrayList<HPdfPage>();
    }
    
    private void clearPageList() {
        this.pageList = null;
    }
    
    private HPdfInfo info = null;
    
    private HPdfInfo getInfo() {
        if (!this.hasDoc()) {
            return null;
        }
        if (this.info == null) {
            HPdfInfo info = new HPdfInfo();
            this.getXref().add(info);
        }
        return this.info;
    }
    
    private void setInfo(HPdfInfo info) {
        this.info = info;
    }
    
    public void setInfoAttr(HPdfInfoType iType, String value) {
        HPdfInfo info = this.getInfo();
        
        if (info == null) {
            // MEMO: original code call 'HPDF_CheckError'.
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_DOCUMENT, 0);
        }
        info.setInfoAttr(iType, value, this.getCurrentEncoder());
    }
    
    public String getInfoAttr(HPdfInfoType iType) {
        // TODO only stub.
        return null;
    }
    
    public void setInfoDateAttr(HPdfInfoType iType, HPdfDate value) {
        // TODO only stub
    }
    
    private HPdfDict trailer;
    
    private HPdfDict getTrailer() {
        return this.trailer;
    }
    
    private void setTrailer(HPdfDict trailer) {
        this.trailer = trailer;
    }
    
    private List<HPdfFont> fontMgr = new ArrayList<HPdfFont>();
    
    private List<HPdfFont> getFontMgr() {
        return this.fontMgr;
    }
    
    private void clearFontMgr() {
        getFontMgr().clear();
    }
    
    public HPdfFont getFont(String fontName, Object param) {
        // TODO only stub
        return null;
    }
    
    private int[] ttfontTag = new int[0];
    
    private void clearTTFontTag() {
        this.ttfontTag = new int[6];
        for (int i = 0; i < this.ttfontTag.length; ++i) {
            this.ttfontTag[i] = 0;
        }
    }
    
    private List<HPdfFontDef> fontDefList = null;
    
    private List<HPdfFontDef> getFontDefList() {
        return this.fontDefList;
    }
    
    private void createFontDefList() {
        this.fontDefList = new ArrayList<HPdfFontDef>();
    }
    
    private void cleanupFontDefList() {
        logger.trace("HPdfDocument#cleanupFontDefList");
        for(final HPdfFontDef def : this.getFontDefList()) {
            def.cleanup();
        }
    }
    
    private void freeFontDefList() {
        logger.trace("HPdfDocument#freeFontDefList");
        for(final HPdfFontDef def : this.getFontDefList()) {
            def.dispose();
        }
        this.fontDefList = null;
    }
    
    private List<HPdfEncoder> encoderList = null;
    
    private List<HPdfEncoder> getEncoderList() {
        return this.encoderList;
    }
    
    private void createEncoderList() {
        this.encoderList = new ArrayList<HPdfEncoder>();
    }
    
    private void freeEncoderList() {
        logger.trace("HPdfDocument#freeEncoderList");
        this.encoderList = null;
    }
    
    private HPdfEncoder curEncoder = null;
    
    private HPdfEncoder getCurrentEncoder() {
        return this.curEncoder;
    }
    
    private void setCurrentEncoder(HPdfEncoder encoder) {
        this.curEncoder = encoder;
    }
    
    private EnumSet<HPdfCompressionMode> compressionMode = EnumSet.noneOf(HPdfCompressionMode.class);
    
    private EnumSet<HPdfCompressionMode> getCompressionMode() {
        return this.compressionMode;
    }
    
    public void setCompressionMode(HPdfCompressionMode mode) {
        if (mode == HPdfCompressionMode.ALL) {
            this.getCompressionMode().add(HPdfCompressionMode.TEXT);
            this.getCompressionMode().add(HPdfCompressionMode.IMAGE);
            this.getCompressionMode().add(HPdfCompressionMode.METADATA);
        } else {
            this.getCompressionMode().add(mode);
        }
    }
    
    private void resetCompressionMode(HPdfCompressionMode mode) {
        if (mode == HPdfCompressionMode.ALL) {
            this.clearCompressionMode();
        } else {
            this.getCompressionMode().remove(mode);
        }
    }
    
    private void clearCompressionMode() {
        this.getCompressionMode().clear();
    }
    
    private EnumSet<HPdfPdfPermission> permission = EnumSet.noneOf(HPdfPdfPermission.class);
    
    private EnumSet<HPdfPdfPermission> getPermission() {
        return this.permission;
    }
    
    public void setPermission(HPdfPdfPermission permission) {
        this.getPermission().add(permission);
    }
    
    private void resetPermission(HPdfPdfPermission permission) {
        this.getPermission().remove(permission);
    }
    
    private boolean encryptOn;
    
    private boolean isEncryptOn() {
        return this.encryptOn;
    }
    
    private void setEncryptOn(boolean flg) {
        this.encryptOn = flg;
    }

    public void setEncryptionMode(HPdfEncryptMode mode, int keyLen) {
        // TODO ꂽ\bhEX^u
        
    }
    
    private HPdfEncryptDict encryptDict;
    
    private HPdfEncryptDict getEncryptDict() {
        return this.encryptDict;
    }
    
    private void setEncryptDict(HPdfEncryptDict dict) {
        this.encryptDict = dict;
    }
    
    private HPdfEncoder defEncoder = null;
    
    private HPdfEncoder getDefEncoder() {
        return this.defEncoder;
    }
    
    private void setDefEncoder(HPdfEncoder encoder) {
        this.defEncoder = encoder;
    }
    
    private int pagePerPages;
    
    private int getPagePerPages() {
        return this.pagePerPages;
    }
    
    private void setPagePerPages(int num) {
        this.pagePerPages = num;
    }

    public void setPagesConfiguration(int pagePerPages) {
        // TODO ꂽ\bhEX^u
        
        this.setPagePerPages(pagePerPages);
    }
    
    private int curPageNum;
    
    private int getCurPageNum() {
        return this.curPageNum;
    }
    
    private void setCurPageNum(int num) {
        this.curPageNum = num;
    }
    
    private EnumSet<HPdfViewerPreference> viewerPreference =
            EnumSet.noneOf(HPdfViewerPreference.class);
    
    private EnumSet<HPdfViewerPreference> getViewerPreference() {
        return this.viewerPreference;
    }
    
    public void setViewerPreference(HPdfViewerPreference mode) {
        this.getViewerPreference().add(mode);
    }
    
    private void resetViewerPreference(HPdfViewerPreference mode) {
        this.getViewerPreference().remove(mode);
    }
    
    private HPdfMemStream docStream = null;
    
    private HPdfMemStream getStream() {
        return this.docStream;
    }
    
    private void createStream(int size) {
        this.docStream = new HPdfMemStream(size);
    }
    
    private void disposeStream() {
        if (this.docStream != null) {
            this.docStream.close();
            this.docStream = null;
        }
    }
    
    private HPdfPageMode pageMode;
    
    private HPdfPageMode getPageMode() {
        return pageMode;
    }

    public void setPageMode(HPdfPageMode useOutline) {
        // TODO ꂽ\bhEX^u
        
    }
    
    private void prepareEncryption() {
        // TODO: only stub
    }
    
    private void internalSaveToStream(HPdfWriteStream stream) {
        writeHeader(stream);
        prepareTrailer(stream);
        if (this.isEncryptOn()) {
            HPdfEncrypt e = this.getEncryptDict().getAttr();
            this.prepareEncryption();
            this.getXref().writeToStream(stream, e);
        } else {
            this.getXref().writeToStream(stream, null);
        }
    }
    
    private void writeHeader(HPdfWriteStream stream) {
        logger.trace("HPdfDocument#writeHeader");
        
        stream.writeStr(this.getPdfVersion().getHeader());
    }
    
    private void prepareTrailer(HPdfWriteStream stream) {
        logger.trace("HPdfDocument#prepareTrailer");
        
        HPdfDict trailer = this.getTrailer();
        trailer.add("Root", this.getCatalog());
        trailer.add("Info", this.getInfo());
    }

}
