/*
 * JHPdf Free PDF Library : HPdfXref.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import java.util.List;
import java.util.ArrayList;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.io.HPdfWriteStream;
import net.sf.jhpdf.pdfobject.HPdfObject.HPdfObjectType;
import net.sf.jhpdf.pdfobject.HPdfXrefEntry.HPdfEntryType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class represents PDF Xref.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public final class HPdfXref {

    private static final Logger logger = LoggerFactory.getLogger(HPdfXref.class);
    
    public static final int HPDF_BYTE_OFFSET_LEN = 10;
    public static final int HPDF_OBJ_ID_LEN = 10;
    public static final int HPDF_GEN_NO_LEN = 5;
    
    private int startOffset;
    private long addr;
    
    private HPdfXref prev;
    private List<HPdfXrefEntry> entries = new ArrayList<HPdfXrefEntry>();
    private HPdfDict trailer;
    
    /**
     * ctor. this is package-private.
     */
    HPdfXref() {
        super();
        setStartOffset(0);
        setPrev(null);
        setAddr(0);
        
        this.trailer = new HPdfDict();
    }
    
    public int getStartOffset() {
        return this.startOffset;
    }
    
    public void setStartOffset(int offset) {
        this.startOffset = offset;
    }
    
    public HPdfXref getPrev() {
        return this.prev;
    }
    
    public void setPrev(HPdfXref prev) {
        this.prev = prev;
    }
    
    public long getAddr() {
        return this.addr;
    }
    
    public void setAddr(long addr) {
        this.addr = addr;
    }
    
    public List<HPdfXrefEntry> getEntries() {
        // TODO: maybe prefers to return immutable list?
        return this.entries;
    }
    
    public HPdfDict getTrailer() {
        return this.trailer;
    }
    
    public void add(HPdfObject obj) {
        logger.trace("HPdfXref#add");
        
        if (obj == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        
        if (obj.isTypeDirect() || obj.isTypeIndirect()) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        
        // MEMO: original code inspects internal list size here(HPDF_LIMIT_MAX_XREF_ELEMENT).
        // it should be class-private concerns, but may PDF specification?
        
        HPdfXrefEntry entry = new HPdfXrefEntry();
        this.entries.add(entry);
        
        entry.setEntryType(HPdfEntryType.IN_USE_ENTRY);
        entry.setByteOffset(0);
        entry.setGenNo(0);
        entry.setPdfObject(obj);
        obj.setObjId(getStartOffset() + this.entries.size() - 1);
        obj.setObjType(HPdfObjectType.INDIRECT);
        
        obj.setGenNo(entry.getGenNo());
    }
    
    public void writeToStream(HPdfWriteStream stream, HPdfEncrypt e) {
        logger.trace("HPdfXref#writeToStream");
        
        StringBuilder buf = new StringBuilder();
        
        HPdfXref tmpXref = this;
        
        /* write each objects of xref to the specified stream */
        while (tmpXref != null) {
            int strIdx;
            if (tmpXref.getStartOffset() == 0) {
                strIdx = 1;
            } else {
                strIdx = 0;
            }
            int i = -1;
            int objId = strIdx;
            for (final HPdfXrefEntry entry : tmpXref.getEntries()) {
                ++i;
                if (i < strIdx) {
                    continue;
                }
                int genNo = entry.getGenNo();
                
                entry.setByteOffset(stream.getSize());
                buf.setLength(0);
                buf.append(objId);
                buf.append(' ');
                buf.append(genNo);
                buf.append(" obj\012");
                stream.writeStr(buf.toString());
                
                if (e != null) {
                    e.initKey(objId, genNo);
                }
                
                entry.getPdfObject().writeValue(stream, e);
                
                stream.writeStr("\012endobj\012");
                
                ++objId;
            }
            tmpXref = tmpXref.getPrev();
        }
        
        /* start to write cross-reference table */
        tmpXref = this;
        while (tmpXref != null) {
            tmpXref.setAddr(stream.getSize());
            
            buf.setLength(0);
            buf.append("xref\012");
            buf.append(tmpXref.getStartOffset());
            buf.append(' ');
            buf.append(tmpXref.getEntries().size());
            buf.append("\012");
            stream.writeStr(buf.toString());
            
            for (final HPdfXrefEntry entry : tmpXref.getEntries()) {
                buf.setLength(0);
                buf.append(String.format("%0" + HPDF_BYTE_OFFSET_LEN + "d", entry.getByteOffset()));
                buf.append(' ');
                buf.append(String.format("%0" + HPDF_GEN_NO_LEN + "d", entry.getGenNo()));
                buf.append(' ');
                buf.append(entry.getEntryType().getPdfRep());
                buf.append("\015\012"); /* Acrobat 8.15 requires both \r and \n here */
                stream.writeStr(buf.toString());
            }
            
            tmpXref = tmpXref.getPrev();
        }
        writeTrailer(stream);
    }
    
    private void writeTrailer(HPdfWriteStream stream) {
        logger.trace("HPdfXref#writeTrailer");
        
        int maxObjId = this.getEntries().size() + this.getStartOffset();
        HPdfDict trailer = this.getTrailer();
        trailer.addNumber("Size", maxObjId);
        trailer.addNumber("Prev", this.getPrev().getAddr());
        
        stream.writeStr("trailer\012");
        trailer.writeValue(stream, null);
        stream.writeStr("\012startxref\012");
        stream.writeUInt(this.getAddr());
        stream.writeStr("\012%%EOF\012");
    }
}
