require 'test/unit'
require_relative 'parser.rb'
require_relative 'ast.rb'

class ProtobufParserTest < Test::Unit::TestCase
  def test_parse
    src = <<EOS
message Person {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_import
    src = <<EOS
import "hoge/huga";
import "";

message Person {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_import_only
    src = <<EOS
import "hoge/huga";
import public "hoge/old";
import "";
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_import_package
    src = <<EOS
import "hoge/huga";
import "";

package foo;

message Person {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_enum
    src = <<EOS
message TestMessage {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;

  enum TestEnum {
    VAL0 = 0;
    VAL1 = 1;
    VAL1 = 2;
  }
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_top_level_enum
    src = <<EOS
enum TestEnum {
  VAL0 = 0;
  VAL1 = 1;
  VAL1 = 2;
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_nested_message
    src = <<EOS
message TestMessage {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;

  message NestedMessage {
    required int32 aaa = 1;
    required string bbb = 2;
    optional string bbb = 2;
  }
}
EOS
    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_scalar_field_type
    src = <<EOS
message TestMessage {
  required double field1 = 1;
  required float field2 = 2;
  required int32 field3 = 3;
  required int64 field4 = 4;
  required uint32 field5 = 5;
  required uint64 field6 = 6;
  required sint32 field7 = 7;
  required sint64 field8 = 8;
  required fixed32 field9 = 9;
  required fixed64 field10 = 10;
  required sfixed32 field11 = 11;
  required sfixed64 field12 = 12;
  required bool field13 = 13;
  required string field14 = 14;
  required bytes field15 = 15;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_composite_field_type
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1;
  required Test1.Test2 field2 = 2;
  required Test1.Test2.Test3 field3 = 3;
  required Test1.Test2.Test3.Test4 field4 = 4;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_oneof
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1;
  oneof one_of_name {
    string field2 = 2;
    Test1 field3 = 3;
    Test1.Test2 field4 = 4;
  }
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_oneof_illegal
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1;
  oneof one_of_name {
    required string field2 = 2;
    Test1 field3 = 3;
    Test1.Test2 field4 = 4;
  }
}
EOS

    assert_raise(Racc::ParseError, "illegal pass") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_tokens
    src = <<EOS
message TestMessage {
  required Test1 message_field = 1;
  required Test1 package_field = 2;
  required Test1 import_field = 3;
  required Test1 enum_field = 4;
  required Test1 oneof_field = 5;
  required Test1 required_field = 6;
  required Test1 optional_field = 7;
  required Test1 repeated_field = 8;
  required Test1 public_field = 9;
  required Test1 double_field = 10;
  required Test1 float_field = 11;
  required Test1 int32_field = 12;
  required Test1 int64_field = 13;
  required Test1 uint32_field = 14;
  required Test1 uint64_field = 15;
  required Test1 sint32_field = 16;
  required Test1 sint64_field = 17;
  required Test1 fixed32_field = 18;
  required Test1 fixed64_field = 19;
  required Test1 sfixed32_field = 20;
  required Test1 sfixed64_field = 21;
  required Test1 bool_field = 22;
  required Test1 string_field = 23;
  required Test1 bytes_field = 24;
}
EOS

    assert_nothing_raised("scan fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end


  def test_parse_top_level_option
    src = <<EOS
option aaa = "value";
option bbb = false;
option (ccc) = custom;

message TestMessage {
  required Test1 field1 = 1;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_message_level_option
    src = <<EOS
message TestMessage {
  option aaa = "value";
  option bbb = false;
  option (ccc) = custom;
  required Test1 field1 = 1;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_enum_level_option
    src = <<EOS
message TestMessage {
  required int32 test = 1;

  enum TestEnum {
    option aaa = "value";
    option bbb = false;
    option (ccc) = custom;
    VAL0 = 0;
  }
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_parse_field_level_option
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1 [aaa = "value"];
  required Test1 field2 = 1 [bbb = false];
  required Test1 field3 = 1 [(ccc) = custom];
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_extentions
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1;
  extensions 123 to 456;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_extentions_to_max
    src = <<EOS
message TestMessage {
  required Test1 field1 = 1;
  extensions 123 to max;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_extend
    src = <<EOS
extend TestMessage {
  required Test1 field1 = 1;
}
EOS

    assert_nothing_raised("parse fail") {
      parser = Protobuf::Parser.new()
      parser.parse_start(src)
    }
  end

  def test_sample_files
    Dir::glob("sample/*.proto").each { |f|
      assert_nothing_raised("parse fail: #{f}") {
        File.open(f) { |fd|
          src = fd.read
          parser = Protobuf::Parser.new()
          parser.parse_start(src)
        }
      }
    }
  end

  def test_return_object_class
    src = <<EOS
message Person {
  required int32 test1 = 1;
  required string test2 = 2;
  optional string test3 = 3;
}
EOS
    parser = Protobuf::Parser.new()
    ir = parser.parse_start(src)

    assert(ir.is_a?(Protobuf::ASTRoot), "invalid object") { }
  end
end
