// -*- C++ -*-
/*!
 *
 * @file BufferBase.h
 * @brief Buffer abstract class
 * @date $Date: 2007-12-31 03:06:12 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: BufferBase.h 1704 2010-01-20 02:30:54Z fsi-katami $
 */

#ifndef RTC_BUFFERBASE_H
#define RTC_BUFFERBASE_H

#include <stddef.h>
#include <coil/Properties.h>
#include <rtm/BufferStatus.h>

/*!
 * @if jp
 * @namespace RTC
 *
 * @brief RTݡͥ
 *
 * @else
 *
 * @namespace RTC
 *
 * @brief RT-Component
 *
 * @endif
 */
namespace RTC
{
  /*!
   * @if jp
   * @class BufferBase
   * @brief BufferBase ݥ饹
   * 
   * ΥХåեΤݥ󥿡ե饹
   * ݥХåե饹ϡʲν貾۴ؿμ󶡤ʤФʤʤ
   * \<DataType\>ȤƥХåեݻǡꤹ롣
   *
   * public󥿡եȤưʲΤΤ󶡤롣
   * - length(): ХåեĹ֤
   * - length(n): ХåեĹn˥åȤ
   * - reset(): ХåեΥݥ󥿤ꥻåȤ
   *
   * ߴϢ
   * - wptr(n=0): ߤν񤭹оݤǤnΥݥ󥿤֤
   * - advanceWptr(n=1): ߥݥ󥿤nʤ롣
   * - put(): ߤν񤭹֤߰˽񤭹ࡢݥ󥿤Ͽʤʤ
   * - write(): Хåե˽񤭹ࡣݥ󥿤1Ĥࡣ
   * - writable(): ߲ǽǿ֤
   * - full(): Хåեե֡
   *
   * ɤ߽ФϢ
   * - rptr(n=0): ߤɤ߽ФоݤnΥݥ󥿤֤
   * - advanceRptr(n=1): ɤ߽Фݥ󥿤nʤ롣
   * - get(): ߤɤ߽Ф֤ɤࡣݥ󥿤Ͽʤʤ
   * - read(): Хåեɤ߽Фݥ󥿤1Ĥࡣ
   * - readable(): ɤ߽Фǽǿ֤
   * - empty(): Хåե֡
   *
   * @param DataType Хåե˳Ǽǡ
   *
   * @since 0.4.0
   *
   * @else
   * @class BufferBase
   * @brief BufferBase abstract class
   *
   * This is the abstract interface class for various Buffer.
   * Concrete buffer classes must implement the following pure virtual
   * functions.
   * The users specify data type to hold it in a buffer as \<DataType\>.
   *
   * This class provides public interface as follows.
   * - write(): Write data into the buffer.
   * - read(): Read data from the buffer.
   * - length(): Get the buffer length.
   * - isFull(): Check on whether the buffer is full.
   * - isEmpty(): Check on whether the buffer is empty.
   *
   * This class provides protected interface as follows.
   * - put(): Store data into the buffer.
   * - get(): Get data from the buffer.
   *
   * @param DataType Data type to be stored to the buffer.
   *
   * @since 0.4.0
   *
   * @endif
   */
  template <class DataType>
  class BufferBase
    : public BufferStatus
  {
  public:
    BUFFERSTATUS_ENUM

    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * @else
     *
     * @brief Virtual destructor
     *
     * @endif
     */
    virtual ~BufferBase(void)
    {
    };
    
    /*!
     * @if jp
     *
     * @brief Хåե
     *
     * @else
     *
     * @brief Set the buffer
     *
     * @endif
     */
    virtual void init(const coil::Properties& prop) = 0;

    /*!
     * @if jp
     *
     * @brief ХåեĹ
     * 
     * ХåեĹ뤿ν貾۴ؿ
     * 
     * @return ХåեĹ
     * 
     * @else
     *
     * @brief Get the buffer length
     *
     * Pure virtual function to get the buffer length.
     *
     * @return buffer length
     * 
     * @endif
     */
    virtual size_t length(void) const = 0;

    /*!
     * @if jp
     *
     * @brief ХåեĹ򥻥åȤ
     * 
     * ХåեĹꤹ롣ԲĤʾNOT_SUPPORTED֤롣
     * 
     * @return BUFFER_OK: ｪλ
     *         NOT_SUPPORTED: ХåեĹѹԲ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Set the buffer length
     *
     * Pure virtual function to set the buffer length.
     *
     * @return BUFFER_OK: Successful
     *         NOT_SUPPORTED: The buffer length cannot be set.
     *         BUFFER_ERROR: Failed
     *         
     * 
     * @endif
     */    
    virtual ReturnCode length(size_t n) = 0;

    /*!
     * @if jp
     *
     * @brief Хåեξ֤ꥻåȤ
     * 
     * Хåեɤ߽Фݥ󥿤Ƚ񤭹ߥݥ󥿤ΰ֤ꥻåȤ롣
     * 
     * @return BUFFER_OK: ｪλ
     *         NOT_SUPPORTED: ꥻåԲǽ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Reset the buffer status
     *
     * Pure virtual function to reset the buffer status.
     *
     * @return BUFFER_OK: Successful
     *         NOT_SUPPORTED: The buffer status cannot be reset.
     *         BUFFER_ERROR: Failed
     * 
     * @endif
     */ 
    virtual ReturnCode reset() = 0;


    //----------------------------------------------------------------------
    /*!
     * @if jp
     *
     * @brief ХåեθߤνǤΥݥ
     * 
     * ХåեθߤνǤΥݥ󥿤ޤϡnΥݥ󥿤֤
     * 
     * @param  n ߥݥ + n ΰ֤Υݥ 
     * @return ֤߰Υݥ
     * 
     * @else
     *
     * @brief Get the writing pointer
     *
     * Pure virtual function to get the writing pointer.
     *
     * @param writeing pinter or n previous pointer
     * @return writing pointer
     * 
     * @endif
     */ 
    virtual DataType* wptr(long int n = 0) = 0;

    /*!
     * @if jp
     *
     * @brief ߥݥ󥿤ʤ
     * 
     * ߤν񤭹֤߰Υݥ󥿤 n Ŀʤ롣
     * 
     * @param  n ߥݥ + n ΰ֤Υݥ 
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Forward n writing pointers.
     *
     * Pure virtual function to forward n writing pointers.
     *
     * @return BUFFER_OK: Successful
     *         BUFFER_ERROR: Failed
     * 
     * @endif
     */ 
    virtual ReturnCode advanceWptr(long int n = 1) = 0;

    /*!
     * @if jp
     *
     * @brief Хåե˥ǡ񤭹
     * 
     * Хåե˥ǡ񤭹ࡣ񤭹ߥݥ󥿤ΰ֤ѹʤ
     * 
     * @param value 񤭹оݥǡ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     *
     * @param value Target data to write.
     *
     * @return BUFFER_OK: Successful
     *         BUFFER_ERROR: Failed
     *
     * @endif
     */
    virtual ReturnCode put(const DataType& value) = 0;

    /*!
     * @if jp
     *
     * @brief Хåե˥ǡ񤭹
     * 
     * Хåե˥ǡ񤭹ࡣ񤭹ߥݥ󥿤ΰ֤1Ĥࡣ
     * 
     * @param value 񤭹оݥǡ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to write data into the buffer.
     *
     * @param value Target data to write.
     *
     * @return BUFFER_OK: Successful
     *         BUFFER_ERROR: Failed
     *
     * @endif
     */
    virtual ReturnCode write(const DataType& value,
                             long int sec = -1, long int nsec = -1) = 0;

    /*!
     * @if jp
     *
     * @brief Хåե˽߲ǽǿ
     * 
     * Хåե˽߲ǽǿ֤
     * 
     * @return 񤭹߲ǽǿ
     *
     * 
     * @else
     *
     * @brief Get a writable number. 
     *
     * Pure virtual function to get a writable number.
     *
     * @return value writable number
     *
     *
     * @endif
     */
    virtual size_t writable() const = 0;

    /*!
     * @if jp
     *
     * @brief Хåեfullå
     * 
     * Хåեfullåѽ貾۴ؿ
     *
     * @return fullå(true:Хåեfullfalse:Хåե)
     * 
     * @else
     *
     * @brief Check on whether the buffer is full.
     *
     * Pure virtual function to check on whether the buffer is full.
     *
     * @return True if the buffer is full, else false.
     *
     * @endif
     */
    virtual bool full(void) const = 0;

    //----------------------------------------------------------------------
    /*!
     * @if jp
     *
     * @brief Хåեθߤɤ߽ФǤΥݥ
     * 
     * Хåեθߤɤ߽ФǤΥݥ󥿤ޤϡnΥݥ󥿤֤
     * 
     * @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
     * @return ɤ߽Ф֤Υݥ
     * 
     * @else
     *
     * @brief Get the reading pointer
     *
     * Pure virtual function to get the reading pointer.
     *
     * @return reading pointer
     * 
     * @endif
     */ 
    virtual DataType* rptr(long int n = 0) = 0;

    /*!
     * @if jp
     *
     * @brief ɤ߽Фݥ󥿤ʤ
     * 
     * ߤɤ߽Ф֤Υݥ󥿤 n Ŀʤ롣
     * 
     * @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Forward n reading pointers.
     *
     * Pure virtual function to forward n reading pointers.
     *
     * @return BUFFER_OK: Successful
     *         BUFFER_ERROR: Failed
     * 
     * @endif
     */ 
    virtual ReturnCode advanceRptr(long int n = 1) = 0;

    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåեǡɤߤɤ߽Фݥ󥿤ΰ֤ѹʤ
     * 
     * @param value ɤ߽Фǡ
     *
     * @return BUFFER_OK: ｪλ
     *         BUFFER_ERROR: ۾ｪλ
     * 
     * @else
     *
     * @brief Read data from the buffer
     *
     * Pure virtual function to read data form the buffer.
     *
     * @param value Data to read.
     *
     * @return BUFFER_OK: Successful
     *         BUFFER_ERROR: Failed
     *
     * @endif
     */
    virtual ReturnCode get(DataType& value) = 0;

    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåեǡɤߤɤ߽Фݥ󥿤ΰ֤ѹʤ
     * 
     * @return ɤ߽Фǡ
     * 
     * @else
     *
     * @brief Read data from the buffer
     *
     * Pure virtual function to read data from the buffer.
     *
     * @return Data got from buffer.
     *
     * @endif
     */
    virtual DataType&  get() = 0;

    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåեǡɤ߽Фν貾۴ؿ
     * 
     * @param value ɤ߽Фǡ
     *
     * @return ǡɤ߽Ф(true:ɤ߽Фfalse:ɤ߽Ф)
     * 
     * @else
     *
     * @brief Read data from the buffer
     *
     * Pure virtual function to read data from the buffer.
     *
     * @param value Read data.
     *
     * @return Result of having read (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual ReturnCode read(DataType& value,
                            long int sec = -1, long int nsec = -1) = 0;
    
    /*!
     * @if jp
     *
     * @brief Хåեɤ߽Фǽǿ
     * 
     * Хåեɤ߽Фǽǿ֤
     * 
     * @return ɤ߽Фǽǿ
     *
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Pure virtual function to get a reading number.
     *
     *
     * @return readable number
     *
     * @endif
     */
    virtual size_t readable() const = 0;

    /*!
     * @if jp
     *
     * @brief Хåեemptyå
     * 
     * Хåեemptyåѽ貾۴ؿ
     *
     * @return emptyå(true:Хåեemptyfalse:Хåեǡ)
     * 
     * @else
     *
     * @brief Check on whether the buffer is empty.
     *
     * Pure virtual function to check on whether the buffer is empty.
     *
     * @return True if the buffer is empty, else false.
     *
     * @endif
     */
    virtual bool empty(void) const = 0;

  };
  
  /*!
   * @if jp
   * @class NullBuffer
   * @brief ߡХåե饹
   * 
   * ХåեĹΥߡХåե饹
   * \<DataType\>ȤƥХåեݻǡꤹ롣
   *
   * @param DataType Хåե˳Ǽǡ
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class NullBuffer
   * @brief Concrete buffer class for dummy
   * 
   * Concrete buffer class for dummy. Buffer length is fixed to 1.
   * The users specify data type to hold it in a buffer as \<DataType\>.
   *
   * @param DataType Data type to hold in a buffer
   *
   * @since 0.4.0
   *
   * @endif
   */
  template <class DataType>
  class NullBuffer
    : public BufferBase<DataType>
  {
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     * ХåեĹ1()ǽ롣
     *
     * @param size ХåեĹ(̵)
     * 
     * @else
     *
     * @brief Constructer
     * 
     * Constructer.
     * Initialize buffer length to always 1.
     *
     * @param size Buffer length(Not use)
     * 
     * @endif
     */
    NullBuffer(long int size = 1)
      : m_length(1)
    {
    }
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     * 
     * ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * Destractor
     *
     * @endif
     */
    virtual ~NullBuffer(void)
    {
    }
    
    /*!
     * @if jp
     *
     * @brief ХåեĹ(1)
     * 
     * ХåեĹ롣(1֤)
     * 
     * @return ХåեĹ(1)
     * 
     * @else
     *
     * @brief Get the buffer length (always 1)
     *
     * Get the buffer length. (Return always 1.)
     *
     * @return buffer length(always 1)
     * 
     * @endif
     */
    virtual long int length(void) const
    {
      return 1;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåե˥ǡ񤭹
     * 
     * Ϳ줿ǡХåե˽񤭹ࡣ
     * 
     * @param value 񤭹оݥǡ
     *
     * @return ǡ񤭹߷(true:񤭹false:񤭹߼)
     * 
     * @else
     *
     * @brief Write data into the buffer
     *
     * Write data which were given with an argument into the buffer.
     *
     * @param value Target data to write.
     *
     * @return Result of having written in data (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual bool write(const DataType& value)
    {
      m_data = value;
      return true;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեǡɤ߽Ф
     * 
     * Хåե˳Ǽ줿ǡɤ߽Ф
     * 
     * @param value ɤ߽Фǡ
     *
     * @return ǡɤ߽Ф(true:ɤ߽Фfalse:ɤ߽Ф)
     * 
     * @else
     *
     * @brief Read data from the buffer
     *
     * Read data stored in the buffer.
     *
     * @param value Read data.
     *
     * @return Result of having read (true:Successful, false:Failed)
     *
     * @endif
     */
    virtual bool read(DataType& value)
    {
      value = m_data;
      return true;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեfullå
     * 
     * Хåեfullå롣(false֤)
     *
     * @return fullå(false)
     * 
     * @else
     *
     * @brief Check on whether the buffer is full.
     *
     * Check on whether the buffer is full. (Always false.)
     *
     * @return Always false.
     *
     * @endif
     */
    virtual bool isFull(void) const
    {
      return false;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեemptyå
     * 
     * Хåեemptyå롣(false֤)
     *
     * @return emptyå(false)
     * 
     * @else
     *
     * @brief Check on whether the buffer is empty.
     *
     * Check on whether the buffer is empty. (Always false.)
     *
     * @return Always false.
     *
     * @endif
     */
    virtual bool isEmpty(void) const
    {
      return false;
    }
    
  protected:
    /*!
     * @if jp
     *
     * @brief Хåե˥ǡǼ
     * 
     * Ϳ줿ǡХåե˳Ǽ롣
     * 
     * @param data оݥǡ
     * 
     * @else
     *
     * @brief Store data into the buffer
     *
     * Store data which were given with an argument into the buffer.
     *
     * @param data Target data to store.
     *
     * @endif
     */
    virtual void put(const DataType& data)
    {
      m_data = data;
    }
    
    /*!
     * @if jp
     *
     * @brief Хåեǡ
     * 
     * Хåե˳Ǽ줿ǡ롣
     *
     * @return ǡ
     * 
     * @else
     *
     * @brief Get data from the buffer
     *
     * Get data from the buffer.
     *
     * @return Data got from buffer.
     *
     * @endif
     */
    virtual const DataType& get(void)
    {
      return m_data;
    }
    
    /*!
     * @if jp
     *
     * @brief ˽񤭹ХåեؤλȤ
     * 
     * 񤭹ߥХåեؤλȤ롣
     * ܥХåեǤϥХåեĹϸǣǤ뤿ᡤ
     * Ʊ֤ؤλȤ֤
     *
     * @return ν񤭹оݥХåեؤλ()
     * 
     * @else
     *
     * @brief Get the buffer's reference to be written the next
     *
     * Get the reference to be written buffer.
     * Return always same position because this buffer's length is always 1.
     *
     * @return Reference to be written the next(always same)
     *
     * @endif
     */
    virtual DataType& getRef(void)
    {
      return m_data;
    }
    
  private:
    DataType m_data;
    long int m_length;
  };
}; // namespace RTC
#endif // BufferBase_h
