package online.model;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Timestamp;
import java.util.Map;
import java.util.Set;

/**
 * 汎用モデル
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public interface UniModel extends Serializable {

	/**
	 * 日時取得
	 *
	 * @return 日時
	 */
	Timestamp getPresent();

	/**
	 * 日時設定
	 *
	 * @param val 日時
	 */
	void setPresent(Timestamp val);

	/**
	 * 保存可状態取得
	 *
	 * @return 保存可能の場合 true を返す。
	 */
	boolean isReservable();

	/**
	 * キー値保存
	 *
	 * @param key キー
	 */
	void reserve(String... key);

	/**
	 * 保存マップ取得
	 *
	 * @return 保存マップ
	 */
	Set<String> getReservedKeySet();

	/**
	 * 値保持マップ取得
	 *
	 * @return Mapオブジェクト
	 */
	Map<String, Serializable> toMap();

	/**
	 * キーに関連した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Boolean value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Boolean... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, String value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, String... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Byte value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Byte... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Short value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Short... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Integer value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Integer... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Long value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Long... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Float value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Float... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Double value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Double... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, BigInteger value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, BigInteger... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, BigDecimal value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, BigDecimal... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Number value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, Number... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, java.util.Date value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, java.util.Date... value);

	/**
	 * キーに対応した値を保持する。
	 *
	 * @param key キー
	 * @param value 値
	 */
	void setValue(String key, UniModel value);

	/**
	 * キーに対応した値を取得する。
	 *
	 * @param key キー
	 * @return 値
	 */
	Boolean getBoolean(String key);

	/**
	 * キーに対応した値を取得する。
	 *
	 * @param key キー
	 * @return 値
	 */
	Boolean[] getBooleanArray(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーに対応した値がnullの場合、空文字列を返す。
	 * 指定したキーの値がモデルの場合、空文字列を返す。
	 * 指定したキーの値が配列の場合、添字0の値を返す。添字0が存在しない場合、空文字列を返す。
	 *
	 * @param key キー
	 * @return 値
	 */
	String getString(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーに対応した値がnullの場合、空配列を返す。
	 *
	 * @param key キー
	 * @return 値
	 */
	String[] getStringArray(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーの値が配列の場合、添字0の値を返す。添字0が存在しない場合、nullを返す。
	 *
	 * @param <T> Type
	 * @param key キー
	 * @return 値
	 */
	<T extends Number> T getNumber(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーに対応した値がnullの場合、空配列を返す。
	 *
	 * @param key キー
	 * @return 値
	 */
	Number[] getNumberArray(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーの値が配列の場合、添字0の値を返す。添字0が存在しない場合、nullを返す。
	 * @param <T> Type
	 * @param key キー
	 * @return 値
	 */
	<T extends java.util.Date> T getDate(String key);

	/**
	 * キーに対応した値を取得する。
	 * 指定したキーに対応した値がnullの場合、空配列を返す。
	 *
	 * @param key キー
	 * @return 値
	 */
	java.util.Date[] getDateArray(String key);

	/**
	 * キーに対応した値を取得する。
	 *
	 * @param key キー
	 * @return 値
	 */
	UniModel getModel(String key);

	/**
	 * キーに対応した値をmodelに複製設定する。
	 *
	 * @param model モデル
	 * @param key キー
	 */
	void copyValueTo(UniModel model, String... key);

	/**
	 * キーに対応した値をmodelに設定する。
	 *
	 * @param model モデル
	 * @param key キー
	 */
	void putValueTo(UniModel model, String... key);

	/**
	 * 他のモデルの内容を全て設定する。
	 *
	 * @param value モデル
	 */
	void putAll(UniModel value);

	/**
	 * 別名キーを作成する。
	 *
	 * @param key キー
	 * @param alias 別名キー
	 */
	void aliasKey(String key, String alias);

	/**
	 * キーが含まれているかを返す。
	 *
	 * @param key キー
	 * @return boolean
	 */
	boolean containsKey(String key);

	/**
	 * キーに対応した値の配列長を返す。
	 *
	 * @param key キー
	 * @return 配列長
	 */
	int getArraySize(String key);

	/**
	 * キーに対応した値のクラスを返す。
	 * @param <T> Type
	 * @param key キー
	 * @return クラス
	 */
	<T> Class<T> getValueClass(String key);

	/**
	 * 空値確認
	 * @param key キー
	 * @return 値が存在して、長さ1以上の配列の場合 true を返す。
	 */
	boolean hasValue(String key);

	/**
	 * 空値設定
	 *
	 * @param key キー
	 */
	void noValue(String... key);

	/**
	 * キーに対応した値を削除する。
	 *
	 * @param key キー
	 */
	void remove(String... key);

	/**
	 * キーの集合を返す。
	 *
	 * @return キー集合
	 */
	Set<String> keySet();

	/**
	 * 内容を消去する。
	 *
	 */
	void clear();

	/**
	 * 保持している値の数を返す。
	 *
	 * @return 保持数
	 */
	int size();

	/**
	 * 空判断
	 * @return sizeが0の場合 true を返す。
	 */
	boolean isEmpty();

	/**
	 * 他のモデルの内容を全て追加する。
	 *
	 * @param value モデル
	 */
	void addAll(UniModel value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Boolean value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Boolean... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, String value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, String... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Byte value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Byte... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Short value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Short... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Integer value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Integer... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Long value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Long... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Float value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Float... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Double value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Double... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, BigInteger value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, BigInteger... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, BigDecimal value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, BigDecimal... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Number value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, Number... value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, java.util.Date value);

	/**
	 * キーに対応した値を追加する。
	 *
	 * @param key キー
	 * @param value 値
	 * @throws ClassCastException クラスキャスト例外
	 */
	void addValue(String key, java.util.Date... value);
}
