/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2;

import java.util.Map;

/**
 * Simple fluent chain EDSL to simplify {@link Chain} instances invocation.
 *
 * @version $Id$
 * @since 2.0
 */
public final class Chains {

    /**
     * Private constructor, this class cannot be instantiated directly.
     */
    private Chains() {
        throw new AssertionError();
    }

    /**
     * Defines the target chain has to be invoked.
     *
     * @param <K> Context key type
     * @param <V> Context value type
     * @param <C> Type of the context associated with this command
     * @param <U> Type of the {@link Chain} to execute
     * @param chain the chain instance reference to execute
     * @return next chain builder
     */
    public static <K, V, C extends Map<K, V>, U extends Chain<K, V, C>>
            ToExecutorCommandSetter<K, V, C> on(final U chain) {
        if (chain == null) {
            throw new IllegalArgumentException("Null Chain can not be executed");
        }
        return new DefaultCommandSetter<>(chain);
    }

    /**
     * Defines the target {@link Catalog} has to be invoked.
     *
     * @param <K> Context key type
     * @param <V> Context value type
     * @param <C> Type of the context associated with this command
     * @param <S> Type of the {@link Catalog} to execute
     * @param catalog the catalog instance reference to be setup
     * @return next chain builder
     */
    public static <K, V, C extends Map<K, V>, S extends Catalog<K, V, C>>
            NamedCommandSetter<K, V, C> on(final S catalog) {
        if (catalog == null) {
            throw new IllegalArgumentException("Null Catalog can not be setup");
        }
        return new DefaultNamedCommandSetter<>(catalog);
    }

    /**
     * DefaultCommandSetter
     *
     * @param <K> Key
     * @param <V> Value
     * @param <C> Context
     */
    private static final class DefaultCommandSetter<K, V, C extends Map<K, V>>
            implements ToExecutorCommandSetter<K, V, C> {

        /** Chain */
        private final Chain<K, V, C> chain;

        /**
         * @param chn Chain
         */
        DefaultCommandSetter(final Chain<K, V, C> chn) {
            this.chain = chn;
        }

        /**
         * @see org.apache.commons.chain2.CommandSetter#add(org.apache.commons.chain2.Command)
         */
        @Override
        public <T extends Command<K, V, C>> ChainExecutor<K, V, C> add(final T command) {
            if (command == null) {
                throw new IllegalArgumentException("Chain does not accept null Command instances");
            }
            this.chain.addCommand(command);
            return new DefaultChainExecutor<>(this.chain);
        }
    }

    /**
     * DefaultChainExecutor
     *
     * @param <K> Key
     * @param <V> Value
     * @param <C> Context
     */
    private static final class DefaultChainExecutor<K, V, C extends Map<K, V>>
            implements ChainExecutor<K, V, C> {

        /** Chain */
        private final Chain<K, V, C> chain;

        /**
         * @param chn Chain
         */
        DefaultChainExecutor(final Chain<K, V, C> chn) {
            this.chain = chn;
        }

        /**
         * @see org.apache.commons.chain2.CommandSetter#add(org.apache.commons.chain2.Command)
         */
        @Override
        public <T extends Command<K, V, C>> ChainExecutor<K, V, C> add(final T command) {
            if (command == null) {
                throw new IllegalArgumentException("Chain does not accept null Command instances");
            }
            this.chain.addCommand(command);
            return this;
        }

        /**
         * @see org.apache.commons.chain2.ChainExecutor#execute(java.util.Map)
         */
        @Override
        public Processing execute(final C context) {
            if (context == null) {
                throw new IllegalArgumentException("Chain cannot be applied to a null context.");
            }
            return this.chain.execute(context);
        }
    }

    /**
     * DefaultNamedCommandSetter
     *
     * @param <K> Key
     * @param <V> Value
     * @param <C> Context
     */
    private static final class DefaultNamedCommandSetter<K, V, C extends Map<K, V>>
            implements NamedCommandSetter<K, V, C> {

        /** Catalog */
        private final Catalog<K, V, C> catalog;

        /**
         * @param ctlg Catalog
         */
        DefaultNamedCommandSetter(final Catalog<K, V, C> ctlg) {
            this.catalog = ctlg;
        }

        /**
         * @see org.apache.commons.chain2.CommandSetter#add(org.apache.commons.chain2.Command)
         */
        @Override
        public <T extends Command<K, V, C>> NameSetter<K, V, C> add(final T command) {
            if (command == null) {
                throw new IllegalArgumentException(
                        "Catalog does not accept null Command instances");
            }
            return name -> {
                if (name == null) {
                    throw new IllegalArgumentException(
                       String.format("Command <%s> cannot be identified by a null name", command));
                }
                this.catalog.addCommand(name, command);
                return new DefaultNamedCommandSetter<>(this.catalog);
            };
        }
    }
}
