/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.base;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import org.apache.commons.chain2.Context;
import org.apache.commons.chain2.Processing;
import org.apache.commons.chain2.impl.ContextBase;
import org.junit.Test;

/**
 * JUnitTest case for class: org.apache.commons.chain2.base.DispatchCommand
 */
public class DispatchCommandTestCase {

    /**
     * testMethodDispatch
     */
    @Test
    public void testMethodDispatch() {
        TestCommand test = new TestCommand();

        test.setMethod("testMethod");
        Context<String, Object> context = new ContextBase();
        assertNull(context.get("foo"));
        Processing result = test.execute(context);
        assertEquals(Processing.FINISHED, result);
        assertNotNull(context.get("foo"));
        assertEquals("foo", context.get("foo"));


    }

    /**
     * testMethodKeyDispatch
     */
    @Test
    public void testMethodKeyDispatch() {
        TestCommand test = new TestCommand();

        test.setMethodKey("foo");
        Context<String, Object> context = new ContextBase();
        context.put("foo", "testMethodKey");
        assertNull(context.get("bar"));
        Processing result = test.execute(context);
        assertEquals(Processing.CONTINUE, result);
        assertNotNull(context.get("bar"));
        assertEquals("bar", context.get("bar"));


    }

    /**
     * testAlternateContext
     */
    @Test
    public void testAlternateContext() {
        TestAlternateContextCommand test = new TestAlternateContextCommand();

        test.setMethod("foo");
        Context<String, Object> context = new ContextBase();
        assertNull(context.get("elephant"));
        Processing result = test.execute(context);
        assertEquals(Processing.FINISHED, result);
        assertNotNull(context.get("elephant"));
        assertEquals("elephant", context.get("elephant"));


    }

    /**
     * TestCommand
     *
     */
    static class TestCommand extends DispatchCommand<String, Object, Context<String, Object>> {

        /**
         * @param context Context
         * @return Processing
         */
        public Processing testMethod(final Context<String, Object> context) {
            context.put("foo", "foo");
            return Processing.FINISHED;
        }

        /**
         * @param context Context
         * @return Processing
         */
        public Processing testMethodKey(final Context<String, Object> context) {

            context.put("bar", "bar");
            return Processing.CONTINUE;
        }
    }

    /**
     * Command which uses alternate method signature.
     * @version 0.2-dev
     */
    static class TestAlternateContextCommand
            extends DispatchCommand<String, Object, Context<String, Object>> {

        /**
         * @see org.apache.commons.chain2.base.DispatchCommand#getSignature()
         */
        @Override
        protected Class<?>[] getSignature() {
            return new Class[] {TestAlternateContext.class};
        }

        /**
         * @see org.apache.commons.chain2.base.DispatchCommand
         * #getArguments(org.apache.commons.chain2.Context)
         */
        @Override
        protected Object[] getArguments(final Context<String, Object> context) {
            return new Object[] {new TestAlternateContext(context)};
        }

        /**
         * @param context TestAlternateContext
         * @return Processing
         */
        public Processing foo(final TestAlternateContext context) {
            context.put("elephant", "elephant");
            return Processing.FINISHED;
        }

    }

    /**
     * TestAlternateContext
     *
     */
    static class TestAlternateContext extends java.util.HashMap<String, Object>
            implements Context<String, Object> {
        /** serialVersionUID */
        private static final long serialVersionUID = 20120724L;

        /** wrappedContext */
        private Context<String, Object> wrappedContext = null;

        /**
         * Constructor
         * @param context Context
         */
        TestAlternateContext(final Context<String, Object> context) {
            this.wrappedContext = context;
        }

        /**
         * @see java.util.HashMap#get(java.lang.Object)
         */
        @Override
        public Object get(final Object o) {
            return this.wrappedContext.get(String.class.cast(o));
        }

        /**
         * @see org.apache.commons.chain2.Context#retrieve(java.lang.Object)
         */
        @Override
        public <T extends Object> T retrieve(final String key) {
            return this.wrappedContext.<T>retrieve(key);
        }

        /**
         * @see java.util.HashMap#put(java.lang.Object, java.lang.Object)
         */
        @Override
        public Object put(final String key, final Object value) {
            return this.wrappedContext.put(key, value);
        }
    }
}
