/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for servlet cookies</p>
 *
 * @version $Id$
 * @since 1.1
 */
final class ServletCookieMap implements Map<String, Cookie>, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = 2368842850955992216L;

    /** HttpServletRequest */
    private transient HttpServletRequest request = null;

    /**
     * Constructor
     *
     * @param req HttpServletRequest
     */
    ServletCookieMap(final HttpServletRequest req) {
        this.request = req;
    }

    /**
     * @see java.util.Map#clear()
     */
    @Override
    public void clear() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    @Override
    public boolean containsKey(final Object key) {
        return get(String.class.cast(key)) != null;
    }

    /**
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    @Override
    public boolean containsValue(final Object value) {
        Cookie[] cookies = this.request.getCookies();
        if (cookies != null) {
            for (final Cookie c : cookies) {
                if (c.equals(value)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @see java.util.Map#entrySet()
     */
    @Override
    public Set<Entry<String, Cookie>> entrySet() {
        Set<Entry<String, Cookie>> set = new HashSet<>();
        Cookie[] cookies = this.request.getCookies();
        if (cookies != null) {
            for (final Cookie c : cookies) {
                set.add(new MapEntry<>(c.getName(), c, false));
            }
        }
        return set;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object o) {
        if (o != this) {
            if (!ServletCookieMap.class.isInstance(o)) {
                return false;
            }
            ServletCookieMap map = (ServletCookieMap) o;
            return this.request.equals(map.request);
        }
        return true;
    }

    /**
     * @see java.util.Map#get(java.lang.Object)
     */
    @Override
    public Cookie get(final Object key) {
        Cookie[] cookies = this.request.getCookies();
        if (cookies != null) {
            for (final Cookie c : cookies) {
                if (c.getName().equals(toKey(key))) {
                    return c;
                }
            }
        }
        return null;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return this.request.hashCode();
    }

    /**
     * @see java.util.Map#isEmpty()
     */
    @Override
    public boolean isEmpty() {
        return size() < 1;
    }

    /**
     * @see java.util.Map#keySet()
     */
    @Override
    public Set<String> keySet() {
        Set<String> set = new HashSet<>();
        Cookie[] cookies = this.request.getCookies();
        if (cookies != null) {
            for (final Cookie c : cookies) {
                set.add(c.getName());
            }
        }
        return set;
    }

    /**
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    @Override
    public Cookie put(final String key, final Cookie value) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#putAll(java.util.Map)
     */
    @Override
    public void putAll(final Map<? extends String, ? extends Cookie> map) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#remove(java.lang.Object)
     */
    @Override
    public Cookie remove(final Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#size()
     */
    @Override
    public int size() {
        Cookie[] cookies = this.request.getCookies();
        return cookies == null ? 0 : cookies.length;
    }

    /**
     * @see java.util.Map#values()
     */
    @Override
    public Collection<Cookie> values() {
        List<Cookie> list = new ArrayList<>();
        Cookie[] cookies = this.request.getCookies();
        if (cookies != null) {
            list.addAll(Arrays.asList(cookies));
        }
        return list;
    }

    /**
     * to Key String
     * @param key Key
     * @return Key String
     */
    private String toKey(final Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        return key.toString();
    }

}
