/*
 * $Id: ComposableRequestProcessor.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.chain;

import java.lang.reflect.Constructor;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.UnavailableException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.beanutils.ConstructorUtils;
import org.apache.commons.chain2.Catalog;
import org.apache.commons.chain2.CatalogFactory;
import org.apache.commons.chain2.Command;
import org.apache.commons.chain2.Context;
import org.apache.commons.chain2.impl.CatalogFactoryBase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionServlet;
import org.apache.struts.action.RequestProcessor;
import org.apache.struts.chain.contexts.ActionContext;
import org.apache.struts.chain.contexts.ServletActionContext;
import org.apache.struts.config.ControllerConfig;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.upload.MultipartRequestWrapper;
import org.apache.struts.util.ResponseUtils;


/**
 * <p> ComposableRequestProcessor uses the Chain Of Resposibility design
 * pattern (as implemented by the commons-chain package in Jakarta Commons) to
 * support external configuration of command chains to be used.  It is
 * configured via the following context initialization parameters: </p>
 *
 * <ul>
 *
 * <li>[org.apache.struts.chain.CATALOG_NAME] - Name of the Catalog in which
 * we will look up the Command to be executed for each request.  If not
 * specified, the default value is struts. </li>
 *
 * <li> org.apache.struts.chain.COMMAND_NAME - Name of the Command which we
 * will execute for each request, to be looked up in the specified Catalog.
 * If not specified, the default value is servlet-standard. </li>
 *
 * </ul>
 *
 * @version $Rev: 471754 $ $Date: 2005-11-12 13:01:44 -0500 (Sat, 12 Nov 2005)
 *          $
 * @since Struts 1.1
 */
public class ComposableRequestProcessor implements RequestProcessor {

    /**
     * <p> Token for ActionContext clazss so that it can be stored in the
     * ControllerConfig. </p>
     */
    public static final String ACTION_CONTEXT_CLASS = "ACTION_CONTEXT_CLASS";

    /**
     * <p> Cache for constructor discovered by setActionContextClass method.
     * </p>
     */
    private static final Class<?>[] SERVLET_ACTION_CONTEXT_CTOR_SIGNATURE =
        new Class[] {
            ServletContext.class, HttpServletRequest.class,
            HttpServletResponse.class
        };

    /**
     * <p>The <code>Log</code> instance for this class.</p>
     */
    private static final Log LOG = LogFactory.getLog(ComposableRequestProcessor.class);

    // ------------------------------------------------------ Instance Variables

    /**
     * <p>The {@link CatalogFactory} from which catalog containing the the
     * base request-processing {@link Command} will be retrieved.</p>
     */
    private CatalogFactory<String, Object, Context<String, Object>> catalogFactory = null;

    /**
     * <p>The {@link Catalog} containing all of the available command chains
     * for this module.
     */
    private Catalog<String, Object, Context<String, Object>> catalog = null;

    /**
     * <p>The {@link Command} to be executed for each request.</p>
     */
    private Command<String, Object, Context<String, Object>> command = null;

    /**
     * <p> ActionContext class as cached by createActionContextInstance
     * method. </p>
     */
    private Class<? extends ActionContext<String, Object>> actionContextClass;

    /**
     * <p> ActionContext constructor as cached by createActionContextInstance
     * method. </p>
     */
    private Constructor<? extends ActionContext<String, Object>> servletActionContextConstructor;

    /**
     * <p>The <code>ModuleConfiguration</code> with which we are
     * associated.</p>
     */
    private ModuleConfig moduleConfig = null;

    /**
     * <p>The servlet with which we are associated.</p>
     */
    private ActionServlet servlet = null;

    // ---------------------------------------------------------- Public Methods

    /**
     * <p>Clean up in preparation for a shutdown of this application.</p>
     */
    @Override
    public void destroy() {
        this.servlet = null;
        this.catalogFactory = null;
        this.catalog = null;
        this.command = null;
        this.actionContextClass = null;
        this.servletActionContextConstructor = null;
    }

    /**
     * <p>Initialize this request processor instance.</p>
     *
     * @param actionServlet      The ActionServlet we are associated with
     * @param config The ModuleConfig we are associated with.
     * @throws ServletException If an error occurs during initialization
     */
    @Override
    public void init(final ActionServlet actionServlet, final ModuleConfig config)
            throws ServletException {
        LOG.info("Initializing composable request processor for module prefix '"
            + config.getPrefix() + "'");

        this.servlet = actionServlet;
        this.moduleConfig = config;
        initCatalogFactory(actionServlet, config);

        ControllerConfig controllerConfig = config.getControllerConfig();

        String catalogName = controllerConfig.getCatalog();

        this.catalog = this.catalogFactory.getCatalog(catalogName);

        if (this.catalog == null) {
            throw new ServletException("Cannot find catalog '" + catalogName + "'");
        }

        String commandName = controllerConfig.getCommand();

        this.command = this.catalog.getCommand(commandName);

        if (this.command == null) {
            throw new ServletException("Cannot find command '" + commandName + "'");
        }

        this.setActionContextClassName(controllerConfig.getProperty(ACTION_CONTEXT_CLASS));
    }

    /**
     * <p> Set and cache ActionContext class. </p><p> If there is a custom
     * class provided and if it uses our "preferred" constructor, cache a
     * reference to that constructor rather than looking it up every time.
     * </p>
     *
     * @param contextClass The ActionContext class to process
     */
    private void setActionContextClass(
            final Class<? extends ActionContext<String, Object>> contextClass) {
        this.actionContextClass = contextClass;

        if (contextClass != null) {
            this.servletActionContextConstructor = ConstructorUtils.getAccessibleConstructor(
                    contextClass, SERVLET_ACTION_CONTEXT_CTOR_SIGNATURE);
        } else {
            this.servletActionContextConstructor = null;
        }
    }

    /**
     * <p>Make sure that the specified <code>className</code> identfies a
     * class which can be found and which implements the
     * <code>ActionContext</code> interface.</p>
     *
     * @param className Fully qualified name of
     * @throws ServletException     If an error occurs during initialization
     * @throws UnavailableException if class does not implement ActionContext
     *                              or is not found
     */
    private void setActionContextClassName(final String className) throws ServletException {
        if ((className != null) && (!className.trim().isEmpty())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("setActionContextClassName: requested context class: " + className);
            }

            try {
                Class<? extends ActionContext<String, Object>> contextClass =
                        ResponseUtils.applicationClass(className);

                if (!ActionContext.class.isAssignableFrom(contextClass)) {
                    throw new UnavailableException("ActionContextClass " + "["
                        + className + "]"
                        + " must implement ActionContext interface.");
                }

                this.setActionContextClass(contextClass);

            } catch (final ReflectiveOperationException e) {
                throw new UnavailableException("ActionContextClass "
                    + className + " not found.: " + e.getMessage());
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("setActionContextClassName: no className specified");
            }

            this.setActionContextClass(null);
        }
    }

    /**
     * <p> Establish the CatalogFactory which will be used to look up the
     * catalog which has the request processing command. </p><p> The base
     * implementation simply calls CatalogFactory.getInstance(), unless the
     * catalogFactory property of this object has already been set, in which
     * case it is not changed. </p>
     *
     * @param actionServlet      The ActionServlet we are processing
     * @param config The ModuleConfig we are processing
     */
    protected void initCatalogFactory(final ActionServlet actionServlet,
            final ModuleConfig config) {
        if (this.catalogFactory != null) {
            return;
        }

        this.catalogFactory = CatalogFactoryBase.getInstance();
    }

    /**
     * <p>Process an <code>HttpServletRequest</code> and create the
     * corresponding <code>HttpServletResponse</code>.</p>
     *
     * @param request  The servlet request we are processing
     * @param response The servlet response we are creating
     * @throws ServletException if a processing exception occurs
     */
    @Override
    public void process(final HttpServletRequest request, final HttpServletResponse response)
            throws ServletException {
        // Wrap the request in the case of a multipart request
        HttpServletRequest req = processMultipart(request);

        // Create and populate a Context for this request
        ActionContext<String, Object> context = contextInstance(req, response);

        // Create and execute the command.
        if (LOG.isDebugEnabled()) {
            LOG.debug("Using processing chain for this request");
        }

        this.command.execute(context);

        // Release the context.
        context.release();
    }

    /**
     * <p>Provide the initialized <code>ActionContext</code> instance which
     * will be used by this request. Internally, this simply calls
     * <code>createActionContextInstance</code> followed by
     * <code>initializeActionContext</code>.</p>
     *
     * @param request  The servlet request we are processing
     * @param response The servlet response we are creating
     * @return Initiliazed ActionContext
     * @throws ServletException if a processing exception occurs
     */
    protected ActionContext<String, Object> contextInstance(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException {
        ActionContext<String, Object> context =
            createActionContextInstance(this.servlet.getServletContext(), request, response);

        initializeActionContext(context);

        return context;
    }

    /**
     * <p>Create a new instance of <code>ActionContext</code> according to
     * configuration.  If no alternative was specified at initialization, a
     * new instance <code>ServletActionContext</code> is returned.  If an
     * alternative was specified using the <code>ACTION_CONTEXT_CLASS</code>
     * property, then that value is treated as a classname, and an instance of
     * that class is created.  If that class implements the same constructor
     * that <code>ServletActionContext</code> does, then that constructor will
     * be used: <code>ServletContext, HttpServletRequest,
     * HttpServletResponse</code>; otherwise, it is assumed that the class has
     * a no-arguments constructor.  If these constraints do not suit you,
     * simply override this method in a subclass.</p>
     *
     * @param servletContext The servlet context we are processing
     * @param request        The servlet request we are processing
     * @param response       The servlet response we are creating
     * @return New instance of ActionContext
     * @throws ServletException if a processing exception occurs
     */
    protected ActionContext<String, Object> createActionContextInstance(
            final ServletContext servletContext, final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException {
        if (this.actionContextClass == null) {
            return new ServletActionContext(servletContext, request, response);
        }

        try {
            if (this.servletActionContextConstructor == null) {
                return this.actionContextClass.newInstance();
            }

            return this.servletActionContextConstructor.newInstance(
                    servletContext, request, response);

        } catch (final ReflectiveOperationException e) {
            throw new ServletException(
                "Error creating ActionContext instance of type "
                + this.actionContextClass, e);
        }
    }

    /**
     * <p>Set common properties on the given <code>ActionContext</code>
     * instance so that commands in the chain can count on their presence.
     * Note that while this method does not require that its argument be an
     * instance of <code>ServletActionContext</code>, at this time many common
     * Struts commands will be expecting to receive an <code>ActionContext</code>
     * which is also a <code>ServletActionContext</code>.</p>
     *
     * @param context The ActionContext we are processing
     */
    protected void initializeActionContext(final ActionContext<String, Object> context) {
        if (context instanceof ServletActionContext) {
            ((ServletActionContext) context).setActionServlet(this.servlet);
        }

        context.setModuleConfig(this.moduleConfig);
    }

    /**
     * <p>If this is a multipart request, wrap it with a special wrapper.
     * Otherwise, return the request unchanged.</p>
     *
     * @param request The HttpServletRequest we are processing
     * @return Original or wrapped request as appropriate
     */
    protected HttpServletRequest processMultipart(final HttpServletRequest request) {
        if (!"POST".equalsIgnoreCase(request.getMethod())) {
            return request;
        }

        String contentType = request.getContentType();
        if ((contentType != null) && contentType.startsWith("multipart/form-data")) {
            return new MultipartRequestWrapper(request);
        }
        return request;
    }

    /**
     * <p>Set the <code>CatalogFactory</code> instance which should be used to
     * find the request-processing command.  In the base implementation, if
     * this value is not already set, then it will be initialized when {@link
     * #initCatalogFactory} is called. </p>
     *
     * @param factory Our CatalogFactory instance
     */
    public void setCatalogFactory(
            final CatalogFactory<String, Object, Context<String, Object>> factory) {
        this.catalogFactory = factory;
    }
}
