/*
 * $Id: PerformForward.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.chain.commands.servlet;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionServlet;
import org.apache.struts.chain.commands.AbstractPerformForward;
import org.apache.struts.chain.contexts.ActionContext;
import org.apache.struts.chain.contexts.ServletActionContext;
import org.apache.struts.config.ForwardConfig;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.util.MessageResources;
import org.apache.struts.util.RequestUtils;
import org.apache.struts.util.ModuleUtils;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * <p>Perform forwarding or redirection based on the specified
 * <code>ForwardConfig</code> (if any).</p>
 *
 * @version $Rev: 471754 $ $Date: 2006-11-06 08:55:09 -0600 (Mon, 06 Nov 2006) $
 */
public class PerformForward extends AbstractPerformForward {

    /** LOGGER */
    private static final Log LOG = LogFactory.getLog(PerformForward.class);

    // ------------------------------------------------------- Protected Methods

    /**
     * <p>Perform the appropriate processing on the specified
     * <code>ForwardConfig</code>.</p>
     *
     * @param context       The context for this request
     * @param forwardConfig The forward to be performed
     */
    @Override
    protected void perform(final ActionContext<String, Object> context,
            final ForwardConfig forwardConfig) {

        ServletActionContext sacontext = (ServletActionContext) context;
        String uri = forwardConfig.getPath();
        if (uri == null) {
            ActionServlet servlet = sacontext.getActionServlet();
            MessageResources resources = servlet.getInternal();
            throw new IllegalArgumentException(resources.getMessage("forwardPathNull"));
        }

        HttpServletRequest request = sacontext.getRequest();
        ServletContext servletContext = sacontext.getContext();
        HttpServletResponse response = sacontext.getResponse();
        ForwardConfig fc = forwardConfig;
        // If the forward can be unaliased into an action, then use the path of the action
        String actionIdPath = RequestUtils.actionIdURL(
                forwardConfig, sacontext.getRequest(), sacontext.getActionServlet());
        if (actionIdPath != null) {
            uri = actionIdPath;
            ForwardConfig actionIdForwardConfig = new ForwardConfig(forwardConfig);
            actionIdForwardConfig.setPath(actionIdPath);
            fc = actionIdForwardConfig;
        }

        if (uri.startsWith("/")) {
            uri = resolveModuleRelativePath(fc, servletContext, request);
        }

        try {
            if (response.isCommitted() && !fc.getRedirect()) {
                handleAsInclude(uri, servletContext, request, response);
            } else if (fc.getRedirect()) {
                handleAsRedirect(uri, request, response);
            } else {
                handleAsForward(uri, servletContext, request, response);
            }
        } catch (final IOException e) {
            LOG.info(e.getMessage());
        } catch (final ServletException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * resolve path
     *
     * @param forwardConfig ForwardConfig
     * @param servletContext ServletContext
     * @param request HttpServletRequest
     * @return url
     */
    private String resolveModuleRelativePath(final ForwardConfig forwardConfig,
            final ServletContext servletContext, final HttpServletRequest request) {
        String prefix = forwardConfig.getModule();
        ModuleConfig moduleConfig = ModuleUtils.getInstance().getModuleConfig(
                prefix, request, servletContext);
        return RequestUtils.forwardURL(request, forwardConfig, moduleConfig);
    }

    /**
     * forward
     *
     * @param uri URI
     * @param servletContext ServletContext
     * @param request HttpServletRequest
     * @param response HttpServletResponse
     * @throws ServletException ServletException
     * @throws IOException IOException
     */
    private void handleAsForward(final String uri, final ServletContext servletContext,
            final HttpServletRequest request, final HttpServletResponse response)
                    throws ServletException, IOException {
        RequestDispatcher rd = servletContext.getRequestDispatcher(uri);

        if (LOG.isDebugEnabled()) {
            LOG.debug("Forwarding to " + uri);
        }

        rd.forward(request, response);
    }

    /**
     * redirect
     *
     * @param uri URI
     * @param request HttpServletRequest
     * @param response HttpServletResponse
     * @throws IOException IOException
     */
    private void handleAsRedirect(final String uri, final HttpServletRequest request,
            final HttpServletResponse response) throws IOException {
        String url = uri;
        if (url.startsWith("/")) {
            url = request.getContextPath() + url;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Redirecting to " + url);
        }

        response.sendRedirect(response.encodeRedirectURL(url));
    }

    /**
     * include
     *
     * @param uri URI
     * @param servletContext ServletContext
     * @param request HttpServletRequest
     * @param response HttpServletResponse
     * @throws IOException IOException
     * @throws ServletException ServletException
     */
    private void handleAsInclude(final String uri, final ServletContext servletContext,
            final HttpServletRequest request, final HttpServletResponse response)
                    throws IOException, ServletException {
        RequestDispatcher rd = servletContext.getRequestDispatcher(uri);
        if (rd == null) {
            response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                "Error getting RequestDispatcher for " + uri);
            return;
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("Including " + uri);
        }

        rd.include(request, response);
    }
}
