/*
 * $Id: TestActionConfigMatcher.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.config;

import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.junit.Assert;
import org.junit.Test;


/**
 * <p>Unit tests for <code>org.apache.struts.util.ActionConfigMatcher</code>.</p>
 *
 * @version $Rev: 471754 $ $Date: 2005-10-27 23:25:01 -0400 (Thu, 27 Oct 2005)
 *          $
 */
public class TestActionConfigMatcher {

    // ----------------------------------------------------------------- Basics

    // ------------------------------------------------------- Individual Tests

    // ---------------------------------------------------------- match()

    /**
     * testNoMatch
     */
    @Test
    public void testNoMatch() {
        ActionConfig[] configs = new ActionConfig[1];
        ActionConfig mapping = buildActionConfig("/foo");

        configs[0] = mapping;

        ActionConfigMatcher matcher = new ActionConfigMatcher(configs);

        Assert.assertNull("ActionConfig shouldn't be matched", matcher.match("/test"));
    }

    /**
     * testNoWildcardMatch
     */
    @Test
    public void testNoWildcardMatch() {
        ActionConfig[] configs = new ActionConfig[1];
        ActionConfig mapping = buildActionConfig("/fooBar");

        configs[0] = mapping;

        ActionConfigMatcher matcher = new ActionConfigMatcher(configs);

        Assert.assertNull("ActionConfig shouldn't be matched", matcher.match("/fooBar"));
    }

    /**
     * testShouldMatch
     */
    @Test
    public void testShouldMatch() {
        ActionConfig[] configs = new ActionConfig[1];
        ActionConfig mapping = buildActionConfig("/foo*");

        configs[0] = mapping;

        ActionConfigMatcher matcher = new ActionConfigMatcher(configs);

        ActionConfig matched = matcher.match("/fooBar");

        Assert.assertNotNull("ActionConfig should be matched", matched);
        Assert.assertTrue("ActionConfig should have two action forward",
            matched.findForwardConfigs().length == 2);
        Assert.assertTrue("ActionConfig should have two exception forward",
            matched.findExceptionConfigs().length == 2);
        Assert.assertTrue("ActionConfig should have properties",
            matched.getProperties().size() == 2);
    }

    /**
     * testCheckSubstitutionsMatch
     */
    @Test
    public void testCheckSubstitutionsMatch() {
        ActionConfig[] configs = new ActionConfig[1];
        ActionConfig mapping = buildActionConfig("/foo*");

        configs[0] = mapping;

        ActionConfigMatcher matcher = new ActionConfigMatcher(configs);
        ActionConfig m = matcher.match("/fooBar");

        Assert.assertTrue("Name hasn't been replaced", "name,Bar".equals(m.getName()));
        Assert.assertTrue("Path hasn't been replaced", "/fooBar".equals(m.getPath()));
        Assert.assertTrue("Scope isn't correct", "request".equals(m.getScope()));
        Assert.assertTrue("Unknown isn't correct", !m.getUnknown());
        Assert.assertTrue("Validate isn't correct", m.getValidate());

        Assert.assertTrue("Prefix hasn't been replaced",
            "foo,Bar".equals(m.getPrefix()));
        Assert.assertTrue("Suffix hasn't been replaced",
            "bar,Bar".equals(m.getSuffix()));
        Assert.assertTrue("Type hasn't been replaced",
            "foo.bar.BarAction".equals(m.getType()));
        Assert.assertTrue("Roles hasn't been replaced",
            "public,Bar".equals(m.getRoles()));
        Assert.assertTrue("Parameter hasn't been replaced",
            "param,Bar".equals(m.getParameter()));
        Assert.assertTrue("Attribute hasn't been replaced",
            "attrib,Bar".equals(m.getAttribute()));
        Assert.assertTrue("Forward hasn't been replaced",
            "fwd,Bar".equals(m.getForward()));
        Assert.assertTrue("Include hasn't been replaced",
            "include,Bar".equals(m.getInclude()));
        Assert.assertTrue("Input hasn't been replaced",
            "input,Bar".equals(m.getInput()));

        Assert.assertTrue("ActionConfig property not replaced",
            "testBar".equals(m.getProperty("testprop2")));

        ForwardConfig[] fConfigs = m.findForwardConfigs();
        boolean found = false;

        for (int x = 0; x < fConfigs.length; x++) {
            ForwardConfig cfg = fConfigs[x];

            if ("name".equals(cfg.getName())) {
                found = true;
                Assert.assertTrue("Path hasn't been replaced",
                    "path,Bar".equals(cfg.getPath()));
                Assert.assertTrue("Property foo hasn't been replaced",
                    "bar,Bar".equals(cfg.getProperty("foo")));
                Assert.assertTrue("Module hasn't been replaced",
                    "modBar".equals(cfg.getModule()));
            }
        }

        Assert.assertTrue("The forward config 'name' cannot be found", found);
    }

    /**
     * testCheckMultipleSubstitutions
     */
    public void testCheckMultipleSubstitutions() {
        ActionMapping[] mapping = new ActionMapping[1];

        mapping[0] = new ActionMapping();
        mapping[0].setPath("/foo*");
        mapping[0].setName("name,{1}-{1}");

        ActionConfigMatcher matcher = new ActionConfigMatcher(mapping);
        ActionConfig m = matcher.match("/fooBar");

        Assert.assertTrue("Name hasn't been replaced correctly: " + m.getName(),
            "name,Bar-Bar".equals(m.getName()));
    }

    /**
     * build ActionConfig
     * @param path String
     * @return ActionConfig
     */
    private ActionConfig buildActionConfig(final String path) {
        ActionMapping mapping = new ActionMapping();

        mapping.setName("name,{1}");
        mapping.setPath(path);
        mapping.setScope("request");
        mapping.setUnknown(false);
        mapping.setValidate(true);

        mapping.setPrefix("foo,{1}");
        mapping.setSuffix("bar,{1}");

        mapping.setType("foo.bar.{1}Action");
        mapping.setRoles("public,{1}");
        mapping.setParameter("param,{1}");
        mapping.setAttribute("attrib,{1}");
        mapping.setForward("fwd,{1}");
        mapping.setInclude("include,{1}");
        mapping.setInput("input,{1}");

        setActionConfig(mapping);

        setExceptionConfig(mapping);


        mapping.freeze();

        return mapping;
    }

    /**
     * set ActionConfig
     * @param mapping ActionMapping
     */
    private void setActionConfig(final ActionMapping mapping) {
        ForwardConfig cfg = new ActionForward();

        cfg.setName("name");
        cfg.setPath("path,{1}");
        cfg.setModule("mod{1}");
        cfg.setProperty("foo", "bar,{1}");
        mapping.addForwardConfig(cfg);

        cfg = new ActionForward();
        cfg.setName("name2");
        cfg.setPath("path2");
        cfg.setModule("mod{1}");
        mapping.addForwardConfig(cfg);
    }

    /**
     * set ExceptionConfig
     * @param mapping ActionMapping
     */
    private void setExceptionConfig(final ActionMapping mapping) {
        ExceptionConfig excfg = new ExceptionConfig();

        excfg.setKey("foo");
        excfg.setType("foo.Bar");
        excfg.setPath("path");
        mapping.addExceptionConfig(excfg);

        excfg = new ExceptionConfig();
        excfg.setKey("foo2");
        excfg.setType("foo.Bar2");
        excfg.setPath("path2");
        mapping.addExceptionConfig(excfg);

        mapping.setProperty("testprop", "testval");
        mapping.setProperty("testprop2", "test{1}");
    }
}
