/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.dbcp.impl;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.StatementEventListener;
import javax.sql.XAConnection;
import javax.transaction.xa.XAResource;

/**
 * {@link XAConnection}の実装です。
 *
 * @author higa
 *
 */
public class XAConnectionImpl implements XAConnection {

    /** コネクション */
    private final Connection connection;
    /** XAリソース */
    private final XAResource xaResource;
    /** リスナーリスト */
    private final List<ConnectionEventListener> listeners = new ArrayList<>();

    /**
     * {@link XAConnectionImpl}を作成します。
     *
     * @param conn コネクション
     */
    public XAConnectionImpl(final Connection conn) {
        this.connection = conn;
        this.xaResource = new DbXAResourceImpl(conn);
    }

    /**
     * @see javax.sql.XAConnection#getXAResource()
     */
    @Override
    public XAResource getXAResource() {
        return this.xaResource;
    }

    /**
     * @see javax.sql.PooledConnection#getConnection()
     */
    @Override
    public Connection getConnection() throws SQLException {
        return this.connection;
    }

    /**
     * @see javax.sql.PooledConnection#close()
     */
    @Override
    public void close() throws SQLException {
        if (this.connection != null && !this.connection.isClosed()) {
            this.connection.close();
            final ConnectionEvent ce = new ConnectionEvent(this);
            for (final ConnectionEventListener cel : this.listeners) {
                cel.connectionClosed(ce);
            }
        }
    }

    /**
     * @see javax.sql.PooledConnection#addConnectionEventListener(javax.sql.ConnectionEventListener)
     */
    @Override
    public void addConnectionEventListener(final ConnectionEventListener listener) {
        synchronized (this.listeners) {
            this.listeners.add(listener);
        }
    }

    /**
     * @see javax.sql.PooledConnection
     * #removeConnectionEventListener(javax.sql.ConnectionEventListener)
     */
    @Override
    public void removeConnectionEventListener(final ConnectionEventListener listener) {
        synchronized (this.listeners) {
            this.listeners.remove(listener);
        }
    }

    /**
     * @see javax.sql.PooledConnection#addStatementEventListener(javax.sql.StatementEventListener)
     */
    @Override
    public void addStatementEventListener(final StatementEventListener listener) {
        return;
    }

    /**
     * @see javax.sql.PooledConnection
     * #removeStatementEventListener(javax.sql.StatementEventListener)
     */
    @Override
    public void removeStatementEventListener(final StatementEventListener listener) {
        return;
    }
}
