/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.jta;

import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

/**
 * XAResourceWrapper
 *
 */
class XAResourceWrapper {

    /** XAResource */
    private final XAResource xaResource;
    /** Xid */
    private final Xid xid;
    /** commitTarget */
    private final boolean commitTarget;

    /** voteOk */
    private boolean voteOk = true;

    /**
     * Constructor
     * @param xar XAResource
     * @param id Xid
     * @param commit commitTarget
     */
    XAResourceWrapper(final XAResource xar, final Xid id, final boolean commit) {
        this.xaResource = xar;
        this.xid = id;
        this.commitTarget = commit;
    }

    /**
     * @return XAResource
     */
    XAResource getXAResource() {
        return this.xaResource;
    }

    /**
     * @return Xid
     */
    Xid getXid() {
        return this.xid;
    }

    /**
     * @return commitTarget
     */
    boolean isCommitTarget() {
        return this.commitTarget;
    }

    /**
     * @return voteOk
     */
    boolean isVoteOk() {
        return this.voteOk;
    }

    /**
     * @param vote boolean
     */
    void setVoteOk(final boolean vote) {
        this.voteOk = vote;
    }

    /**
     * start
     * @param flag int
     * @throws XAException XAException
     */
    void start(final int flag) throws XAException {
        this.xaResource.start(this.xid, flag);
    }

    /**
     * end
     * @param flag int
     * @throws XAException XAException
     */
    void end(final int flag) throws XAException {
        this.xaResource.end(this.xid, flag);
    }

    /**
     * prepare
     * @return result
     * @throws XAException XAException
     */
    int prepare() throws XAException {
        return this.xaResource.prepare(this.xid);
    }

    /**
     * commit
     * @param onePhase boolean
     * @throws XAException XAException
     */
    void commit(final boolean onePhase) throws XAException {
        this.xaResource.commit(this.xid, onePhase);
    }

    /**
     * rollback
     * @throws XAException XAException
     */
    void rollback() throws XAException {
        this.xaResource.rollback(this.xid);
    }

    /**
     * forget
     * @throws XAException XAException
     */
    void forget() throws XAException {
        this.xaResource.forget(this.xid);
    }
}
