/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.jta.xa;

import java.io.Serializable;
import java.nio.charset.StandardCharsets;

import javax.transaction.xa.Xid;

/**
 * {@link Xid}の実装クラスです。
 *
 * @author higa
 *
 */
public class XidImpl implements Xid, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = 1L;
    /** FORMAT_ID */
    private static final int FORMAT_ID = 0x1108;
    /** INITIAL_BRANCH_ID */
    private static final byte[] INITIAL_BRANCH_ID = convert64bytes(new byte[0]);
    /** GLOBAL_ID_BASE */
    private static final String GLOBAL_ID_BASE = System.currentTimeMillis() + "/";

    /** nextId */
    private static int nextId = 0;

    /** hashCode */
    private final int hashCode;
    /** globalId */
    private final byte[] globalId;
    /** branchId */
    private final byte[] branchId;

    /**
     * {@link XidImpl}を作成します。
     */
    public XidImpl() {
        this.hashCode = getNextId();
        this.globalId = createGlobalId();
        this.branchId = INITIAL_BRANCH_ID;
    }

    /**
     * {@link XidImpl}を作成します。
     *
     * @param xid トランザクション識別子
     * @param bid ブランチ識別子
     */
    public XidImpl(final Xid xid, final int bid) {
        this.hashCode = xid.hashCode();
        this.globalId = xid.getGlobalTransactionId();
        this.branchId = convert64bytes(Integer.toString(bid).getBytes(StandardCharsets.ISO_8859_1));
    }

    /**
     * createGlobalId
     * @return byte[]
     */
    private byte[] createGlobalId() {
        return convert64bytes((GLOBAL_ID_BASE + Integer.toString(this.hashCode)).
                getBytes(StandardCharsets.ISO_8859_1));
    }

    /**
     * @see javax.transaction.xa.Xid#getGlobalTransactionId()
     */
    @Override
    public byte[] getGlobalTransactionId() {
        return this.globalId.clone();
    }

    /**
     * @see javax.transaction.xa.Xid#getBranchQualifier()
     */
    @Override
    public byte[] getBranchQualifier() {
        return this.branchId.clone();
    }

    /**
     * @see javax.transaction.xa.Xid#getFormatId()
     */
    @Override
    public int getFormatId() {
        return FORMAT_ID;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !(obj instanceof XidImpl)) {
            return false;
        }
        final XidImpl other = (XidImpl) obj;
        if (this.hashCode != other.hashCode) {
            return false;
        }
        if (this.globalId.length != other.globalId.length
                || this.branchId.length != other.branchId.length) {
            return false;
        }
        for (int i = 0; i < this.globalId.length; ++i) {
            if (this.globalId[i] != other.globalId[i]) {
                return false;
            }
        }
        for (int i = 0; i < this.branchId.length; ++i) {
            if (this.branchId[i] != other.branchId[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return this.hashCode;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "[FormatId=" + FORMAT_ID + ", GlobalId="
                + new String(this.globalId, StandardCharsets.ISO_8859_1).trim() + ", BranchId="
                + new String(this.branchId, StandardCharsets.ISO_8859_1).trim() + "]";
    }

    /**
     * convert64bytes
     * @param bytes input
     * @return byte[]
     */
    private static byte[] convert64bytes(final byte[] bytes) {
        final byte[] new64bytes = new byte[64];
        System.arraycopy(bytes, 0, new64bytes, 0, bytes.length);
        return new64bytes;
    }

    /**
     * getNextId
     * @return nextId
     */
    private static synchronized int getNextId() {
        return nextId++;
    }
}
