/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.ServletContext;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for servlet context
 * init parameters.</p>
 *
 * @version $Id$
 */
final class ServletInitParamMap implements Map<String, String>, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = 3962218821550387405L;

    /** ServletContext */
    private transient ServletContext context = null;

    /**
     * Constructor
     *
     * @param ctx ServletContext
     */
    ServletInitParamMap(final ServletContext ctx) {
        this.context = ctx;
    }

    /**
     * @see java.util.Map#clear()
     */
    @Override
    public void clear() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    @Override
    public boolean containsKey(final Object key) {
        return this.context.getInitParameter(toKey(key)) != null;
    }

    /**
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    @Override
    public boolean containsValue(final Object value) {
        return values().contains(String.class.cast(value));
    }

    /**
     * @see java.util.Map#entrySet()
     */
    @Override
    public Set<Entry<String, String>> entrySet() {
        Set<Entry<String, String>> set = new HashSet<>();
        for (final String key : Collections.list(this.context.getInitParameterNames())) {
            set.add(new MapEntry<>(key, this.context.getInitParameter(key), false));
        }
        return set;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object o) {
        if (o != this) {
            if (!ServletInitParamMap.class.isInstance(o)) {
                return false;
            }
            ServletInitParamMap map = (ServletInitParamMap) o;
            return this.context.equals(map.context);
        }
        return true;
    }

    /**
     * @see java.util.Map#get(java.lang.Object)
     */
    @Override
    public String get(final Object key) {
        return this.context.getInitParameter(toKey(key));
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return this.context.hashCode();
    }

    /**
     * @see java.util.Map#isEmpty()
     */
    @Override
    public boolean isEmpty() {
        return size() < 1;
    }

    /**
     * @see java.util.Map#keySet()
     */
    @Override
    public Set<String> keySet() {
        return new HashSet<>(Collections.list(this.context.getInitParameterNames()));
    }

    /**
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    @Override
    public String put(final String key, final String value) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#putAll(java.util.Map)
     */
    @Override
    public void putAll(final Map<? extends String, ? extends String> map) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#remove(java.lang.Object)
     */
    @Override
    public String remove(final Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see java.util.Map#size()
     */
    @Override
    public int size() {
        return Collections.list(this.context.getInitParameterNames()).size();
    }

    /**
     * @see java.util.Map#values()
     */
    @Override
    public Collection<String> values() {
        List<String> list = new ArrayList<>();
        for (final String key : Collections.list(this.context.getInitParameterNames())) {
            list.add(this.context.getInitParameter(key));
        }
        return list;
    }

    /**
     * to Key String
     * @param key Key
     * @return Key String
     */
    private String toKey(final Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        return key.toString();
    }

}
