/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web;

import java.security.Principal;
import java.util.Objects;
import java.util.stream.Stream;

/**
 * <p>Mock <strong>Principal</strong> object for low-level unit tests.</p>
 */

public final class MockPrincipal implements Principal {

    /** name */
    private final String name;
    /** roles */
    private final String[] roles;

    /**
     * constructor
     */
    public MockPrincipal() {
        super();
        this.name = "";
        this.roles = new String[0];
    }

    /**
     * @param val String
     */
    public MockPrincipal(final String val) {
        super();
        this.name = val;
        this.roles = new String[0];
    }

    /**
     * @param val String
     * @param rls String[]
     */
    public MockPrincipal(final String val, final String[] rls) {
        super();
        this.name = val;
        this.roles = rls.clone();
    }


    /**
     * @see java.security.Principal#getName()
     */
    @Override
    public String getName() {
        return this.name;
    }

    /**
     * @param role String
     * @return boolean
     */
    public boolean isUserInRole(final String role) {
        return Stream.of(this.roles).anyMatch(role::equals);
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object o) {
        if (o == null) {
            return false;
        }
        if (!(o instanceof Principal)) {
            return false;
        }
        Principal p = (Principal) o;
        if (this.name == null) {
            return p.getName() == null;
        }
        return this.name.equals(p.getName());
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return Objects.toString(this.name, "").hashCode();
    }
}
