package core.file;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;

import common.db.JdbcSource;

import core.util.MojiUtil;

/**
 * テキストファイル 読込クラス
 *
 * @author Tadashi Nakayama
 */
public class EncodeInputStreamReader extends InputStreamReader {

	/** マッピング */
	private final Charset charset;

	/**
	 * コンストラクタ
	 *
	 * @param in 入力ストリーム
	 */
	public EncodeInputStreamReader(final InputStream in) {
		super(in);
		this.charset = JdbcSource.getCharset();
	}

	/**
	 * コンストラクタ
	 *
	 * @param in 入力ストリーム
	 * @param cs キャラセット
	 */
	public EncodeInputStreamReader(final InputStream in, final Charset cs) {
		super(in, cs);
		this.charset = cs;
	}

	/**
	 * コンストラクタ
	 *
	 * @param in 入力ストリーム
	 * @param enc キャラセットデコーダ
	 */
	public EncodeInputStreamReader(final InputStream in, final CharsetDecoder enc) {
		super(in, enc);
		this.charset = enc.charset();
	}

	/**
	 * @see java.io.InputStreamReader#read()
	 */
	@Override
	public int read() throws IOException {
		return MojiUtil.correctGarbled(super.read(), this.charset);
	}

	/**
	 * @see java.io.InputStreamReader#read(char[], int, int)
	 */
	@Override
	public int read(final char[] cbuf, final int off, final int len) throws IOException {
		final var ret = super.read(cbuf, off, len);
		for (var i = 0; i < len; i++) {
			cbuf[off + i] = (char)MojiUtil.correctGarbled(cbuf[off + i], this.charset);
		}
		return ret;
	}
}
