package project.check.attribute;

import java.math.BigDecimal;
import java.util.Optional;
import java.util.stream.Stream;

import core.util.NumberUtil;
import core.util.bean.Pair;
import online.context.check.ItemCheck;
import online.context.check.ItemConvert;
import online.model.ModelUtil;
import project.check.BaseCheck;
import project.common.CheckUtil;

/**
 * 小数点指定桁数内チェック
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class DecimalCheck extends BaseCheck implements ItemCheck, ItemConvert<BigDecimal> {

	/** 指定桁数 */
	private final int number;
	/** 指定桁数 */
	private final int decimal;
	/** メッセージID */
	private final String mid;
	/** メッセージパラメータ */
	private final String[] prm;

	/**
	 * コンストラクタ
	 * @param num 指定整数部桁数
	 * @param dec 指定小数部桁数
	 */
	public DecimalCheck(final int num, final int dec) {
		this.number = num;
		this.decimal = dec;
		this.mid = null;
		this.prm = null;
	}

	/**
	 * コンストラクタ
	 *
	 * @param num 指定整数部桁数
	 * @param dec 指定小数部桁数
	 * @param msgId メッセージID
	 * @param param メッセージパラメータ
	 */
	public DecimalCheck(final int num, final int dec, final String msgId, final String... param) {
		this.number = num;
		this.decimal = dec;
		this.mid = msgId;
		this.prm = extend(param, String.valueOf(num), String.valueOf(dec));
	}

	/**
	 * @see online.context.check.ItemCheck#check(java.lang.String...)
	 */
	@Override
	public void check(final String... items) {
		Optional<Pair<String, Integer>> check = Optional.empty();
		for (final var item : items) {
			final var val = getArrayParameter(item);
			for (var j = 0; val != null && j < val.length; j++) {
				if (!isTarget(j)) {
					val[j] = ModelUtil.getValueAsString(getUniModel(), item, j);
				} else if (!CheckUtil.isDecimal(val[j], this.number, this.decimal)) {
					setMessage(item, j, this.mid, this.prm);
					check = Optional.of(check.orElse(Pair.of(item, j)));
				}
			}
		}

		check.ifPresent(this::throwKeepCheckException);
	}

	/**
	 * @see online.context.check.ItemConvert#convert(java.lang.String...)
	 */
	@Override
	public BigDecimal[] convert(final String... val) {
		return Stream.of(val).map(NumberUtil::toBigDecimal).toArray(BigDecimal[]::new);
	}
}
