/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.seasar.extension.dbcp.impl;

import java.io.PrintWriter;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Hashtable;
import java.util.Map;
import java.util.logging.Logger;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.sql.XAConnection;
import javax.sql.XADataSource;

/**
 * {@link javax.sql.DataSource}から取得したJDBCコネクションを使用する{@link javax.sql.XADataSource}の実装です。
 * <p>
 * このXAデータソースが提供する{@link javax.sql.XAConnection}は非XAコネクションのラッパーであり、
 * 2フェーズ・コミット・プロトコルをシミュレートします。 これは真の2フェーズ・コミット・プロトコルではありません。
 * </p>
 *
 * @author koichik
 */
public class DataSourceXADataSource implements XADataSource {

    /** JNDIの{@link javax.naming.InitialContext 初期コンテキスト}を作成するための環境 */
    private final Hashtable<String, Object> env = new Hashtable<>();

    /** JNDIからルックアップするデータソース名 */
    private String dataSourceName;
    /** JNDIからルックアップしたデータソース */
    private DataSource dataSource;

    /**
     * JNDIからルックアップするデータソース名を設定します。
     *
     * @param lookupName JNDIからルックアップするデータソース名
     */
    public synchronized void setDataSourceName(final String lookupName) {
        this.dataSourceName = lookupName;
    }

    /**
     * JNDIの{@link javax.naming.InitialContext 初期コンテキスト}を作成するための環境を設定します。
     *
     * @param map JNDIの初期コンテキストを作成するための環境
     */
    public void setEnv(final Map<String, ?> map) {
        this.env.putAll(map);
    }

    /**
     * @see javax.sql.CommonDataSource#getLogWriter()
     */
    @Override
    public PrintWriter getLogWriter() {
        return null;
    }

    /**
     * @see javax.sql.CommonDataSource#getLoginTimeout()
     */
    @Override
    public int getLoginTimeout() {
        return 0;
    }

    /**
     * @see javax.sql.XADataSource#getXAConnection()
     */
    @Override
    public XAConnection getXAConnection() throws SQLException {
        return new XAConnectionImpl(getDataSource().getConnection());
    }

    /**
     * @see javax.sql.XADataSource#getXAConnection(java.lang.String, java.lang.String)
     */
    @Override
    public XAConnection getXAConnection(final String user, final String password)
            throws SQLException {
        return new XAConnectionImpl(getDataSource().getConnection(user, password));
    }

    /**
     * @see javax.sql.CommonDataSource#setLogWriter(java.io.PrintWriter)
     */
    @Override
    public void setLogWriter(final PrintWriter out) {
        return;
    }

    /**
     * @see javax.sql.CommonDataSource#setLoginTimeout(int)
     */
    @Override
    public void setLoginTimeout(final int seconds) {
        return;
    }

    /**
     * JNDIからルックアップしたデータソースを返します。
     *
     * @return JNDIからルックアップしたデータソース
     */
    protected synchronized DataSource getDataSource() {
        if (this.dataSource == null) {
            final InitialContext ctx = create(this.env);
            this.dataSource = (DataSource) lookup(ctx, this.dataSourceName);
        }
        return this.dataSource;
    }

    /**
     * 指定した環境を使用して初期コンテキストを作成して返します。
     *
     * @param env 初期コンテキストの作成に使用される環境。<code>mull</code>は空の環境を示す
     * @return 初期コンテキスト
     */
    private static InitialContext create(final Hashtable<String, ?> env) {
        try {
            return new InitialContext(env);
        } catch (final NamingException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 指定した初期コンテキストから指定されたオブジェクトを取得して返します。
     *
     * @param ctx 初期コンテキスト
     * @param jndiName 検索するオブジェクトの名前
     * @return <code>jndiName</code>にバインドされているオブジェクト
     */
    private static Object lookup(final InitialContext ctx, final String jndiName) {
        try {
            return ctx.lookup(jndiName);
        } catch (final NamingException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * @see javax.sql.CommonDataSource#getParentLogger()
     */
    @Override
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException();
    }
}
