/*
 * $Id: MockHttpServletRequest.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.mock;

import javax.servlet.AsyncContext;
import javax.servlet.DispatcherType;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpUpgradeHandler;
import javax.servlet.http.Part;

import java.io.BufferedReader;
import java.io.IOException;
import java.security.Principal;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * <p>Mock <strong>HttpServletRequest</strong> object for low-level unit tests
 * of Struts controller components.  Coarser grained tests should be
 * implemented in terms of the Cactus framework, instead of the mock object
 * classes.</p>
 *
 * <p><strong>WARNING</strong> - Only the minimal set of methods needed to
 * create unit tests is provided, plus additional methods to configure this
 * object as necessary.  Methods for unsupported operations will throw
 * <code>UnsupportedOperationException</code>.</p>
 *
 * <p><strong>WARNING</strong> - Because unit tests operate in a single
 * threaded environment, no synchronization is performed.</p>
 *
 * @version $Rev: 471754 $ $Date: 2006-11-06 08:55:09 -0600 (Mon, 06 Nov 2006) $
 */
public class MockHttpServletRequest implements HttpServletRequest {

    // ----------------------------------------------------- Instance Variables

    /**
     * <p> The set of request attributes. </p>
     */
    private Map<String, Object> attributes = new HashMap<>();

    /**
     * <p> The context path for this request. </p>
     */
    private String contextPath = null;

    /**
     * <p> The preferred locale for this request. </p>
     */
    private Locale locale = null;

    /**
     * <p> The set of arrays of parameter values, keyed by parameter name.
     * </p>
     */
    private Map<String, String[]> parameters = new HashMap<>();

    /**
     * <p> The extra path information for this request. v     * </p>
     */
    private String pathInfo = null;

    /**
     * <p> The authenticated user for this request. </p>
     */
    private Principal principal = null;

    /**
     * <p> The query string for this request. </p>
     */
    private String queryString = null;

    /**
     * <p> The servlet path for this request. </p>
     */
    private String servletPath = null;

    /**
     * <p> The HttpSession with which we are associated. </p>
     */
    private HttpSession session = null;

    /**
     * <p> The HTTP request method. </p>
     */
    private String method = null;

    /**
     * <p> The Content Type for this request. </p>
     */
    private String contentType = null;

    // ----------------------------------------------------------- Constructors

    /**
     * MockHttpServletRequest
     */
    public MockHttpServletRequest() {
        super();
    }

    /**
     * @param val HttpSession
     */
    public MockHttpServletRequest(final HttpSession val) {
        super();
        setHttpSession(val);
    }

    /**
     * @param context String
     * @param servlet String
     * @param path String
     * @param query String
     */
    public MockHttpServletRequest(final String context, final String servlet,
            final String path, final String query) {
        super();
        setPathElements(context, servlet, path, query);
    }

    /**
     * @param context String
     * @param servlet String
     * @param path String
     * @param query String
     * @param val HttpSession
     */
    public MockHttpServletRequest(final String context, final String servlet,
            final String path, final String query, final HttpSession val) {
        super();
        setPathElements(context, servlet, path, query);
        setHttpSession(val);
    }

    // --------------------------------------------------------- Public Methods

    /**
     * @param name String
     * @param value String
     */
    public void addParameter(final String name, final String value) {
        String[] values = this.parameters.get(name);

        if (values == null) {
            String[] results = new String[] {value};
            this.parameters.put(name, results);
            return;
        }

        String[] results = new String[values.length + 1];
        System.arraycopy(values, 0, results, 0, values.length);
        results[values.length] = value;
        this.parameters.put(name, results);
    }

    /**
     * @param val HttpSession
     */
    public void setHttpSession(final HttpSession val) {
        this.session = val;
    }

    /**
     * @param val Locale
     */
    public void setLocale(final Locale val) {
        this.locale = val;
    }

    /**
     * @param val String
     */
    public void setMethod(final String val) {
        this.method = val;
    }

    /**
     * @param val String
     */
    public void setContentType(final String val) {
        this.contentType = val;
    }

    /**
     * @param context String
     * @param servlet String
     * @param path String
     * @param query String
     */
    public void setPathElements(final String context, final String servlet,
            final String path, final String query) {
        this.contextPath = context;
        this.servletPath = servlet;
        this.pathInfo = path;
        this.queryString = query;
    }

    /**
     * @param val Principal
     */
    public void setUserPrincipal(final Principal val) {
        this.principal = val;
    }

    // --------------------------------------------- HttpServletRequest Methods

    /**
     * @see javax.servlet.http.HttpServletRequest#getAuthType()
     */
    @Override
    public String getAuthType() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getContextPath()
     */
    @Override
    public String getContextPath() {
        return this.contextPath;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getCookies()
     */
    @Override
    public Cookie[] getCookies() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getDateHeader(java.lang.String)
     */
    @Override
    public long getDateHeader(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getHeader(java.lang.String)
     */
    @Override
    public String getHeader(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getHeaderNames()
     */
    @Override
    public Enumeration<String> getHeaderNames() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getHeaders(java.lang.String)
     */
    @Override
    public Enumeration<String> getHeaders(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getIntHeader(java.lang.String)
     */
    @Override
    public int getIntHeader(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getMethod()
     */
    @Override
    public String getMethod() {
        return this.method;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getPathInfo()
     */
    @Override
    public String getPathInfo() {
        return this.pathInfo;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getPathTranslated()
     */
    @Override
    public String getPathTranslated() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getQueryString()
     */
    @Override
    public String getQueryString() {
        return this.queryString;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getRemoteUser()
     */
    @Override
    public String getRemoteUser() {
        if (this.principal != null) {
            return this.principal.getName();
        }
        return null;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getRequestedSessionId()
     */
    @Override
    public String getRequestedSessionId() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getRequestURI()
     */
    @Override
    public String getRequestURI() {
        StringBuffer sb = new StringBuffer();

        if (this.contextPath != null) {
            sb.append(this.contextPath);
        }

        if (this.servletPath != null) {
            sb.append(this.servletPath);
        }

        if (this.pathInfo != null) {
            sb.append(this.pathInfo);
        }

        if (sb.length() > 0) {
            return sb.toString();
        }

        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getRequestURL()
     */
    @Override
    public StringBuffer getRequestURL() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getServletPath()
     */
    @Override
    public String getServletPath() {
        return this.servletPath;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getSession()
     */
    @Override
    public HttpSession getSession() {
        return getSession(true);
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getSession(boolean)
     */
    @Override
    public HttpSession getSession(final boolean create) {
        if (create && (this.session == null)) {
            this.session = new MockHttpSession();

            // modified to act like the real deal,
            // call with (false) if you want null
            // throw new UnsupportedOperationException();
        }

        return this.session;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getUserPrincipal()
     */
    @Override
    public Principal getUserPrincipal() {
        return this.principal;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#isRequestedSessionIdFromCookie()
     */
    @Override
    public boolean isRequestedSessionIdFromCookie() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#isRequestedSessionIdFromUrl()
     */
    @Override
    public boolean isRequestedSessionIdFromUrl() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#isRequestedSessionIdFromURL()
     */
    @Override
    public boolean isRequestedSessionIdFromURL() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#isRequestedSessionIdValid()
     */
    @Override
    public boolean isRequestedSessionIdValid() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#isUserInRole(java.lang.String)
     */
    @Override
    public boolean isUserInRole(final String role) {
        if ((this.principal != null) && (this.principal instanceof MockPrincipal)) {
            return ((MockPrincipal) this.principal).isUserInRole(role);
        }
        return false;
    }

    // ------------------------------------------------- ServletRequest Methods

    /**
     * @see javax.servlet.ServletRequest#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(final String name) {
        return this.attributes.get(name);
    }

    /**
     * @see javax.servlet.ServletRequest#getAttributeNames()
     */
    @Override
    public Enumeration<String> getAttributeNames() {
        return new MockEnumeration<>(this.attributes.keySet().iterator());
    }

    /**
     * @see javax.servlet.ServletRequest#getCharacterEncoding()
     */
    @Override
    public String getCharacterEncoding() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getContentLength()
     */
    @Override
    public int getContentLength() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getContentType()
     */
    @Override
    public String getContentType() {
        return this.contentType;
    }

    /**
     * @see javax.servlet.ServletRequest#getInputStream()
     */
    @Override
    public ServletInputStream getInputStream() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getLocale()
     */
    @Override
    public Locale getLocale() {
        return this.locale;
    }

    /**
     * @see javax.servlet.ServletRequest#getLocales()
     */
    @Override
    public Enumeration<Locale> getLocales() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getParameter(java.lang.String)
     */
    @Override
    public String getParameter(final String name) {
        String[] values = this.parameters.get(name);
        if (values != null) {
            return values[0];
        }
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#getParameterMap()
     */
    @Override
    public Map<String, String[]> getParameterMap() {
        return this.parameters;
    }

    /**
     * @see javax.servlet.ServletRequest#getParameterNames()
     */
    @Override
    public Enumeration<String> getParameterNames() {
        return new MockEnumeration<>(this.parameters.keySet().iterator());
    }

    /**
     * @see javax.servlet.ServletRequest#getParameterValues(java.lang.String)
     */
    @Override
    public String[] getParameterValues(final String name) {
        return this.parameters.get(name);
    }

    /**
     * @see javax.servlet.ServletRequest#getProtocol()
     */
    @Override
    public String getProtocol() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getReader()
     */
    @Override
    public BufferedReader getReader() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getRealPath(java.lang.String)
     */
    @Override
    public String getRealPath(final String path) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getRemoteAddr()
     */
    @Override
    public String getRemoteAddr() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getRemoteHost()
     */
    @Override
    public String getRemoteHost() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getRequestDispatcher(java.lang.String)
     */
    @Override
    public RequestDispatcher getRequestDispatcher(final String path) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getScheme()
     */
    @Override
    public String getScheme() {
        return "http";
    }

    /**
     * @see javax.servlet.ServletRequest#getServerName()
     */
    @Override
    public String getServerName() {
        return "localhost";
    }

    /**
     * @see javax.servlet.ServletRequest#getServerPort()
     */
    @Override
    public int getServerPort() {
        return 8080;
    }

    /**
     * @see javax.servlet.ServletRequest#isSecure()
     */
    @Override
    public boolean isSecure() {
        return false;
    }

    /**
     * @see javax.servlet.ServletRequest#removeAttribute(java.lang.String)
     */
    @Override
    public void removeAttribute(final String name) {
        this.attributes.remove(name);
    }

    /**
     * @see javax.servlet.ServletRequest#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(final String name, final Object value) {
        if (value == null) {
            this.attributes.remove(name);
        } else {
            this.attributes.put(name, value);
        }
    }

    /**
     * @see javax.servlet.ServletRequest#setCharacterEncoding(java.lang.String)
     */
    @Override
    public void setCharacterEncoding(final String name) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.ServletRequest#getAsyncContext()
     */
    @Override
    public AsyncContext getAsyncContext() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#getDispatcherType()
     */
    @Override
    public DispatcherType getDispatcherType() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#getLocalAddr()
     */
    @Override
    public String getLocalAddr() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#getLocalName()
     */
    @Override
    public String getLocalName() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#getLocalPort()
     */
    @Override
    public int getLocalPort() {
        return 0;
    }

    /**
     * @see javax.servlet.ServletRequest#getRemotePort()
     */
    @Override
    public int getRemotePort() {
        return 0;
    }

    /**
     * @see javax.servlet.ServletRequest#getServletContext()
     */
    @Override
    public ServletContext getServletContext() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#isAsyncStarted()
     */
    @Override
    public boolean isAsyncStarted() {
        return false;
    }

    /**
     * @see javax.servlet.ServletRequest#isAsyncSupported()
     */
    @Override
    public boolean isAsyncSupported() {
        return false;
    }

    /**
     * @see javax.servlet.ServletRequest#startAsync()
     */
    @Override
    public AsyncContext startAsync() {
        return null;
    }

    /**
     * @see javax.servlet.ServletRequest#startAsync(javax.servlet.ServletRequest,
     * javax.servlet.ServletResponse)
     */
    @Override
    public AsyncContext startAsync(final ServletRequest arg0, final ServletResponse arg1) {
        return null;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#authenticate(
     * javax.servlet.http.HttpServletResponse)
     */
    @Override
    public boolean authenticate(final HttpServletResponse arg0)
            throws IOException, ServletException {
        return false;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getPart(java.lang.String)
     */
    @Override
    public Part getPart(final String arg0) throws IOException, ServletException {
        return null;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#getParts()
     */
    @Override
    public Collection<Part> getParts() throws IOException, ServletException {
        return null;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#login(java.lang.String, java.lang.String)
     */
    @Override
    public void login(final String arg0, final String arg1) throws ServletException {
        return;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#logout()
     */
    @Override
    public void logout() throws ServletException {
        return;
    }

    /**
     * @see javax.servlet.ServletRequest#getContentLengthLong()
     */
    @Override
    public long getContentLengthLong() {
        return 0;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#changeSessionId()
     */
    @Override
    public String changeSessionId() {
        return null;
    }

    /**
     * @see javax.servlet.http.HttpServletRequest#upgrade(java.lang.Class)
     */
    @Override
    public <T extends HttpUpgradeHandler> T upgrade(final Class<T> arg0)
            throws IOException, ServletException {
        return null;
    }
}
