/**************************************************************************
 UIFontChanger - Allow to change UI font for OmegaT.
 
 Copyright (C) 2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.uifontchanger;

import groovy.util.ConfigObject;
import groovy.util.ConfigSlurper;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.omegat.util.FileUtil;
import org.omegat.util.StaticUtils;

/**
 * Manage config
 * 
 * @author Yu-Tang
 */
public class Config {

    static final String FILE_NAME = "uifontchanger.conf";  // case insensitive

    private final File file;
    private ConfigObject config = null;

    public Config(File file) throws IOException {
        this.file = file;
        if (file.isFile()) {
            String script = FileUtil.readTextFile(file); // as UTF-8
            script = getSafeScript(script);
            config = createBindedConfigSlurper().parse(script);
        } else {
            config = getDefaultConfig();
        }
    }

    private ConfigSlurper createBindedConfigSlurper() {
        ConfigSlurper configSlurper = new ConfigSlurper();
        /* Bind something here!
        Map map = new HashMap<String, String>();

        // Effect.Type enum
        for (Type type: Type.values()) {
            map.put(type.name(), type);
        }

        configSlurper.setBinding(map);
        */
        return configSlurper;
    }
    
    public Object get(String key) {
        if (config != null) {
            Map flatten = config.flatten();
            if (flatten.containsKey(key)) {
                return flatten.get(key);
            }
        }
        return null;
    }

    public boolean get(String key, boolean defaultValue) {
        Object ret = get(key);
        return ret != null ? (Boolean) ret : defaultValue;
    }

    public void save() throws IOException {
        Writer writer = null;
        try {
            writer = new BufferedWriter(new FileWriter(file));
            config.writeTo(writer);
        } finally {
            try {
                if (writer != null) {
                    writer.close();
                }
            } catch(IOException ex) { /* ignore */ }
        }
    }

    public static ConfigObject getDefaultConfig() {
        ConfigObject conf = new ConfigObject();
        String defaultFontName = L10n.get(L10n.Key.UI_DEFAULT_FONT);
        if (defaultFontName != null) {
            conf.put("*", defaultFontName);
        }
        return conf;
    }

    public static File getPrimaryConfigFile() {
        File configPluginsDir = new File(StaticUtils.getConfigDir(), "plugins");

        // list all conf files in /plugins/
        List<File> fs = FileUtil.findFiles(configPluginsDir, new FileFilter() {
            @Override
            public boolean accept(File file) {
                return file.getName().equalsIgnoreCase(FILE_NAME);
            }
        });

        if (fs.isEmpty()) {
            return new File(configPluginsDir, FILE_NAME);
        } else {
            return fs.get(0); // get the first one.
        }
    }

    public static File getSecondaryConfigFile() {
        File installPluginsDir = new File(StaticUtils.installDir(), "plugins");

        // list all conf files in /plugins/
        List<File> fs = FileUtil.findFiles(installPluginsDir, new FileFilter() {
            @Override
            public boolean accept(File file) {
                return file.getName().equalsIgnoreCase(FILE_NAME);
            }
        });

        if (fs.isEmpty()) {
            return new File(installPluginsDir, FILE_NAME);
        } else {
            return fs.get(0); // get the first one.
        }
    }

    /**
     * 設定ファイルを取得する
     * @return 候補パスのいずれかにファイルが実在すればそれを、どれも実在
     *          しなければ優先パスを返す
     */
    public static File getConfigFile() {
        File primaryConfigFile = getPrimaryConfigFile();
        if (primaryConfigFile.isFile()) {
            return primaryConfigFile;
        }
        File secondaryConfigFile = getSecondaryConfigFile();
        if (secondaryConfigFile.isFile()) {
            return secondaryConfigFile;
        }
        return primaryConfigFile;
    }

    /**
     * スクリプトを安全な内容に変換して返します。
     * 
     * @param script
     * @return 
     */
    private String getSafeScript(final String script) {
        final String RE_IMPLICIT_LIST_VARIANT = "(?m)^\\s*List(?=\\.font\\s*=)";
        final String RE_EXPLICIT_LIST_VARIANT = "this\\.'List'";
        final String RE_IMPLICIT_UNIVERSAL_VARIANT = "(?m)^\\s*\\*(?=\\s*=)";
        final String RE_EXPLICIT_UNIVERSAL_VARIANT = "this\\.'\\*'";
        final Pattern PATTERN_FONT_IMPORT_DECLARATION = Pattern.compile("^import\\s+java\\.awt\\.Font;", Pattern.MULTILINE);
        String ret = script;

        // case 1. List.font キー
        // "List.font = " で始まる行が有った場合、"this.'List'.font = " に置換します。
        // "List" は java.util.List インターフェイスを指すと解釈されるため、
        // そのままでは List に font というフィールドは無い旨のエラーが発生
        // します。そこで、事前に "this.'List'" の形に直しておくことで、
        // キー名文字列であることを明示してエラーを回避します。
        ret = ret.replaceAll(RE_IMPLICIT_LIST_VARIANT, RE_EXPLICIT_LIST_VARIANT);
        
        // case 2. 全称キー
        // case 1 と同様の理由で、"* = " で始まる行が有った場合、"this.'*' = " 
        // に置換します。
        ret = ret.replaceAll(RE_IMPLICIT_UNIVERSAL_VARIANT, RE_EXPLICIT_UNIVERSAL_VARIANT);

        // case 3. Font クラスのインポート
        // 設定ファイル内で "new Font(...)" や "Font.SANS_SERIF" のような記述を
        // 許容するために、Font クラスのインポート宣言を付加します。
        if (!PATTERN_FONT_IMPORT_DECLARATION.matcher(script).find()) {
            ret = "import java.awt.Font;" + ret;
        }

        return ret;
    }
}
