/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.EventQueue.invokeLater;
import static java.sql.Types.*;
import static java.util.Collections.nCopies;
import static net.argius.stew.Iteration.*;
import static net.argius.stew.ui.window.Resource.EMPTY_STRING;

import java.math.*;
import java.sql.*;
import java.util.*;
import java.util.concurrent.*;
import java.util.regex.*;

import javax.swing.table.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * ʃZbge[utTableModelB
 * ɍXV@\(DB)gĂB
 */
final class ResultSetTableModel extends DefaultTableModel {

    private static final Logger log = LoggerFactory.getLogger(ResultSetTableModel.class);

    private static final long serialVersionUID = -8861356207097438822L;
    private static final String PTN1 = "\\s*SELECT\\s.+?\\sFROM\\s([^\\s]+).*";

    private final int[] types;
    private final String commandString;

    private Connection conn;
    private Object tableName;
    private String[] primaryKeys;
    private boolean updatable;
    private boolean linkable;

    ResultSetTableModel(ResultSetReference ref) throws SQLException {
        super(0, getColumnCount(ref));
        ResultSet rs = ref.getResultSet();
        ColumnOrder order = ref.getOrder();
        final String cmd = ref.getCommandString();
        final boolean orderIsEmpty = order.size() == 0;
        ResultSetMetaData meta = rs.getMetaData();
        final int columnCount = getColumnCount();
        int[] types = new int[columnCount];
        for (int i = 0; i < columnCount; i++) {
            final int type;
            final String name;
            if (orderIsEmpty) {
                type = meta.getColumnType(i + 1);
                name = meta.getColumnName(i + 1);
            } else {
                type = meta.getColumnType(order.getOrder(i));
                name = order.getName(i);
            }
            types[i] = type;
            @SuppressWarnings({"unchecked", "unused"})
            Object o = columnIdentifiers.set(i, name);
        }
        this.types = types;
        this.commandString = cmd;
        try {
            analyzeForLinking(rs, cmd);
        } catch (Exception ex) {
            log.warn("", ex);
        }
    }

    private static final class UnlinkedRow extends Vector<Object> {

        UnlinkedRow(Vector<?> rowData) {
            super(rowData);
        }

        UnlinkedRow(Object[] rowData) {
            super(rowData.length);
            for (final Object o : rowData) {
                add(o);
            }
        }

    }

    private static int getColumnCount(ResultSetReference ref) throws SQLException {
        final int size = ref.getOrder().size();
        return (size == 0) ? ref.getResultSet().getMetaData().getColumnCount() : size;
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        switch (types[columnIndex]) {
            case CHAR:
            case VARCHAR:
            case LONGVARCHAR:
                return String.class;
            case BOOLEAN:
            case BIT:
                return Boolean.class;
            case TINYINT:
                return Byte.class;
            case SMALLINT:
                return Short.class;
            case INTEGER:
                return Integer.class;
            case BIGINT:
                return Long.class;
            case REAL:
                return Float.class;
            case DOUBLE:
            case FLOAT:
                return Double.class;
            case DECIMAL:
            case NUMERIC:
                return BigDecimal.class;
            default:
                return Object.class;
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        if (primaryKeys == null || primaryKeys.length == 0) {
            return false;
        }
        return super.isCellEditable(row, column);
    }

    @Override
    public void setValueAt(Object newValue, int row, int column) {
        if (!linkable) {
            return;
        }
        final Object oldValue = getValueAt(row, column);
        final boolean changed;
        if (newValue == null) {
            changed = (newValue != oldValue);
        } else {
            changed = !newValue.equals(oldValue);
        }
        if (changed) {
            if (isLinkedRow(row)) {
                Object[] keys = columnIdentifiers.toArray();
                try {
                    executeUpdate(getRowData(keys, row), keys[column], newValue);
                } catch (Exception ex) {
                    log.error("", ex);
                    throw new RuntimeException(ex);
                }
            } else {
                if (log.isTraceEnabled()) {
                    log.debug("update unlinked row");
                }
            }
        } else {
            if (log.isDebugEnabled()) {
                log.debug("skip to update");
            }
        }
        super.setValueAt(newValue, row, column);
    }

    /**
     * 񃊃NsǉB
     * @param rowData sf[^
     */
    void addUnlinkedRow(Object[] rowData) {
        addUnlinkedRow(convertToVector(rowData));
    }

    /**
     * 񃊃NsǉB
     * @param rowData sf[^
     */
    void addUnlinkedRow(Vector<?> rowData) {
        addRow(new UnlinkedRow(rowData));
    }

    /**
     * 񃊃Ns}B
     * @param row }s
     * @param rowData sf[^
     */
    void insertUnlinkedRow(int row, Object[] rowData) {
        insertUnlinkedRow(row, new UnlinkedRow(rowData));
    }

    /**
     * 񃊃Ns}B
     * @param row }s
     * @param rowData sf[^
     */
    void insertUnlinkedRow(int row, Vector<?> rowData) {
        insertRow(row, new UnlinkedRow(rowData));
    }

    /**
     * sDBƃNB
     * @param rowIndex sCfbNX
     * @return Nɐꍇ<code>true</code>A
     *         ɃNĂꍇ<code>false</code>
     * @throws SQLException SQLG[ɂ胊Nsꍇ
     */
    boolean linkRow(int rowIndex) throws SQLException {
        if (isLinkedRow(rowIndex)) {
            return false;
        }
        executeInsert(getRowData(columnIdentifiers.toArray(), rowIndex));
        @SuppressWarnings("unchecked")
        Vector<Object> rows = getDataVector();
        rows.set(rowIndex, new Vector<Object>((Vector<?>)rows.get(rowIndex)));
        return true;
    }

    /**
     * Ns폜B
     * @param rowIndex sCfbNX
     * @return Nɐꍇ<code>true</code>A
     *         ɃNĂꍇ<code>false</code>
     * @throws SQLException SQLG[ɂ胊Nsꍇ
     */
    boolean removeLinkedRow(int rowIndex) throws SQLException {
        if (!isLinkedRow(rowIndex)) {
            return false;
        }
        executeDelete(getRowData(columnIdentifiers.toArray(), rowIndex));
        super.removeRow(rowIndex);
        return true;
    }

    /**
     * sf[^̎擾B
     * @param keys L[ꗗ
     * @param rowIndex sCfbNX
     * @return sf[^
     */
    private Map<Object, Object> getRowData(Object[] keys, int rowIndex) {
        Map<Object, Object> rowData = new LinkedHashMap<Object, Object>();
        for (int columnIndex = 0, n = keys.length; columnIndex < n; columnIndex++) {
            rowData.put(keys[columnIndex], getValueAt(rowIndex, columnIndex));
        }
        return rowData;
    }

    /**
     * \[gB
     * @param columnIndex CfbNX
     * @param reverse ~ɂ邩ǂ
     */
    void sort(final int columnIndex, boolean reverse) {
        final int f = (reverse) ? -1 : 1;
        @SuppressWarnings("unchecked")
        List<List<Object>> dataVector = getDataVector();
        Collections.sort(dataVector, new Comparator<List<Object>>() {

            public int compare(List<Object> row1, List<Object> row2) {
                return c(row1, row2) * f;
            }

            private int c(List<Object> row1, List<Object> row2) {
                if (row1 == null || row2 == null) {
                    return row1 == null ? row2 == null ? 0 : -1 : 1;
                }
                final Object o1 = row1.get(columnIndex);
                final Object o2 = row2.get(columnIndex);
                if (o1 == null || o2 == null) {
                    return o1 == null ? o2 == null ? 0 : -1 : 1;
                }
                if (o1 instanceof Comparable<?> && o1.getClass() == o2.getClass()) {
                    @SuppressWarnings("unchecked")
                    Comparable<Object> c1 = (Comparable<Object>)o1;
                    @SuppressWarnings("unchecked")
                    Comparable<Object> c2 = (Comparable<Object>)o2;
                    return c1.compareTo(c2);
                }
                return o1.toString().compareTo(o2.toString());
            }

        });
    }

    /**
     * ̃e[uXV\ǂ𒲂ׂB
     * @return e[uXV\ǂ
     */
    boolean isUpdatable() {
        return updatable;
    }

    /**
     * ̃e[u\ǂ𒲂ׂB
     * @return e[u\ǂ
     */
    boolean isLinkable() {
        return linkable;
    }

    /**
     * Nsǂ𒲂ׂB
     * @param rowIndex sCfbNX
     * @return Nsǂ
     */
    boolean isLinkedRow(int rowIndex) {
        return !(getDataVector().get(rowIndex) instanceof UnlinkedRow);
    }

    /**
     * 񃊃Ns݂邩ǂ𒲂ׂB
     * @param rowIndex sCfbNX
     * @return 񓯊s݂邩ǂ
     */
    boolean hasUnlinkedRows() {
        for (final Object row : getDataVector()) {
            if (row instanceof UnlinkedRow) {
                return true;
            }
        }
        return false;
    }

    /**
     * fێConnectionƓǂ𒲂ׂB
     * @param conn rConnection
     * @return ǂ
     */
    boolean isSameConnection(Connection conn) {
        return conn == this.conn;
    }

    /**
     * Model𐶐R}h擾B
     * @return R}h
     */
    String getCommandString() {
        return commandString;
    }

    /**
     * ZҏWDBɔf(UPDATE)B
     * @param keyMap L[̃}bsO
     * @param target XVΏۗƍXVl
     * @throws SQLException 
     */
    private void executeUpdate(Map<Object, Object> keyMap, Object targetKey, Object targetValue) throws SQLException {
        final String sql = String.format("UPDATE %s SET %s=? WHERE %s",
                                         tableName,
                                         quoteIfNeeds(targetKey),
                                         getPrimaryKeyClauseString());
        List<Object> a = new ArrayList<Object>();
        a.add(targetValue);
        for (Object pk : primaryKeys) {
            a.add(keyMap.get(pk));
        }
        executeSql(sql, a.toArray());
    }

    /**
     * ZҏWDBɔf(INSERT)B
     * @param rowData sf[^
     * @throws SQLException 
     */
    private void executeInsert(Map<Object, Object> rowData) throws SQLException {
        Set<Object> keys = rowData.keySet();
        Iteration.Correspondence<Object, String> c = new Iteration.Correspondence<Object, String>() {
            public String f(Object o) {
                // never null
                return quoteIfNeeds(o.toString());
            }
        };
        final String sql = String.format("INSERT INTO %s (%s) VALUES (%s)",
                                         tableName,
                                         join(map(keys, c), ", "),
                                         join(nCopies(keys.size(), "?"), ","));
        executeSql(sql, rowData.values().toArray());
    }

    /**
     * ZҏWDBɔf(DELETE)B
     * @param keyMap L[̃}bsO
     * @throws SQLException 
     */
    private void executeDelete(Map<Object, Object> keyMap) throws SQLException {
        final String sql = String.format("DELETE FROM %s WHERE %s",
                                         tableName,
                                         getPrimaryKeyClauseString());
        List<Object> a = new ArrayList<Object>();
        for (Object pk : primaryKeys) {
            a.add(keyMap.get(pk));
        }
        executeSql(sql, a.toArray());
    }

    /**
     * vC}L[啶̎擾B
     * @return vC}L[啶
     */
    private String getPrimaryKeyClauseString() {
        Iteration.Correspondence<String, String> c = new Iteration.Correspondence<String, String>() {
            public String f(String s) {
                return String.format("%s=?", quoteIfNeeds(s));
            }
        };
        return join(map(primaryKeys, c), " AND ");
    }

    /**
     * SQLsB
     * @param sql SQL
     * @param parameters oChp[^
     * @throws SQLException 
     */
    private void executeSql(final String sql, final Object[] parameters) throws SQLException {
        if (log.isDebugEnabled()) {
            log.debug("SQL: " + sql);
            log.debug("parameters: " + Arrays.asList(parameters));
        }
        final CountDownLatch latch = new CountDownLatch(1);
        final List<SQLException> errors = new ArrayList<SQLException>();
        ExecutorService executor = Executors.newSingleThreadExecutor();
        // 񓯊s
        executor.execute(new Runnable() {
            @SuppressWarnings("synthetic-access")
            public void run() {
                try {
                    final PreparedStatement stmt = conn.prepareStatement(sql);
                    try {
                        ValueTransporter transporter = ValueTransporter.getInstance("");
                        int index = 0;
                        for (Object o : parameters) {
                            boolean isNull = false;
                            if (o == null || String.valueOf(o).length() == 0) {
                                if (getColumnClass(index) != String.class) {
                                    isNull = true;
                                }
                            }
                            ++index;
                            if (isNull) {
                                stmt.setNull(index, types[index - 1]);
                            } else {
                                transporter.setObject(stmt, index, o);
                            }
                        }
                        final int updatedCount = stmt.executeUpdate();
                        if (updatedCount != 1) {
                            throw new SQLException("updated count is not 1, but " + updatedCount);
                        }
                    } finally {
                        stmt.close();
                    }
                } catch (SQLException ex) {
                    errors.add(ex);
                }
                latch.countDown();
            }
        });
        try {
            // 񓯊s𐧌Ԃ܂ő҂
            latch.await(3L, TimeUnit.SECONDS);
        } catch (InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        if (latch.getCount() != 0) {
            // SQLsԓɏIȂꍇ
            executor.execute(new Runnable() {
                @SuppressWarnings("synthetic-access")
                public void run() {
                    try {
                        latch.await();
                    } catch (InterruptedException ex) {
                        log.warn("", ex);
                    }
                    if (!errors.isEmpty()) {
                        invokeLater(new Runnable() {
                            public void run() {
                                WindowOutputProcessor.showErrorDialog(null, errors.get(0));
                            }
                        });
                    }
                }
            });
        } else if (!errors.isEmpty()) {
            // SQLsԓɏIăG[ꍇ
            if (log.isDebugEnabled()) {
                for (final Exception ex : errors) {
                    log.debug("", ex);
                }
            }
            throw errors.get(0);
        }
    }

    /**
     * Kvȏꍇ͈pň͂ށB
     * @param o V{
     * @return ꂽ
     */
    static String quoteIfNeeds(Object o) {
        final String s = String.valueOf(o);
        if (s.matches(".*\\W.*")) {
            return String.format("\"%s\"", s);
        }
        return s;
    }

    /**
     * Ns߂̉͂sB
     * @param rs ʃZbg
     * @param cmd R}h
     * @throws SQLException
     */
    private void analyzeForLinking(ResultSet rs, String cmd) throws SQLException {
        if (rs == null) {
            return;
        }
        Statement stmt = rs.getStatement();
        if (stmt == null) {
            return;
        }
        Connection conn = stmt.getConnection();
        if (conn == null) {
            return;
        }
        this.conn = conn;
        if (conn.isReadOnly()) {
            return;
        }
        final String tableName = findTableName(cmd);
        if (tableName.length() == 0) {
            return;
        }
        this.tableName = tableName;
        this.updatable = true;
        List<String> pkList = findPrimaryKeys(conn, tableName);
        if (pkList.isEmpty()) {
            return;
        }
        @SuppressWarnings("unchecked")
        final Collection<Object> columnIdentifiers = this.columnIdentifiers;
        if (!columnIdentifiers.containsAll(pkList)) {
            return;
        }
        if (findUnion(cmd)) {
            return;
        }
        this.primaryKeys = pkList.toArray(new String[pkList.size()]);
        this.linkable = true;
    }

    /**
     * e[u̒TB
     * SELECTP̃e[ułꍇ̂݁Ae[uԂB
     * @param cmd R}h
     * @return e[u łȂꍇ͋󕶎 
     */
    static String findTableName(String cmd) {
        if (cmd != null) {
            // sERgĂ}b`O
            StringBuilder buffer = new StringBuilder();
            Scanner scanner = new Scanner(cmd);
            try {
                while (scanner.hasNextLine()) {
                    final String line = scanner.nextLine();
                    buffer.append(line.replaceAll("/\\*.*?\\*/|//.*", ""));
                    buffer.append(' ');
                }
            } finally {
                scanner.close();
            }
            Pattern p = Pattern.compile(PTN1, Pattern.CASE_INSENSITIVE);
            Matcher m = p.matcher(buffer);
            if (m.matches()) {
                String afterFrom = m.group(1);
                String[] words = afterFrom.split("\\s");
                boolean foundComma = false;
                for (int i = 0; i < 2 && i < words.length; i++) {
                    String word = words[i];
                    if (word.indexOf(',') >= 0) {
                        foundComma = true;
                    }
                }
                if (!foundComma) {
                    String word = words[0];
                    if (word.matches("[A-Za-z0-9_\\.]+")) {
                        return word;
                    }
                }
            }
        }
        return EMPTY_STRING;
    }

    /**
     * vC}L[̒TB
     * P̃e[uׂẴvC}L[łꍇ̂݁Ae[uԂB
     * @param conn RlNV
     * @param tableName e[u
     * @return vC}L[ꗗ łȂꍇ͋󃊃Xg 
     * @throws SQLException
     */
    private static List<String> findPrimaryKeys(Connection conn, String tableName) throws SQLException {
        // ݒ
        DatabaseMetaData dbmeta = conn.getMetaData();
        String schema = dbmeta.getUserName();
        if (schema == null) {
            schema = EMPTY_STRING;
        }
        String schemaCondition;
        String tableNameCondition;
        if (dbmeta.storesLowerCaseIdentifiers()) {
            schemaCondition = schema.toLowerCase();
            tableNameCondition = tableName.toLowerCase();
        } else if (dbmeta.storesUpperCaseIdentifiers()) {
            schemaCondition = schema.toUpperCase();
            tableNameCondition = tableName.toUpperCase();
        } else {
            schemaCondition = schema;
            tableNameCondition = tableName;
        }
        if (tableNameCondition.indexOf('.') >= 0) {
            String[] splitted = tableNameCondition.split("\\.");
            schemaCondition = splitted[0];
            tableNameCondition = splitted[1];
        }
        // 
        List<String> pkList = getPrimaryKeys(dbmeta, schemaCondition, tableNameCondition);
        if (pkList.isEmpty()) {
            return getPrimaryKeys(dbmeta, null, tableNameCondition);
        }
        return pkList;
    }

    /**
     * vC}L[̎擾B
     * @param dbmeta DatabaseMetaData
     * @param schema XL[}
     * @param table e[u
     * @return vC}L[̃Xg
     * @throws SQLException
     */
    private static List<String> getPrimaryKeys(DatabaseMetaData dbmeta, String schema, String table) throws SQLException {
        ResultSet rs = dbmeta.getPrimaryKeys(null, schema, table);
        try {
            Map<Short, String> result = new TreeMap<Short, String>();
            Set<String> schemaSet = new HashSet<String>();
            while (rs.next()) {
                result.put(rs.getShort(5), rs.getString(4));
                schemaSet.add(rs.getString(2));
            }
            if (result.isEmpty() || schemaSet.size() != 1) {
                return Collections.emptyList();
            }
            List<String> pkList = new ArrayList<String>();
            for (Short key : result.keySet()) {
                pkList.add(result.get(key));
            }
            return pkList;
        } finally {
            rs.close();
        }
    }

    /**
     * UNIONL[[h̒TB
     * @param sql SQL
     * @return UNIONL[[hꍇ <code>true</code>A
     *         Ȃꍇ <code>false</code>
     */
    private static boolean findUnion(String sql) {
        String s = sql;
        if (s.indexOf('\'') >= 0) {
            if (s.indexOf("\\'") >= 0) {
                s = s.replaceAll("\\'", "");
            }
            s = s.replaceAll("'[^']+'", "''");
        }
        StringTokenizer tokenizer = new StringTokenizer(s);
        while (tokenizer.hasMoreTokens()) {
            if (tokenizer.nextToken().equalsIgnoreCase("UNION")) {
                return true;
            }
        }
        return false;
    }

}
