/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.ALT_DOWN_MASK;
import static java.awt.event.KeyEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.event.*;

import javax.swing.*;
import javax.swing.text.*;
import javax.swing.undo.*;

import net.argius.stew.ui.window.ActionUtility.ActionCommandListener;

/**
 * [^CṽeLXgGAB
 */
final class ConsoleTextArea extends JTextArea implements ActionCommandListener {

    private static final String cmdSubmit = "submit";
    private static final String cmdCopyOrBreak = "copy-or-break";
    private static final String cmdBreak = "break";
    private static final String cmdAddNewLine = "add-new-line";
    private static final String cmdJumpToHomePosition = "jump-to-home-position";

    private final UndoManager undoManager;

    private int homePosition;

    ConsoleTextArea() {
        ActionUtility au = ActionUtility.getInstance(this);
        // [CX^X]
        this.undoManager = au.setUndoAction();
        ((AbstractDocument)getDocument()).setDocumentFilter(new DocumentFilter() {
            @Override
            public void insertString(FilterBypass fb, int offset, String string, AttributeSet attr) throws BadLocationException {
                if (isEditablePosition(offset)) {
                    super.insertString(fb, offset, string, attr);
                }
            }
            @Override
            public void remove(FilterBypass fb, int offset, int length) throws BadLocationException {
                if (isEditablePosition(offset)) {
                    super.remove(fb, offset, length);
                }
            }
            @Override
            public void replace(FilterBypass fb,
                                int offset,
                                int length,
                                String text,
                                AttributeSet attrs) throws BadLocationException {
                if (isEditablePosition(offset)) {
                    super.replace(fb, offset, length, text, attrs);
                }
            }
        });
        // [ANV]
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        au.bindAction(this, cmdSubmit, getKeyStroke(VK_ENTER, 0), getKeyStroke(VK_M, shortcutKey));
        au.bindAction(this, cmdCopyOrBreak, getKeyStroke(VK_C, shortcutKey));
        au.bindAction(this, cmdBreak, getKeyStroke(VK_B, ALT_DOWN_MASK));
        au.bindAction(this, cmdAddNewLine, getKeyStroke(VK_ENTER, shortcutKey));
        au.bindAction(this, cmdJumpToHomePosition, getKeyStroke(VK_HOME, 0));
    }

    /* @see net.argius.stew.ui.window.ActionUtility.ActionCommandListener#actionCommandPerform(java.awt.event.ActionEvent, java.lang.String) */
    public void actionCommandPerform(ActionEvent e, String cmd) {
        if (cmd.equals(cmdSubmit)) {
            int cp = getCaretPosition();
            int ep = getEndPosition();
            if (cp != ep) {
                setCaretPosition(ep);
                return;
            }
            fireActionPerformed();
        } else if (cmd.equals(cmdCopyOrBreak)) {
            if (getSelectedText() == null) {
                sendBreak();
            } else {
                Action copyAction = new DefaultEditorKit.CopyAction();
                copyAction.actionPerformed(e);
            }
        } else if (cmd.equals(cmdBreak)) {
            sendBreak();
        } else if (cmd.equals(cmdAddNewLine)) {
            insert(EOL, getCaretPosition());
        } else if (cmd.equals(cmdJumpToHomePosition)) {
            setCaretPosition(getHomePosition());
        }
    }

    /**
     * ActionListener̓o^B
     * @param listener ActionListener
     */
    void addActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.add(ActionListener.class, listener);
    }

    /**
     * ActionListener̍폜B
     * @param listener ActionListener
     */
    void removeActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.remove(ActionListener.class, listener);
    }

    /**
     * Action̒ʒmB
     */
    void fireActionPerformed() {
        ActionEvent event = new ActionEvent(this, 0, getEditableText());
        for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
            listener.actionPerformed(event);
        }
    }

    boolean canUndo() {
        return undoManager.canUndo();
    }

    boolean canRedo() {
        return undoManager.canRedo();
    }

    /**
     * ǉB
     * @param s 
     * @param movesCaretToEnd Lbg𖖔Ɉړ邩ǂ
     */
    void append(String s, boolean movesCaretToEnd) {
        super.append(s);
        if (movesCaretToEnd) {
            setCaretPosition(getEndPosition());
        }
    }

    /**
     * o͂B
     * @param s 
     */
    void output(String s) {
        super.append(s);
        undoManager.discardAllEdits();
        homePosition = getEndPosition();
        setCaretPosition(homePosition);
    }

    /**
     * bZ[W̖m͈͂uB
     * m͈͂Ƃ́Avvg烁bZ[W̖܂ł̂ƁB
     * @param string u镶
     */
    void replace(String string) {
        replaceRange(string, homePosition, getEndPosition());
    }

    /**
     * bZ[WNAB
     */
    void clear() {
        homePosition = 0;
        setText(EMPTY_STRING);
    }

    /**
     * ҏW\ȃeLXg̎擾B
     * @return ҏW\ȃeLXg
     */
    String getEditableText() {
        try {
            return getText(homePosition, getEndPosition() - homePosition);
        } catch (BadLocationException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * w肵ʒuҏW\ǂ𒲍B
     * @param position ʒu
     * @return w肵ʒuҏW\Ȃ<code>true</code>AłȂ<code>false</code>
     */
    boolean isEditablePosition(int position) {
        return (position >= homePosition);
    }

    /**
     * ҏWJnʒu̎擾B
     * @return ҏWJnʒu
     */
    int getHomePosition() {
        return homePosition;
    }

    /**
     * ʒu̎擾B
     * @return ʒu
     */
    int getEndPosition() {
        Document document = getDocument();
        Position position = document.getEndPosition();
        return position.getOffset() - 1;
    }

    /**
     * ҏWJnʒu̍ĐݒB 
     */
    void resetHomePosition() {
        undoManager.discardAllEdits();
        homePosition = getEndPosition();
    }

    /**
     * f𑗐MB
     */
    void sendBreak() {
        append(Resource.getString("ConsoleTextArea.breakprompt"));
        resetHomePosition();
        validate();
    }

    @Override
    public void updateUI() {
        if (getCaret() == null) {
            super.updateUI();
        } else {
            final int p = getCaretPosition();
            super.updateUI();
            setCaretPosition(p);
        }
    }

}
