package hiro.yoshioka.sdh2;

import hiro.yoshioka.sdh.BindObject;
import hiro.yoshioka.sdh.CSVRecordDataHolder.HeaderData;
import hiro.yoshioka.sdh.DatabaseType;
import hiro.yoshioka.sdh.HeaderType;
import hiro.yoshioka.sdh.ResultSetDataHolder;
import hiro.yoshioka.sdh.ResultSetMetaCopy;
import hiro.yoshioka.sdh.StringRecordData;
import hiro.yoshioka.util.SQLDataType;
import hiro.yoshioka.util.SQLUtil;
import hiro.yoshioka.util.StringUtil;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ReflectionPreparedStatement {
	public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";
	private Map<String, String> columnNameSubstitutionMap;
	ResultSetMetaCopy meta;
	StringRecordData[] newData, oldData;
	int[] primaryIdx;
	String tableName;
	String schemaName;
	// boolean withOutSchemaAlias;
	ArrayList<BindObject> bindList = new ArrayList<BindObject>();
	private DatabaseType database_type;

	protected transient Log fLogger = LogFactory.getLog(getClass());

	public ReflectionPreparedStatement(DatabaseType database_type,
			ResultSetMetaCopy meta, StringRecordData[] newData,
			StringRecordData[] oldData, int[] primaryIdx, String tableName,
			String schemaName, Map<String, String> columnNameSubstitutionMap) {
		this.database_type = database_type;
		this.meta = meta;
		this.newData = newData;
		this.oldData = oldData;
		this.primaryIdx = primaryIdx;
		this.tableName = tableName;
		this.schemaName = schemaName;
		this.columnNameSubstitutionMap = columnNameSubstitutionMap;
		// this.withOutSchemaAlias = withOutSchemaAlias;
	}

	public ReflectionPreparedStatement(DatabaseType database_type,
			ResultSetMetaCopy meta, String[] datumn, String tableName,
			String schemaName) {
		this.database_type = database_type;
		this.meta = meta;
		this.tableName = tableName;
		this.schemaName = schemaName;
		newData = new StringRecordData[datumn.length + 1];
		newData[0] = new ResultSetDataHolder().new HeaderData(
				StringUtil.EMPTY_STRING, HeaderType.INSERT);
		for (int i = 1; i < newData.length; i++) {
			newData[i] = new StringRecordData(datumn[i - 1]);
		}
	}

	public void setBinds(PreparedStatement st) throws SQLException {
		for (int i = 0; i < bindList.size(); i++) {
			BindObject o = bindList.get(i);
			SQLUtil.setBinds(st, i + 1, o);
		}
	}

	public boolean isSelectQuery() {
		return newData == null;
	}

	public boolean isUpdateQuery() {
		if (newData != null) {
			HeaderData header = (HeaderData) newData[0];
			return header.update();
		}
		return false;
	}

	public boolean isInsertQuery() {
		if (newData != null) {
			HeaderData header = (HeaderData) newData[0];
			return header.insert();
		}
		return false;
	}

	public boolean isDeleteQuery() {
		if (newData != null) {
			HeaderData header = (HeaderData) newData[0];
			return header.delete();
		}
		return false;
	}

	public String getStatement() throws SQLException {
		bindList.clear();
		StringBuffer buff = new StringBuffer();
		if (newData == null) { // blobRef
			buff.append("SELECT ");
			buff.append(meta.getBlobOrBinary());
			buff.append(" FROM ");
			buff.append(getSchemaTableName(1));
			buff.append(createWhere(oldData));
			return buff.toString();
		}
		HeaderData header = (HeaderData) newData[0];
		if (header.update()) {
			for (int i = 1; i < newData.length; i++) {
				if (!newData[i].getString().equals(oldData[i].getString())) {
					if (buff.length() == 0) {
						buff.append("UPDATE ");
						if (this.database_type.needsTableNameQuote()) {
							buff.append(String.format("\"%s\"", getTableName()));
						} else {
							buff.append(getSchemaTableName(i));
						}
						buff.append(" SET ");
					}
					buff.append(getSubstitutionColumnName(meta.getColumnName(i))
							+ " = ?,");
					updateByType(SQLDataType.parse(meta.getColumnType(i)),
							newData[i].getString());
				}
			}
			buff.setLength(buff.length() - 1);
			buff.append(createWhere(oldData));
		} else if (header.insert()) {
			StringBuffer valueBuff = new StringBuffer();
			for (int i = 1; i < newData.length; i++) {
				if (meta.getColumnName(i).indexOf("$") >= 0) {
					if (fLogger.isTraceEnabled()) {
						fLogger.trace("this column [" + meta.getColumnName(i)
								+ "] is ignored.");
					}
					continue;
				}
				if (buff.length() == 0) {
					buff.append("INSERT INTO ");
					if (this.database_type.needsTableNameQuote()) {
						buff.append(String.format("\"%s\"", getTableName()));
					} else {
						buff.append(getSchemaTableName(i));
					}
					buff.append(" ( ");
				}
				buff.append(getSubstitutionColumnName(meta.getColumnName(i))
						+ " ,");
				valueBuff.append("?,");
				updateByType(SQLDataType.parse(meta.getColumnType(i)),
						newData[i].getString());
			}
			buff.setLength(buff.length() - 1);
			valueBuff.setLength(valueBuff.length() - 1);
			buff.append(") VALUES (").append(valueBuff).append(")");
		} else if (header.delete()) {
			buff.append("DELETE FROM ");
			if (this.database_type.needsTableNameQuote()) {
				buff.append(String.format("\"%s\"", getTableName()));
			} else {
				buff.append(getSchemaTableName(1));
			}
			buff.append(" ");
			buff.append(createWhere(newData));
		}
		return buff.toString();
	}

	private void updateByType(SQLDataType columnType, String value) {
		fLogger.trace("columnType[" + columnType + "]");
		bindList.add(new BindObject(value, columnType));
	}

	public String getTableName() {
		return tableName;
	}

	private String getSchemaTableName(int column) throws SQLException {
		String schema = meta.getSchemaName(column);
		String table = meta.getTableName(column);
		if (!StringUtil.isEmpty(schemaName)) {
			schema = schemaName;
		}
		if (schema == null || schema.length() == 0) {
			if (table == null || table.length() == 0) {
				return tableName;
			}
			return meta.getTableName(column);
		}
		return schema + "." + meta.getTableName(column);
	}

	private String createWhere(StringRecordData[] sr) throws SQLException {
		StringBuffer buff = new StringBuffer();
		buff.append(" WHERE ");
		for (int i = 0; i < primaryIdx.length; i++) {
			if (i > 0) {
				buff.append(" AND ");
			}
			int idx = primaryIdx[i] + 1;
			buff.append(getSubstitutionColumnName(meta.getColumnName(idx)))
					.append("=?");
			updateByType(SQLDataType.parse(meta.getColumnType(idx)),
					sr[idx].getString());
		}
		return buff.toString();
	}

	private String getSubstitutionColumnName(String sourceName) {
		if (this.database_type != null) {
			if (this.database_type.needsColumnNameQuote(sourceName)) {
				return this.database_type.getQuotedColumnName(sourceName, null);
			}
		}
		if (this.columnNameSubstitutionMap == null) {
			return sourceName;
		}
		String name = this.columnNameSubstitutionMap.get(sourceName
				.toUpperCase());
		if (StringUtil.isEmpty(name)) {
			return sourceName;
		}
		return name;
	}

	public Map<String, BindObject> getUpdateValuesMap() throws SQLException {
		Map<String, BindObject> ret = new LinkedHashMap<String, BindObject>();
		HeaderData header = (HeaderData) newData[0];
		if (header.update()) {
			for (int i = 1; i < newData.length; i++) {
				if (!newData[i].getString().equals(oldData[i].getString())) {
					String key = getSubstitutionColumnName(meta
							.getColumnName(i));
					BindObject value = new BindObject(newData[i].getString(),
							SQLDataType.parse(meta.getColumnType(i)));
					ret.put(key, value);
				}
			}
		}
		return ret;
	}

	public Map<String, BindObject> getInsertValuesMap() throws SQLException {
		Map<String, BindObject> ret = new LinkedHashMap<String, BindObject>();
		HeaderData header = (HeaderData) newData[0];
		if (header.insert()) {
			for (int i = 1; i < newData.length; i++) {
				String key = getSubstitutionColumnName(meta.getColumnName(i));
				BindObject value = new BindObject(newData[i].getString(),
						SQLDataType.parse(meta.getColumnType(i)));
				ret.put(key, value);
			}
		}
		return ret;
	}

	public Map<String, BindObject> getConditionParamsMap() throws SQLException {
		HeaderData header = (HeaderData) newData[0];
		StringRecordData[] targetData = null;

		if (header.update()) {
			targetData = oldData;
		} else if (header.delete()) {
			targetData = newData;
		} else if (header.insert()) {
			return Collections.emptyMap();
		}
		Map<String, BindObject> ret = new LinkedHashMap<String, BindObject>();
		for (int i = 0; i < primaryIdx.length; i++) {
			int idx = primaryIdx[i] + 1;
			String key = getSubstitutionColumnName(meta.getColumnName(idx));
			BindObject value = new BindObject(targetData[idx].getString(),
					SQLDataType.parse(meta.getColumnType(idx)));
			ret.put(key, value);
		}

		return ret;
	}

}
