package hiro.yoshioka.sql.cassandra;

import hiro.yoshioka.sdh.DatabaseType;
import hiro.yoshioka.sdh2.ResultSetDataHolder2;
import hiro.yoshioka.sql.AbsNoSQL;
import hiro.yoshioka.sql.SQLExecutionStatus;
import hiro.yoshioka.sql.SqlBasicListener;
import hiro.yoshioka.sql.engine.GettingResourceRequest;
import hiro.yoshioka.sql.engine.MirroringRequest;
import hiro.yoshioka.sql.engine.Request;
import hiro.yoshioka.sql.engine.SQLOperationType;
import hiro.yoshioka.sql.engine.TransactionRequest;
import hiro.yoshioka.sql.params.ConnectionProperties;
import hiro.yoshioka.sql.resource.DBColumn;
import hiro.yoshioka.sql.resource.DBRoot;
import hiro.yoshioka.sql.resource.DBSchema;
import hiro.yoshioka.sql.resource.DBTable;
import hiro.yoshioka.sql.resource.IDBSchema;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.util.SQLDataType;
import hiro.yoshioka.util.StringUtil;

import java.lang.reflect.InvocationTargetException;
import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ColumnDefinitions.Definition;
import com.datastax.driver.core.ColumnMetadata;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.TableMetadata;

public class CassandraSQL extends AbsNoSQL {
	Cluster cluster;
	String host;
	int port = 9042;
	String keyspace_nm;

	public CassandraSQL() {
	}

	public DatabaseType getDatabaseType() {
		return DatabaseType.CASSANDRA;
	}

	public boolean connect(ConnectionProperties properties) throws SQLException {
		this._info = properties;
		boolean ret = false;

		host = properties.getURLString();
		keyspace_nm = null;
		Pattern P = Pattern.compile("([a-zA-Z0-9\\._-]+):([a-zA-Z0-9\\._-]+):([0-9]+)");
		Matcher m = P.matcher(host);
		port = 9042;
		if (m.matches()) {
			host = m.group(1);
			keyspace_nm = m.group(2);
			port = Integer.parseInt(m.group(3), 10);
		}

		cluster = Cluster.builder().withPort(port).addContactPoints(host)
				.build();
		ret = true;

		if (ret && fConnectionListenerList != null) {
			for (SqlBasicListener listener : fConnectionListenerList) {
				listener.connected();
			}
		}
		properties.setConnected(ret);
		return ret;
	}

	@Override
	public boolean close() throws SQLException {
		boolean ret = true;
		cluster.close();

		if (fConnectionListenerList != null) {
			for (SqlBasicListener listener : fConnectionListenerList) {
				listener.disconnected();
			}
		}
		this._info.setConnected(ret);
		return ret;
	}

	@Override
	public boolean doOperation(SQLOperationType operation, Request request)
			throws SQLException {

		TransactionRequest treq = null;
		setMakeBlobData(request.makeBlob);
		if (request instanceof TransactionRequest) {
			treq = (TransactionRequest) request;
			setMaxRowNum(treq.getMaxRownum());
		}
		ConnectionProperties prop = request.getConnectionProperties();
		long time = System.currentTimeMillis();

		boolean retCode = true;
		try {
			switch (operation) {
			case CONNECT:
				return connect(prop);
			case CLOSE:
				return close();
			case RESOURCE_MIRRORING:
				MirroringRequest mirroring_request = (MirroringRequest) request;
				retCode = createMirroredTableTo(mirroring_request);
				break;
			// case DATA_GENERATE:
			// DataGenerateRequest dreq = (DataGenerateRequest) request;
			// Table accessTbl = getAccessTable(dreq.getTableName(), false);
			// InsertCursorWorker worker = new InsertCursorWorker(accessTbl,
			// dreq.getInsertValuesMap());
			// ResultSetDataHolder2 r2 = worker.execute();
			// retCode = (r2.getIntData(0, "InsertCount") > 0);
			// break;
			case COUNT:
				int retCount = 0;
				treq.setResultCount(count(treq.getIDBTable()));
				break;
			case RESOURCE_CAPTION:
				getMetaData((GettingResourceRequest) request);
				break;
			case EXPLAIN_PLAN:
				break;
			case SELECT_SESSION:
				break;
			case SELECT_LOCK:
				break;
			case UNID_EXECUTE_QUERY:

			case SELECT_ALL:
				notifyExecute(SQLExecutionStatus.BEFORE_EXECUTE);
				treq.setRDH(getAllData2(treq.getIDBTable(), treq));
				break;
			case CREATE_TRIG_FNC_PROC:
				break;
			case WORST_SQL:
				break;
			case CHECK_VALIDATION:
				break;

			case PREPARED_EXECUTE_QUERY:
				notifyExecute(SQLExecutionStatus.BEFORE_EXECUTE);
				treq.setRDH(getAllData2(treq.getIDBTable(), treq));

				break;

			case PREPARED_EXECUTE:
				break;
			default:
				System.out.println("what's this operation ? " + operation);
				break;
			}
		} catch (Exception e) {
			notifyExecute(SQLExecutionStatus.EXCEPTION, e.getMessage());
			throw cnvSQLException(e);
		}

		return retCode;

	}

	static SQLException cnvSQLException(Throwable e) {
		if (e instanceof ExecutionException) {
			Throwable c = e.getCause();
			if (c instanceof InvocationTargetException) {
				InvocationTargetException iv = (InvocationTargetException) c;
				e = iv.getTargetException();
			}
		}
		SQLException se = new SQLException("CassandraSQLException["
				+ e.getLocalizedMessage() + "]", "", e);
		return se;
	}

	protected DBRoot getMetaData(GettingResourceRequest request) {
		IDBSchema schema = null;
		DBRoot root = getRoot();
		try {
			this.capturing = true;

			if (request.canceld()) {
				return null;
			}
			if (!request.targetType.isOnlyTable()) {
				root = new DBRoot("Cassandra");
				setRoot(root);
				root.setPropertyValue("DatabaseProductName", "Cassandra");
				fLogger.info("new DBRoot");

				root.setPropertyValue("ClusterName", cluster.getClusterName());
				root.setPropertyValue("DriverVersion",
						cluster.getDriverVersion());

				Metadata clu_meta = cluster.getMetadata();
				for (KeyspaceMetadata key_meta : clu_meta.getKeyspaces()) {
					schema = new DBSchema(root);
					schema.setName(key_meta.getName());
					schema.setComment(key_meta.getClass().getSimpleName());
					root.putResource(schema.getName(), schema);
					root.setDefaultSchema(schema);
					root.setCurrentSchema(schema);

					System.out
							.println("------------------------------------------------");
					System.out.println("Replication : ["
							+ key_meta.getReplication() + "]");
					System.out.println("DurableWrites : ["
							+ key_meta.isDurableWrites() + "]");
					System.out.println("CQL : [" + key_meta.asCQLQuery() + "]");

					for (TableMetadata tbl_meta : key_meta.getTables()) {
						DBTable table = new DBTable(schema);
						table.setName(tbl_meta.getName());
						table.setComment(tbl_meta.getClass().getSimpleName());
						table.setTableType("TABLE");
						schema.putTable(table);

						System.out.println("  Tbl-Id : [" + tbl_meta.getId()
								+ "] " + table.getName());

						if (doCaptureColumn(root, table, request)) {
							System.out.println("cap");
							setTableColumns(schema.getName(), table);
						}

					}

				}

			}
		} catch (Throwable e) {
			fLogger.error(StringUtil.EMPTY_STRING, e);
			return null;
		} finally {
			this.capturing = false;
		}
		return root;
	}

	boolean doCaptureColumn(DBRoot root, IDBTable dbTable, Request request) {
		if (this._info == null) {
			return false;
		}
		if (this._info.isCaptureWithColumnInfo()) {
			return true;
		}
		DBRoot old = request.getConnectionProperties().getDBRootResource();
		if (old != null && old.isRecentryUsed(dbTable)) {
			return true;
		}
		if (root != null && root.isRecentryUsed(dbTable)) {
			return true;
		}
		return false;
	}

	private KeyspaceMetadata findKeyspaceMetadata(String keyspace) {
		for (KeyspaceMetadata meta : cluster.getMetadata().getKeyspaces()) {
			if (meta.getName().equals(keyspace)) {
				return meta;
			}
		}
		return null;
	}

	private TableMetadata findTableMetadata(KeyspaceMetadata key_meta,
			String table) {
		for (TableMetadata tbl_meta : key_meta.getTables()) {
			if (tbl_meta.getName().equals(table)) {
				return tbl_meta;
			}
		}
		return null;
	}

	@Override
	public void setTableColumns(String schema, IDBTable table)
			throws SQLException {

		KeyspaceMetadata key_meta = findKeyspaceMetadata(schema);

		TableMetadata tbl_meta = findTableMetadata(key_meta, table.getName());
		for (ColumnMetadata col_meta : tbl_meta.getColumns()) {

			DBColumn column = new DBColumn(table);
			column.setName(col_meta.getName());
			column.setComment(col_meta.getClass().getSimpleName());

			// column.setPKey(true);
			SQLDataType data_type = SQLDataType.parse(col_meta.getType()
					.getName().name());
			column.setDataType(data_type);
			column.setDataTypeString(col_meta.getType().getName().name());
			column.setSize(0);

			table.putResource(column.getName(), column);
		}

	}

	@Override
	public ResultSetDataHolder2 getAllData2(IDBTable table,
			Request request) throws SQLException {
		String sql_statement = null;

		sql_statement = String.format("SELECT * FROM %s", table.getName());

		fLogger.info(sql_statement);
		ResultSetDataHolder2 rdh2 = query(sql_statement);
		if (rdh2 != null) {
//			rdh2.setTableNameE(table);
		}
		return rdh2;
	}

	private ResultSetDataHolder2 query(String select_query){
		ResultSetDataHolder2 rdh2 = null;
		Session session = null;
		try {
			session = cluster.connect(String.format("\"%s\"", keyspace_nm));
			ResultSet rs = session.execute(select_query);
			List<Definition> def_list = rs.getColumnDefinitions().asList();


			int colsize = def_list.size();
			String[] columns = new String[colsize];
			int[] colSize = new int[colsize];
			int i=0;
			for(Definition def: def_list){
				columns[i] = def.getName();
				i++;
			}
			rdh2 = new ResultSetDataHolder2(columns, null, DatabaseType.CASSANDRA);
			for(Row rrow : rs.all()){
				String[] row = new String[colsize];
				for (i = 0; i < colsize; i++) {
					row[i] = rrow.getString(i);
				}
				rdh2.addRow(row);
			}

			rdh2.setSqlStatement(select_query);
			// rdh2.setBinds(binds);
			rdh2.setConnectionDisplayString(_info.getDisplayString());

		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return rdh2;
	}

	public long count(IDBTable table) throws SQLException {
		if (table == null) {
			return 0L;
		}
		long sum = 0;
		KeyspaceMetadata key_meta = findKeyspaceMetadata(table.getParent()
				.getName());
		TableMetadata tbl_meta = findTableMetadata(key_meta, table.getName());

		Session session = null;
		try {
			session = cluster.connect(String.format("\"%s\"", keyspace_nm));
			ResultSet rs = session.execute(String.format("SELECT COUNT(*) FROM %s ",
					table.getName()));
			sum = rs.one().getLong(0);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (session != null) {
				session.close();
			}
		}

		return sum;
	}
}
