package map;

import java.awt.Shape;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import ksj.ShapeIO;
import web.WebUtilities;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された都道府県に関するユーティリティクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class Prefectures {
	/**
	 * 必要に応じて市区町村のデータを読み込んだり、開放したりします。
	 * @param prefectures 都道府県の一覧
	 * @param panel 地図を表示するパネル
	 * @param maps 地図
	 * @param loadMap 地図を読み込むためのオブジェクト
	 * @throws IOException 入出力例外
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public static void loadCities(final Collection<Prefecture> prefectures, final MapPanel panel,
			final Map<String, MapData> maps, final LoadMap loadMap)
			throws UnsupportedEncodingException, IOException {
		if (panel.getZoom() >= Const.ZOOM_LOAD_CITIES) {
			final Rectangle2D visibleRectangle = panel.getVisibleRectangle();
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.getBounds().intersects(visibleRectangle)) {
					final Shape shape = prefecture.hasFine() ? prefecture.getFineShape()
							: prefecture.getShape();
					if (shape.intersects(visibleRectangle)) {
						if (prefecture.hasCities()) {
						} else {
							final File textFile = new File(Const.Ksj.CACHE_DIR + File.separator
									+ Const.Ksj.TXT_PREFIX + prefecture.getId() + Const.Ksj.TXT_SUFFIX);
							if (textFile.exists()) {
								System.out.println("DEBUG: skipped getting " + textFile);
							} else {
								final File cacheDir = new File(Const.Ksj.CACHE_DIR);
								if (!cacheDir.exists()) {
									cacheDir.mkdir();
								}
								final URL url = new URL(Const.Ksj.BASE_URL + Const.Ksj.ZIP_PREFIX
										+ prefecture.getId() + Const.Ksj.ZIP_SUFFIX);
								final File file = new File(Const.Ksj.CACHE_DIR + File.separator
										+ Const.Ksj.ZIP_PREFIX + prefecture.getId()
										+ Const.Ksj.ZIP_SUFFIX);
								file.createNewFile();
								System.out.println("DEBUG: getting " + url + " -> " + file);
								WebUtilities.copy(url.openStream(), new FileOutputStream(file));
								final ZipFile zipFile = new ZipFile(file);
								for (final Enumeration<? extends ZipEntry> enumeration = zipFile
										.entries(); enumeration.hasMoreElements();) {
									final ZipEntry entry = enumeration.nextElement();
									if (entry.getName().endsWith(".txt")) {
										WebUtilities.copy(zipFile.getInputStream(entry),
												new FileOutputStream(Const.Ksj.CACHE_DIR
														+ File.separator + new File(entry.getName())));
									}
								}
							}
							prefecture.loadCities();
							panel.setNeedsRepaint(true);
							panel.repaint();
						}
					} else {
						prefecture.freeCities();
						prefecture.freeFine();
					}
				} else {
					prefecture.freeCities();
					prefecture.freeFine();
				}
			}
		} else {
			for (final Prefecture prefecture : prefectures) {
				prefecture.freeCities();
				prefecture.freeFine();
			}
		}
		if (panel.getZoom() >= Const.ZOOM_LOAD_FINE_CITIES) {
			final Rectangle2D visibleRectangle = panel.getVisibleRectangle();
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.getBounds().intersects(visibleRectangle)) {
					final Shape shape = prefecture.hasFine() ? prefecture.getFineShape()
							: prefecture.getShape();
					if (shape.intersects(visibleRectangle)) {
						if (!prefecture.hasFine()) {
							prefecture.loadFine();
							panel.setNeedsRepaint(true);
							panel.repaint();
						}
					}
				}
			}
		} else {
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.hasFine()) {
					prefecture.freeFine();
					panel.setNeedsRepaint(true);
					panel.repaint();
				}
			}
		}
		if (panel.getZoom() >= Const.ZOOM_LOAD_2500) {
			final Rectangle2D visibleRectangle = panel.getVisibleRectangle();
			final Collection<URL> urls = new ArrayList<URL>();
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.hasCities()) {
					for (final City city : prefecture.getCities()) {
						if (city.getURL() != null) {
							final Shape shape = city.hasFineShape() ? city.getFineShape() : city
									.getShape();
							if (shape.getBounds2D().intersects(visibleRectangle)) {
								if (shape.intersects(visibleRectangle)) {
									if (!city.has2500()) {
										urls.add(city.getURL());
										city.setHas2500(true);
									}
								}
							}
						}
					}
				}
			}
			if (!urls.isEmpty()) {
				loadMap.loadMaps(urls, maps);
			}
		} else {
			maps.clear();
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.hasCities()) {
					for (final City city : prefecture.getCities()) {
						city.setHas2500(false);
					}
				}
			}
		}
	}

	/**
	 * 都道府県を読み込みます。
	 * @param preferences 色の設定
	 * @return 都道府県の一覧
	 */
	public static Collection<Prefecture> loadPrefectures(final MapPreferences preferences) {
		final Collection<Prefecture> ret = new ArrayList<Prefecture>();
		for (final Map.Entry<Shape, String> entry : ShapeIO.readShapes(Const.PREFECTURES)
				.entrySet()) {
			final String[] values = entry.getValue().split("_");
			if (values.length == 2) {
				final String idString = values[0];
				final int id = Integer.parseInt(idString);
				final String label = values[1];
				ret.add(new Prefecture(entry.getKey(), label, idString, preferences
						.getTyomeFillColor(id == 30 || id == 13 ? (id + 2) % 6 + 1 : id % 6 + 1)));
			} else {
				System.out.println("WARNING: 都道府県名の表記がおかしいです。" + entry.getValue());
			}
		}
		return ret;
	}
}
