/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicecomb.common.rest.codec.query;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class QueryCodecs {
  private static final Logger LOGGER = LoggerFactory.getLogger(QueryCodecs.class);

  static QueryCodecs createForTest() {
    return new QueryCodecs(Arrays.asList(
        new QueryCodecMulti(),
        new QueryCodecCsv(),
        new QueryCodecSsv(),
        new QueryCodecTsv(),
        new QueryCodecPipes()
    ));
  }

  private final Map<String, QueryCodec> codecs = new HashMap<>();

  private final QueryCodec defaultCodec;

  public QueryCodecs(List<QueryCodec> orderedCodecs) {
    orderedCodecs.forEach(this::register);
    defaultCodec = codecs.get(QueryCodecMulti.CODEC_NAME);
  }

  private void register(@Nonnull QueryCodec codec) {
    QueryCodec exists = codecs.put(codec.getCodecName(), codec);
    if (exists != null) {
      LOGGER.info("override QueryCodec, exists={}, new={}.",
          exists.getClass().getName(), codec.getClass().getName());
    }
  }

  @Nonnull
  public QueryCodec find(@Nullable String name) {
    if (name == null) {
      return defaultCodec;
    }

    QueryCodec codec = codecs.get(name);
    if (codec == null) {
      throw new IllegalStateException("not support QueryCodec, name=" + name);
    }
    return codec;
  }
}
