﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/migrationhuborchestrator/MigrationHubOrchestratorRequest.h>
#include <aws/migrationhuborchestrator/MigrationHubOrchestrator_EXPORTS.h>
#include <aws/migrationhuborchestrator/model/TemplateSource.h>

#include <utility>

namespace Aws {
namespace MigrationHubOrchestrator {
namespace Model {

/**
 */
class CreateTemplateRequest : public MigrationHubOrchestratorRequest {
 public:
  AWS_MIGRATIONHUBORCHESTRATOR_API CreateTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTemplate"; }

  AWS_MIGRATIONHUBORCHESTRATOR_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the migration workflow template.</p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  CreateTemplateRequest& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the migration workflow template.</p>
   */
  inline const Aws::String& GetTemplateDescription() const { return m_templateDescription; }
  inline bool TemplateDescriptionHasBeenSet() const { return m_templateDescriptionHasBeenSet; }
  template <typename TemplateDescriptionT = Aws::String>
  void SetTemplateDescription(TemplateDescriptionT&& value) {
    m_templateDescriptionHasBeenSet = true;
    m_templateDescription = std::forward<TemplateDescriptionT>(value);
  }
  template <typename TemplateDescriptionT = Aws::String>
  CreateTemplateRequest& WithTemplateDescription(TemplateDescriptionT&& value) {
    SetTemplateDescription(std::forward<TemplateDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source of the migration workflow template.</p>
   */
  inline const TemplateSource& GetTemplateSource() const { return m_templateSource; }
  inline bool TemplateSourceHasBeenSet() const { return m_templateSourceHasBeenSet; }
  template <typename TemplateSourceT = TemplateSource>
  void SetTemplateSource(TemplateSourceT&& value) {
    m_templateSourceHasBeenSet = true;
    m_templateSource = std::forward<TemplateSourceT>(value);
  }
  template <typename TemplateSourceT = TemplateSource>
  CreateTemplateRequest& WithTemplateSource(TemplateSourceT&& value) {
    SetTemplateSource(std::forward<TemplateSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. For more information, see <a
   * href="https://smithy.io/2.0/spec/behavior-traits.html#idempotencytoken-trait">Idempotency</a>
   * in the Smithy documentation.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateTemplateRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to add to the migration workflow template.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateTemplateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateTemplateRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_templateName;
  bool m_templateNameHasBeenSet = false;

  Aws::String m_templateDescription;
  bool m_templateDescriptionHasBeenSet = false;

  TemplateSource m_templateSource;
  bool m_templateSourceHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubOrchestrator
}  // namespace Aws
