/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* CANVAS.H --- Canvas data structures. */

/* Line thickesses. */
typedef enum { ltNONE, ltTHIN, ltTHICK } LINE_THICKNESS;

/* Object index and null-value for index. */
typedef unsigned index;
#define INULL 0xffff
#define swap_indices(I1, I2) swap((int*)(I1), (int*)(I2))

/* Canvas coordinate. */
typedef int CC;

/* Canvas distance. */
typedef int DIST;
#define INFINITE_DIST 32767

/* Canvas point. */
typedef struct {
  CC x, y;
} CP_REC, *CP;

/* Object type tags. */
typedef enum {
  oFREE = -1,

# define DEF_OBJECT(TypeTag, Id, NonStdFieldStruct, Cmd, TextFlag, LTflag) TypeTag,
# include "obj.def"
# undef DEF_OBJECT

} OBJECT_TYPE;

enum {
  TEXT_OBJECT_TYPES = 0
# define DEF_OBJECT(TypeTag, Id, NonStdFieldStruct, Cmd, TextFlag, LTflag) | (bit(TypeTag) & (-TextFlag))
# include "obj.def"
# undef DEF_OBJECT
};

enum {
  LINE_OBJECT_TYPES = 0
# define DEF_OBJECT(TypeTag, Id, NonStdFieldStruct, Cmd, TextFlag, LTflag) | (bit(TypeTag) & (-LTflag))
# include "obj.def"
# undef DEF_OBJECT
};

typedef unsigned OBJECT_TYPE_MASK;
#define ALL_OBJECT_TYPES        (0xffffu)

/* Return non-0 iff object type tag reflects text object. */
#define text_object_p(T)        (oTEXT <= (T) && (T) <= oDASH_BOX)

typedef COLOR_TYPE CANVAS_COLOR_TYPE;
typedef COLOR_VECTOR CANVAS_COLOR_VECTOR;

/* Object. */
typedef struct object {

  /* Type tag. */
  unsigned char type;

  unsigned char 
  /* Object is picked? */
		picked_p:1,

  /* Object is transient pick for editing? */
		edit_picked_p:1, 

  /* Object is labeled by last outside_rect call? */
		outside_p:1,

  /* Thickness of lines in object. */
		line_thickness:4;

  /* Previous and next object in display list. */
  index prev, next;

  /* Geometric reference point for object. */
  CP_REC ref;

  /* Fields unique to object types. */
  union {

#   define DEF_OBJECT(TypeTag, Id, NonStdFieldStruct, Cmd, TextFlag, LTflag) \
	struct NonStdFieldStruct Id;
#   include "obj.def"
#   undef DEF_OBJECT

  } u;

} OBJECT_REC, *OBJECT;

typedef struct display_list *DISPLAY_LIST;
typedef void (*DISPLAY_LIST_ACTION_CODE)(DISPLAY_LIST, ENV);
typedef struct {
  ENV env;
  DISPLAY_LIST_ACTION_CODE code;
} DISPLAY_LIST_ACTION_CLOSURE_REC, *DISPLAY_LIST_ACTION_CLOSURE;

/* Display list of objects. */
typedef struct display_list {
  OBJECT blk[256];      /* block pointers for display list */
  int n_blks;           /* number of blocks currently allocated. */
  index head,           /* head of loop */
	free,           /* object free list */
	origin;         /* origin object or INULL if none */
  unsigned n_obj,       /* number of objects on list */
	n_picked;       /* number of objects picked */
  DISPLAY_LIST_ACTION_CLOSURE_REC change_action; /* code to call for changes */
} DISPLAY_LIST_REC;

#define DL_INIT(Code, Env)      { {0},0,INULL,INULL,INULL,0,0, { Env, Code } }

/* Iterate over display list.
   Args:
	P -- OBJECT var to be used in body
	L -- Display list
	H -- Head index, head or origin
	D -- Direction, prev or next
	B -- Body of loop. */

#ifndef NDEBUG

/* Version to verify display list count matches
   number of items on list. */
#define do_display_list_from(P, L, H, D, B) \
  { DISPLAY_LIST _dl = (L); \
    index _loop = _dl->H; \
    int _n = 0; \
    if (_loop != INULL) { \
      OBJECT P; \
      index _o = _loop; \
      do { \
	++_n; \
	P = deref(_dl, _o); B \
      } while ((_o = P->D) != _loop); \
      assert(_n == _dl->n_obj); } }

#else

#define do_display_list_from(P, L, H, D, B) \
  { DISPLAY_LIST _dl = (L); \
    index _loop = _dl->H; \
    if (_loop != INULL) { \
      OBJECT P; \
      index _o = _loop; \
      do { \
	P = deref(_dl, _o); B \
      } while ((_o = P->D) != _loop); } }

#endif

#define do_display_list(P, L, D, B) do_display_list_from(P, L, head, D, B)

/* Status flags for canvas. */
typedef enum {
  vENABLED,             /* Not currently used. */
  vBOTTOM_SCROLLBAR,
  vRIGHT_SCROLLBAR,
  vRULER,
  vAUTO_REDRAW,
} CANVAS_STATUS;

/* Canvas. */
typedef struct canvas {
  WINDOW_REC window;
  CC size,              /* dimension of canvas */
     vp_x_size;         /* x size of window in canvas coords */
  CP_REC vp_org;        /* origin of window in canvas coords */
  SP em_size,ex_size,   /* one em and ex size in TeX sp units for drawing text extent */
     unitlength;
  CANVAS_COLOR_VECTOR color;    /* Colors of things on canvas. */
  unsigned status;              /* flags */
  DISPLAY_LIST_REC dl;          /* display list */
  SCROLLBAR_REC x_scrollbar, 
		y_scrollbar;
  CURSOR_REC cursor;            /* cursor */
  WINDOW_REC draw;              /* drawing window */
  CP_REC pick0, pick1;          /* Point box diagonal. */
  CC pick_rad;                  /* Max radius of interest for point picks. */
  CP_REC cont_pos;              /* Continuation position for edits. */
  CP_REC copy_displacement;	/* Stored displacement for copy ops. */
} CANVAS_REC, *CANVAS;

#define DisplayListAction(Code, Env)    DL_INIT((DISPLAY_LIST_ACTION_CODE)Code, Env)
#define NullDisplayAction               DL_INIT((DISPLAY_LIST_ACTION_CODE)do_nothing, NULL)

#define DefCanvas(Name, Width, Height, Size, \
		  ViewPortSize, ViewPortX, ViewPortY, \
		  EmSize, ExSize, UnitLength, \
		  BackgroundColor, ThinColor, ThickColor, PickColor, RulerColor, \
		  ChangeAction, Flags) \
  CANVAS_REC Name[1] = {{ \
     /* window */ {0xffff,0,0,0,0,0,0,0,0,Width,Height,0,0,0,0}, \
     Size, \
     ViewPortSize, \
     { ViewPortX, ViewPortY }, \
     PT2SP(EmSize), PT2SP(ExSize), MM2SP(UnitLength), \
     { BackgroundColor, ThinColor, ThickColor, PickColor, RulerColor }, \
     Flags, ChangeAction }};

#define NO_CONT_POS     ((CC)0x8000)
#define cont_pos_valid_p(P)     ((P)->x != NO_CONT_POS)
#define set_cont_pos_invalid(P) ((P)->x = NO_CONT_POS)
#define NO_DISPLACEMENT ((CC)0x8000)
#define copy_displacement_valid_p(P)     ((P)->x != NO_DISPLACEMENT)
#define set_copy_displacement_invalid(P) ((P)->x = NO_DISPLACEMENT)
#define vp_y_size(C)    (scale_int((C)->vp_x_size - 1, (C)->draw.height - 1, (C)->draw.width - 1) + 1)
#define enable_auto_redraw(C)   ((C)->status |= bit(vAUTO_REDRAW))
#define disable_auto_redraw(C)  ((C)->status &= ~bit(vAUTO_REDRAW))
#define get_canvas_colors(C, V) (memcpy((V), (C)->color, sizeof(CANVAS_COLOR_VECTOR)))

/* Coordinate conversion: canvas <-> screen coordinates. */
int cc2sc(CANVAS c, CC xy);
CC sc2cc(CANVAS c, int xy);
int cc2scx(CANVAS c, CC x);
int cc2scy(CANVAS c, CC y);
CC sc2ccx(CANVAS c, int x);
CC sc2ccy(CANVAS c, int y);
#define sc2ccdx(C, X1, X0)      sc2cc(C, (X1) - (X0))
#define sc2ccdy(C, Y1, Y0)      sc2cc(C, (Y0) - (Y1))
extern char som2com_tbl[];
#define som2com(M)              (som2com_tbl[M])
#define com2som(M)              som2com(M)
CP_REC sp2cp(CANVAS c, int x, int y);

/* General. */
void swap_cp_recs(CP p1, CP p2);
void diagonalize(CP p0, CP p1);
OBJECT dereference(CANVAS c, index i);

typedef struct {
  CP_REC p0, p1;
  unsigned cursor_mask;
  int slope_num, slope_den;
  unsigned oval_mask;
} EDIT_INFO_REC, *EDIT_INFO;

/* Constructors. */
index make_origin(CANVAS c, CC x, CC y, CC width, CC height, int auto_p);
typedef index (LINEAR_CONSTRUCTOR)(CANVAS c, LINE_THICKNESS lt, CC x, CC y, CC len, int num, int den, CP p1_rtn);
LINEAR_CONSTRUCTOR make_line, make_vector;
typedef index (CIRCLE_CONSTRUCTOR)(CANVAS c, LINE_THICKNESS lt, CC x, CC y, CC rad);
CIRCLE_CONSTRUCTOR make_circle, make_fill_circle;
index make_oval(CANVAS c, LINE_THICKNESS lt, CC x, CC y, CC width, CC height, CC rad, unsigned mask);
typedef index (BOX_CONSTRUCTOR)(CANVAS c, LINE_THICKNESS lt, CC x, CC y, CC width, CC height, char *str, int hjust, int vjust, CC dash_len);
BOX_CONSTRUCTOR make_text, make_frame_box, make_dash_box, make_fill_box;
index make_text_edited_box(CANVAS c, index o, char *str, int hjust, int vjust, int picked_p);
index make_edited_object(CANVAS c, index o, EDIT_INFO info);

/* Canvas pick. */
typedef struct pick {
  index o;
  DIST dist;
} PICK_REC, *PICK;

/* Test used by do_point_pick to decide which
   objects to consider. */
typedef int (*PICK_PREDICATE)(OBJECT);

/* Use default yellow entry of palette for flashing pick color,
   default brown entry for flashing edit pick color. */
enum { cPICK = cYELLOW, cEDIT = cBROWN };

void open_canvas(CANVAS c, int x, int y, WINDOW parent);
void enable_canvas_ruler(CANVAS c);
void disable_canvas_ruler(CANVAS c);
void enable_flashing_pick(CANVAS c);
void disable_flashing_pick(void);
void set_canvas_colors(CANVAS c, CANVAS_COLOR_VECTOR colors);
int set_canvas_viewport(CANVAS c, 
			CC vp_size, CC vp_org_x, CC vp_org_y,
			int reset_scrollbars_p);
void locate_canvas_scrollbars(CANVAS c);
int do_point_pick(CANVAS c, PICK_PREDICATE elligible_p, PICK buf, int max_picks, int x, int y, int rad);
void do_rect_pick(CANVAS c, int pick_p, int x0, int y0, int x1, int y1);
void rect_pick(CANVAS c, int x0, int y0, int x1, int y1, int pick_p);
int get_picks(CANVAS c, index *obj, PICK_PREDICATE elligible_p);
unsigned n_picked(CANVAS c, int include_origin_p);
void pick_all(CANVAS c, int pick_p);
void set_canvas_unitlength(CANVAS c, SP unitlength);
void copy_objects(CANVAS c, index *from, index *to, int n, CC dx, CC dy);
void add_objects(CANVAS c, index *o, int n, index *old_origin_rtn);
void move_objects(CANVAS c, index *o, int n, CP_REC d);
void delete_objects(CANVAS c, index *o, int n);
void destroy_objects(CANVAS c, index *o, int n);
void set_object_line_thickness(CANVAS c, index *o_vec, int n, LINE_THICKNESS lt);
void edit_info(CANVAS c, index o, CP_REC pick, EDIT_INFO info);
void clear_canvas(CANVAS c);
int write_objects(CANVAS c, char *file_name, SETTINGS settings, int save_mem_p);
int text_is_latext(char *text);
int read_objects(CANVAS c, char *file_name, CC oval_rad, SETTINGS settings);
void pick_objects(CANVAS c, index *o, int n, int on_p);
void edit_pick_object(CANVAS c, index o, int on_p);
int mark_outside(CANVAS c, index *o_vec, int n, CP_REC org, CP_REC size);
int unpick_outside(CANVAS c, index *o_vec, int n);

#define v_status_p(V, S)        ((V)->status & bit(S))
#define BottomScrollbar bit(vBOTTOM_SCROLLBAR)
#define RightScrollbar  bit(vRIGHT_SCROLLBAR)
#define NoChangeAction  ((DISPLAY_LIST_CHANGE_CODE)do_nothing)

/*  ----- Commands -------------------------------------------------- */

typedef enum {
  cNOOP,
  cADD,
  cADD_AND_MOVE_CONT_POS,
  cADD_AND_UNPICK,
  cDELETE,
  cDELETE_AND_PICK,
  cMOVE,
  cREPLACE,
  cMAKE_THICK,
  cMAKE_THIN,
} COMMAND_TYPE;

/* Predicate for commands that store two mirror sets of objects. */
#define binary_command_type_p(T)                \
	     (bit(T) & (bit(cREPLACE)|          \
			bit(cADD_AND_UNPICK)|   \
			bit(cDELETE_AND_PICK)))

/* Portion of user environment command can affect. */
typedef struct {
  CP_REC cont_pt,       /* current and old continuation points */
	 old_cont_pt;
} ENV_STATE_REC;

typedef struct {
  COMMAND_TYPE type;    /* type of command */
  ENV_STATE_REC env;    /* user environment state */
  CP_REC move_d;        /* amount to move */
  CP_REC cont_pos_from, 
	 cont_pos_to;   /* Change continuation position from/to. */
  index other_origin;   /* Deleted origin for add and added one for delete. */
  CANVAS canvas;        /* canvas command is for */
  int n_obj;            /* number of objects */
  index obj[1];         /* vector of objects affected (must be last) */
} COMMAND_REC, *COMMAND;

COMMAND cmd_alloc(COMMAND_TYPE type, CANVAS canvas, unsigned n_obj, ... );
COMMAND cmd_realloc(COMMAND cmd, unsigned n_obj);
void destroy_command(COMMAND cmd);
void neg(int *i);
void swap(int *x, int *y);
char* cp2str(CP_REC p);
char* cc2str(CC c);
void exec_command(COMMAND cmd);
void undo_last_command(void);
void clear_undo(void);
int string(FILE *f, char *str);
void force_canvas_change(CANVAS c);
