/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.engine.python.ui.handlers;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.lsat.common.emf.common.util.URIHelper;
import org.eclipse.lsat.common.emf.ecore.resource.Persistor;
import org.eclipse.lsat.common.emf.ecore.resource.PersistorFactory;
import org.eclipse.lsat.common.emf.ecore.resource.ResourceSetUtil;
import org.eclipse.lsat.common.graph.directed.Node;
import org.eclipse.lsat.common.graph.directed.util.DirectedGraphQueries;
import org.eclipse.lsat.common.qvto.util.QvtoTransformationException;
import org.eclipse.lsat.common.scheduler.algorithm.BellmanFordScheduler;
import org.eclipse.lsat.common.scheduler.algorithm.SchedulerException;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.graph.TaskDependencyGraph;
import org.eclipse.lsat.common.scheduler.schedule.Schedule;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.engine.python.Activator;
import org.eclipse.lsat.engine.python.generator.GenerateActivityDispatching;
import org.eclipse.lsat.motioncalculator.MotionException;
import org.eclipse.lsat.scheduler.AddExecutionTimes;
import org.eclipse.lsat.scheduler.AnnotateClaimRelease;
import org.eclipse.lsat.scheduler.CleanupGraph;
import org.eclipse.lsat.scheduler.CleanupGraph.RemoveClaimReleaseStrategy;
import org.eclipse.lsat.scheduler.CleanupSchedule;
import org.eclipse.lsat.scheduler.Dispatching2Graph;
import org.eclipse.lsat.scheduler.Dispatching2GraphOutput;
import org.eclipse.lsat.scheduler.RemoveEvents;
import org.eclipse.lsat.timing.calculator.MotionCalculatorExtension;
import org.eclipse.lsat.timing.util.SpecificationException;
import org.eclipse.lsat.timing.util.TimingCalculator;

import activity.ActivitySet;
import activity.util.Event2Resource;
import common.util.ImportsFlattener;
import dispatching.ActivityDispatching;
import dispatching.util.DispatchingUtil;
import setting.SettingUtil;
import setting.Settings;

public class ExecutionEngineGeneratorJob extends Job {
    private final IFile modelIFile;

    private final File saveFolder;

    private final boolean removeClaimAndReleases;

    private final boolean simulate;

    public ExecutionEngineGeneratorJob(IFile modelIFile, File saveFolder, boolean removeClaimAndReleases,
            boolean simulate)
    {
        super("TwinscanCodeGenJob");
        this.modelIFile = modelIFile;
        this.saveFolder = saveFolder;
        this.removeClaimAndReleases = removeClaimAndReleases;
        this.simulate = simulate;
    }

    @Override
    protected IStatus run(IProgressMonitor monitor) {
        try {
            doGenerate(monitor);
            modelIFile.getProject().refreshLocal(IFile.DEPTH_INFINITE, monitor);
            return Status.OK_STATUS;
        } catch (Exception e) {
            return new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e);
        }
    }

    public File doGenerate(IProgressMonitor monitor) throws QvtoTransformationException, MotionException, IOException,
            CoreException, SpecificationException, SchedulerException
    {
        try {
            monitor.beginTask("Generating TWINSCAN code", 100);

            PersistorFactory factory = new PersistorFactory();
            URI modelURI = URIHelper.asURI(modelIFile);

            monitor.subTask("Preparing specification for code generation");

            Persistor<ActivityDispatching> dispatchingPersistor = factory.getPersistor(ActivityDispatching.class);
            ActivityDispatching activityDispatching = dispatchingPersistor.loadOne(modelURI);
            Settings settings = SettingUtil.getSettings(activityDispatching.eResource());

            // Make sure all resources are loaded before processing
            EcoreUtil.resolveAll(factory.getResourceSet());

            // See TimingAnalysisLaunchDelegate#launch(...)
            // flatten the import containers to a single root per type and grab them again
            ResourceSet resourceSet = ImportsFlattener.flatten(modelURI, activityDispatching, settings);
            activityDispatching = ResourceSetUtil.getObjectByType(resourceSet, ActivityDispatching.class);
            settings = ResourceSetUtil.getObjectByType(resourceSet, Settings.class);

            // convert event to resources in loaded model
            ActivitySet activitySet = ResourceSetUtil.getObjectByType(resourceSet, ActivitySet.class);
            // important the next line must be done before expanding the activities!!
            Event2Resource.surroundEventsWithClaimRelease(activitySet);
            DispatchingUtil.expand(activityDispatching);
            DispatchingUtil.removeUnusedActivities(activityDispatching);

            // See TimingAnalysisLaunchDelegate#generateGraphFromActivityDispatching(...)
            Dispatching2Graph<Task> d2g = new Dispatching2Graph<Task>();
            Dispatching2GraphOutput<Task> d2gResult = d2g.transformModel(activityDispatching, monitor);

            TaskDependencyGraph<Task> taskDependencyGraph = d2gResult.getTaskDependencyGraph();
            taskDependencyGraph = RemoveEvents.transformModel(taskDependencyGraph);

            CleanupGraph<Task> cleanupGraph = new CleanupGraph<Task>(false, RemoveClaimReleaseStrategy.KeepAll);
            TaskDependencyGraph<Task> graph = cleanupGraph.transformModel(taskDependencyGraph, monitor);

            // See TimingAnalysisLaunchDelegate#annotateClaimRelease(...)
            graph = AnnotateClaimRelease.transformModel(graph);

            // See TimingAnalysisLaunchDelegate#refineGraphWithExecutionTimes(...)
            MotionCalculatorExtension motionCalculator = MotionCalculatorExtension.getSelectedMotionCalculator();
            TimingCalculator timingCalculator = new TimingCalculator(settings, motionCalculator);
            AddExecutionTimes addExecutionTimes = new AddExecutionTimes(timingCalculator);
            graph = addExecutionTimes.transformModel(graph);

            // See TimingAnalysisLaunchDelegate#scheduleGraphOnResources(...)
            BellmanFordScheduler<Task> scheduler = new BellmanFordScheduler<Task>();
            Schedule<Task> schedule = scheduler.createSchedule(graph);
            schedule.setName(URIHelper.baseName(modelIFile));

            if (removeClaimAndReleases) {
                CleanupSchedule<Task> cleanupSchedule = new CleanupSchedule<Task>(CleanupSchedule.ClaimReleaseStrategy.RemoveCompletely);
                schedule = cleanupSchedule.transformModel(schedule, null);
            }

            monitor.worked(50);
            monitor.subTask("Generating code");

            List<ScheduledTask<Task>> tasks = DirectedGraphQueries.reverseTopologicalOrdering(
                    DirectedGraphQueries.allSubNodes(schedule), Comparator.comparing(Node::getName));
            schedule.getNodes().clear();
            schedule.getNodes().addAll(tasks);

            GenerateActivityDispatching codeGenerator = new GenerateActivityDispatching(schedule, saveFolder,
                    Arrays.asList(simulate, settings));
            codeGenerator.generate(BasicMonitor.toMonitor(monitor));

            monitor.worked(50);
            return new File(saveFolder, graph.getName() + ".py");
        } finally {
            monitor.done();
        }
    }
}
