/* wptKeylist.cpp - Keylist element
 *	Copyright (C) 2001-2004 Timo Schulz
 *	Copyright (C) 2004 Andreas Jobs
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <commctrl.h>
#include <time.h>

#include "wptCommonCtl.h"
#include "wptTypes.h"
#include "wptGPG.h"
#include "wptKeylist.h"
#include "wptKeyManager.h"
#include "wptW32API.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptUTF8.h"
#include "wptRegistry.h"

static struct listview_column_s klist_enc[] = {
    {0, 242, (char *)_("User ID")},
    {1, 80, (char *)_("Key ID")},
    {3, 46, (char *)_("Size")},
    {4, 50, (char *)_("Cipher")},
    {5, 70, (char *)_("Validity")},
    {0, 0, NULL}
};
#define KLIST_ENC_ITEMS (DIM(klist_enc) -1)

static struct listview_column_s klist[] = {
    {0, 242, (char *)_("User ID")},
    {1, 78, (char *)_("Key ID")},
    {2, 52, (char *)_("Type")},    
    {3, 68, (char *)_("Size")},
    {4, 66, (char *)_("Cipher")},
    {5, 70, (char *)_("Validity")},
    {6, 40, (char *)_("Trust")},
    {7, 72, (char *) _("Creation")},
    {0, 0, NULL}
};
#define KLIST_ITEMS (DIM(klist) - 1)

struct key_array_s {
    char keyid[32];
    int checked;
};

static int keylist_add_key( listview_ctrl_t lv, int mode, gpgme_key_t key );
static int find_secret_key( gpgme_key_t key );


static key_array_s*
key_array_new( size_t items )
{
    key_array_s *ka;
    size_t j;
    
    if( items == 0 )
	return NULL;
    ka = new key_array_s[items + 1];
    if( ka == NULL )
	return NULL;	
    for ( j = 0; j < items; j++ )
	ka[j].checked = 0;
    return ka;
} /* key_array_new */


static void
key_array_release( key_array_s *ka )
{
    free_if_alloc( ka );
} /* key_array_release */


static int
key_array_search( key_array_s *ka, size_t items, const char *keyid )
{
    size_t j;
    
    /* fixme: we need a faster search method */
    for( j = 0; j < items; j++ ) {
	if( !strcmp( keyid, ka[j].keyid ) )
	    return 1;	
    }
    
    return 0;
} /* key_array_search */


const char*
get_key_algo( gpgme_key_t key, int keyidx )
{
    static char algo_id[128];
    int n, algo_main, algo_sub;
    
    if( keyidx > 0 ) {
	algo_main = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, keyidx-1 );
	_snprintf( algo_id, sizeof algo_id-1, "%s", 
	    gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo_main ) );
	return algo_id;
    }
    algo_main = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, 0 );
    n = gpgme_key_count_items( key, GPGME_ATTR_KEYID );
    if( n > 1 ) {
	algo_sub = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, n-1 );
	_snprintf( algo_id, sizeof algo_id - 1, "%s/%s",
	    gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo_main ),
	    gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo_sub ) );
	return algo_id;
    }
    return gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo_main );
} /* get_key_algo */


const char*
get_key_created( long timestamp )
{
    static char timebuf[128];
    struct tm *warp;

    if( timestamp == 0 || timestamp == -1 )
	return "????-??-??";
    warp = localtime( &timestamp );
    _snprintf( timebuf, sizeof timebuf - 1, "%04d-%02d-%02d",
		warp->tm_year + 1900, warp->tm_mon + 1, warp->tm_mday );
    return timebuf;
} /* get_key_created */


const char*
get_key_expire_date( long timestamp )
{
    static char timebuf[64];
    struct tm *warp;

    if( !timestamp )
	return _("Never");
    warp = localtime( &timestamp );
    _snprintf( timebuf, sizeof timebuf -1, "%04d-%02d-%02d",
		warp->tm_year + 1900, warp->tm_mon + 1, warp->tm_mday );
    return timebuf;
} /* get_key_expire_date */


const char *
get_key_type( gpgme_key_t key )
{
    int valid = gpgme_key_get_ulong_attr( key, GPGME_ATTR_VALIDITY, NULL, 0 );
    if( find_secret_key( key ) || valid == GPGME_VALIDITY_ULTIMATE )	
	return _("Key Pair");
    return _("Public Key");
} /* get_key_type */


const char *
get_key_size( gpgme_key_t key, int keyidx )
{
    static char size_id[64];
    int n, size_main, size_sub;
    
    if( keyidx > 0 ) {
	size_main = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, keyidx-1 );
	_snprintf( size_id, sizeof size_id-1, "%d", size_main );
	return size_id;
    }
    size_main = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, 0 );
    n = gpgme_key_count_items( key, GPGME_ATTR_KEYID );
    if( n > 1 ) {    
	size_sub = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, n-1 );
	_snprintf( size_id, sizeof size_id - 1, "%d/%d", size_main, size_sub );
	return size_id;	
    }
    _snprintf( size_id, sizeof size_id - 1, "%d", size_main );
    return size_id;
} /* get_key_size */


const char *
get_key_fpr( gpgme_key_t key )
{
    static char fpr_md[64];
    const char * fpr;
    char t[16];
    size_t i;
    
    memset( fpr_md, 0, sizeof fpr_md );
    fpr = gpgme_key_get_string_attr( key, GPGME_ATTR_FPR, NULL, 0 );
    for( i = 0; i < strlen( fpr ) / 4; i++ ) {
	sprintf( t, "%c%c%c%c ", fpr[4*i], fpr[4*i+1], fpr[4*i+2], fpr[4*i+3] );
	strcat( fpr_md, t );
    }
    return fpr_md;
} /* get_key_fpr */



static const char *
get_key_trust2 (gpgme_key_t key, int val, int uididx, int listmode)
{
    if (key)
	val = gpgme_key_get_ulong_attr (key, GPGME_ATTR_OTRUST, NULL, uididx);
    switch (val) {
    case GPGME_TRUST_UNKNOWN:
    case GPGME_TRUST_DONTKNOW:    
	return "None";
    case GPGME_TRUST_NEVER:    
	return "Never";
    case GPGME_TRUST_MARGINAL:
	return "Marginal";
    case GPGME_TRUST_FULLY:
    case GPGME_TRUST_ULTIMATE:
	return "Full";
    }
    return "";
}


const char *
get_key_trust (gpgme_key_t key, int uididx, int listmode)
{
    return get_key_trust2 (key, 0, uididx, listmode);
}


const char *
get_key_trust_str (int val)
{
    return get_key_trust2 (NULL, val, 0, 0);
}


char*
get_key_status( gpgme_key_t key, int uididx, int listmode )
{
    char fmt[128], * p;
    const char * attr;
    int i = 0;
    u32 key_attr =0;

    if( uididx < 0 || gpgme_key_count_items( key, GPGME_ATTR_USERID ) > uididx )
	uididx = 0;
    memset( fmt, 0, sizeof fmt );
    if( listmode ) {
	fmt[i++] = '[';
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_REVOKED, NULL, 0 ) )
	    fmt[i++] = 'R';	
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_EXPIRED, NULL, 0 ) )
	    fmt[i++] = 'E';	
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_DISABLED, NULL, 0 ) )	
	    fmt[i++] = 'D';
	fmt[i++] = ']';
	fmt[i++] = ' ';
    }
    key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_VALIDITY, NULL, uididx );
    attr = gpgme_key_expand_attr( GPGME_ATTR_VALIDITY, key_attr );
    p = new char[strlen( fmt ) + 1 + strlen( attr ) + 2];
    sprintf( p, "%s%s", fmt, attr );
    return p;
} /* get_key_status */


static inline int
int_cmp( int a, int b )
{
    if( a == b ) return 0;	
    else if( a > b ) return 1;
    else return -1;
    return 0;
}


static int CALLBACK
keylist_cmp_cb (LPARAM first, LPARAM second, LPARAM sortby)
{
    static char tmpa[128], tmpb[128];    
    gpgme_key_t a, b;    
    const char *aa = NULL, *bb = NULL;
    long ta, tb;
    int na = 0, nb = 0;
    int cmpresult = 0;
    
    a = (gpgme_key_t)first;
    b = (gpgme_key_t)second;    
    
    switch( sortby & ~KEYLIST_SORT_DESC ) {
    case GPGME_ATTR_USERID:
	aa = gpgme_key_get_string_attr( a, GPGME_ATTR_USERID, NULL, 0 );
	bb = gpgme_key_get_string_attr( b, GPGME_ATTR_USERID, NULL, 0 );
        cmpresult = strcmpi( aa? aa : "", bb? bb : "" );
	break;
        
    case GPGME_ATTR_KEYID:
	aa = gpgme_key_get_string_attr( a, GPGME_ATTR_KEYID, NULL, 0) + 8;
	bb = gpgme_key_get_string_attr( b, GPGME_ATTR_KEYID, NULL, 0) + 8;
	cmpresult = strcmpi( aa? aa : "", bb? bb : "" );
	break;

    case GPGME_ATTR_VALIDITY:
	na = gpgme_key_get_ulong_attr( a, GPGME_ATTR_VALIDITY, NULL, 0 );
	nb = gpgme_key_get_ulong_attr( b, GPGME_ATTR_VALIDITY, NULL, 0 );	
	cmpresult = int_cmp( na, nb );
        break;

    case GPGME_ATTR_OTRUST:
	na = gpgme_key_get_ulong_attr (a, GPGME_ATTR_OTRUST, NULL, 0);
	nb = gpgme_key_get_ulong_attr (b, GPGME_ATTR_OTRUST, NULL, 0);
	cmpresult = int_cmp (na, nb);
	break;

    case GPGME_ATTR_IS_SECRET:	
	aa = gpgme_key_get_string_attr( a, GPGME_ATTR_KEYID, NULL, 0 );
	bb = gpgme_key_get_string_attr( b, GPGME_ATTR_KEYID, NULL, 0 );
	get_seckey( aa, &a );
	get_seckey( bb, &b );
	if( a )
	    na = gpgme_key_get_ulong_attr( a, GPGME_ATTR_IS_SECRET, NULL, 0 );
	if( b )
	    nb = gpgme_key_get_ulong_attr( b, GPGME_ATTR_IS_SECRET, NULL, 0 );
	cmpresult = int_cmp( na, nb );
	break;

    case GPGME_ATTR_LEN:
	na = gpgme_key_get_ulong_attr( a, GPGME_ATTR_LEN, NULL, 0 );
	nb = gpgme_key_get_ulong_attr( b, GPGME_ATTR_LEN, NULL, 0 );
	cmpresult = int_cmp( na, nb );
        break;

    case GPGME_ATTR_CREATED:
	ta = gpgme_key_get_ulong_attr( a, GPGME_ATTR_CREATED, NULL, 0 );
	tb = gpgme_key_get_ulong_attr( b, GPGME_ATTR_CREATED, NULL, 0 );
	strcpy( tmpa, get_key_created( ta ) ); aa = tmpa;
	strcpy( tmpb, get_key_created( tb ) ); bb = tmpb;
	cmpresult = strcmpi( aa? aa : "", bb? bb : "" );
	break;

    case GPGME_ATTR_ALGO:
	aa = gpgme_key_get_string_attr (a, GPGME_ATTR_ALGO, NULL, 0);
	bb = gpgme_key_get_string_attr (b, GPGME_ATTR_ALGO, NULL, 0);
	cmpresult = strcmpi (aa? aa : "", bb? bb : "");
	break;
        
    default:
	aa = gpgme_key_get_string_attr( a, GPGME_ATTR_USERID, NULL, 0 );
	bb = gpgme_key_get_string_attr( b, GPGME_ATTR_USERID, NULL, 0 );
	cmpresult = strcmpi( aa? aa : "", bb? bb : "" );
	break;
    }
    if (sortby & KEYLIST_SORT_DESC)
        return (~cmpresult + 1);
    else
        return cmpresult;
} /* keylist_cmp_cb */


static const char*
calc_validity( gpg_group_t grp )
{
    int level = 0, valid;
    gpg_member_t mbr;
    gpgme_key_t key;

    for( mbr = grp->list; mbr; mbr = mbr->next ) {
	if( get_pubkey( mbr->name, &key ) )
	    continue;
	valid = gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_VALIDITY, NULL, 0 );
	switch( valid ) {	
	case GPGME_VALIDITY_MARGINAL:
	case GPGME_VALIDITY_NEVER:
	case GPGME_VALIDITY_UNDEFINED:
	    return gpgme_key_expand_attr( GPGME_ATTR_VALIDITY, valid );
	}
    }
    return gpgme_key_expand_attr( GPGME_ATTR_VALIDITY, GPGME_VALIDITY_FULL );
} /* calc_validity */


int
keylist_add_groups( listview_ctrl_t lv )
{
    gpg_optfile_t gh;
    gpg_group_t grp;
    const char *valid;

    gh = km_groupdb_open( );    
    if( !gh )	
	return WPTERR_FILE_OPEN;

    for( grp = gh->grp; grp; grp = grp->next ) {
	valid = calc_validity( grp );
	listview_add_item( lv, " " );	
	listview_add_sub_item( lv, 0, 0, grp->name );	
	listview_add_sub_item( lv, 0, 1, "gpg_group_t" );	
	listview_add_sub_item( lv, 0, 2, "" );	
	listview_add_sub_item( lv, 0, 3, "Unknown" );
	listview_add_sub_item( lv, 0, 4, valid?valid : "Unknown" );
    }
    return 0;
} /* keylist_add_groups */


static int
keylist_build( listview_ctrl_t *r_lv, HWND ctrl, int mode )
{
    listview_ctrl_t lv;
    listview_column_t col;
    int j, n = 0;
    int kl_nolist = 0, rc = 0;
    
    rc = listview_new( &lv );
    if( rc )
	return rc;
    
    lv->ctrl = ctrl;
    if( (mode & KEYLIST_ENCRYPT) || (mode & KEYLIST_ENCRYPT_MIN) ) {
	col = klist_enc;
	n = KLIST_ENC_ITEMS;
    }	
    else if( (mode & KEYLIST_SIGN) ) {
	col = klist_enc;
	n = KLIST_ENC_ITEMS - 1;
    }
    else {
	col = klist;
	n = KLIST_ITEMS;
    }
    
    for( j = 0; j < n; j++ )	
	listview_add_column( lv, &col[j] );    
    listview_set_ext_style( lv );
    *r_lv = lv;
    
    return 0;
} /* keylist_build */


static void
keylist_load_keycache( listview_ctrl_t lv, int mode, gpgme_keycache_t pubkc,
		       gpgme_keycache_t seckc )
{
    gpgme_error_t err = GPGME_No_Error;
    gpgme_key_t key, skey;
    const char * keyid;

    if( pubkc && seckc ) {
	gpgme_keycache_rewind( pubkc );
        while( !gpgme_keycache_next_key( pubkc, 0, &key ) ) {
            keyid = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
            if( keyid && !gpgme_keycache_find_key( seckc, keyid, 0, &skey ) )
		keylist_add_key( lv, mode, key );            
        }	
    }
    else if( pubkc ) {
	gpgme_keycache_rewind( pubkc );
	while( !err ) {    
	    err = gpgme_keycache_next_key( pubkc, 0, &key );
	    if( !err )
		keylist_add_key( lv, mode, key );
	}
    }
} /* keylist_load_keycache */


listview_ctrl_t
keylist_load( HWND ctrl, gpgme_keycache_t pubkc, gpgme_keycache_t seckc, 
	      int mode, int sortby )
{    
    listview_ctrl_t lv;
    int rc = 0;

    rc = keylist_build( &lv, ctrl, mode );
    if( rc )
	return NULL;    	
    keylist_load_keycache( lv, mode, pubkc, seckc );
    keylist_sort( lv, sortby );
    if( (mode & KEYLIST_ENCRYPT) || (mode & KEYLIST_ENCRYPT_MIN) )
	keylist_add_groups( lv );
    return lv;
} /* keylist_load */


int
keylist_reload( listview_ctrl_t lv, gpgme_keycache_t pubkc, int mode, int sortby )
{
    listview_del_all( lv );
    keylist_load_keycache( lv, mode, pubkc, NULL );
    keylist_sort( lv, sortby );
    return 0;
} /* keylist_reload */


void
keylist_delete( listview_ctrl_t lv )
{
    if( lv ) {
	listview_release( lv );
    }
} /* keylist_delete */


static int
find_secret_key( gpgme_key_t key )
{
    const char * keyid;
    gpgme_key_t skey;   

    keyid = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
    if( !keyid )
	return 0;
    get_seckey( keyid, &skey );
    return skey? 1 : 0;
} /* find_secret_key */


static int
do_addkey( listview_ctrl_t lv, gpgme_key_t key, int uididx, int keyidx, int list )
{    
    LV_ITEM lvi;
    gpgme_key_t seckey;
    char fmt[128];
    const char *attr;
    u32 key_attr;
    int idx = 0;

    /* we check the pubkey algorithm here to make sure that no ElGamal
       sign+encrypt key is used in _any_ mode */
    if( list != 1 && gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, 0 )
		 == GPGME_PK_ELG_ES )
	return 0;
        
    if( listview_add_item( lv, " " ) )	
	return WPTERR_GENERAL;
	
    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
    memset( &lvi, 0, sizeof lvi );
    lvi.mask = LVIF_TEXT | LVIF_PARAM;
    lvi.pszText = (char *)attr;
    lvi.lParam = (LPARAM )key;
    if( ListView_SetItem( lv->ctrl, &lvi ) == FALSE )	
	return WPTERR_GENERAL;
        
    if( uididx == -1 ) {
	/* request the primary user-id of the key. */
	attr = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
	uididx = 0;
    }
    else {
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_UID_REVOKED, NULL, uididx ) || uididx < 0 )
	    uididx = 0; /* fixme: this happen sometimes but it's illegal! (<0) */
	attr = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, uididx );
    }
    if( attr == NULL || strlen( attr ) < 5 ) { /* normal userids are >= 5 chars */
	attr = _("Invalid User ID");
	listview_add_sub_item( lv, 0, idx++, attr );
    }	
    else {
	char * uid = utf8_to_wincp (attr, strlen (attr));
	if( uid ) {	 
	    listview_add_sub_item( lv, 0, idx++, uid );
	    free( uid );
	}
    }
    attr = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, keyidx );
    if( attr ) {
	_snprintf( fmt, sizeof fmt -1, "0x%s", attr + 8 );
	listview_add_sub_item( lv, 0, idx++, fmt );
    }
	
    if( list > 0 ) {
	attr = find_secret_key( key )? "pub/sec" : "pub";
	if( strchr( attr, '/' ) ) {
	    const char * kid;
	    kid = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
	    get_seckey( kid, &seckey );
	    if( gpgme_key_get_ulong_attr( seckey, GPGME_ATTR_DIVERT_CARD, NULL, 0 ) )
		attr = "pub/crd";
	}
	listview_add_sub_item( lv, 0, idx++, attr );
    }
    if( lv->cols >= 2 ) {
	attr = get_key_size( key, list==-1? keyidx+1 : 0 );
	if( attr )
	    listview_add_sub_item( lv, 0, idx++, attr );	
    }
    if( lv->cols >= 3 ) {
	attr = get_key_algo( key, list==-1? keyidx+1 : 0 );
	if( attr ) 
	    listview_add_sub_item( lv, 0, idx++, attr );
    }
    if( lv->cols >= 4 ) {
	char * status = get_key_status( key, uididx, list>0? 1 : 0 );
	if( !status )
	    return WPTERR_GENERAL;
	listview_add_sub_item( lv, 0, idx++, status );
	free_if_alloc( status );
    }
    if (lv->cols >= 5) {
	const char * s = get_key_trust (key, uididx, list>0? 1 : 0);
	listview_add_sub_item (lv, 0, idx++, s);
    }
    if( lv->cols >= 6 ) {
	key_attr = gpgme_key_get_ulong_attr( key, GPGME_ATTR_CREATED, NULL, keyidx );
	if( key_attr ) {
	    attr = gpgme_key_expand_attr( GPGME_ATTR_CREATED, key_attr );
	    listview_add_sub_item( lv, 0, idx++, attr );
	}	
    }

    return 0;
} /* do_addkey */


static int
keylist_add_key (listview_ctrl_t lv, int mode, gpgme_key_t key)
{
    int uids, rc = 0, i, n = 0;
        
    for ( i = 0; i < gpgme_key_count_items( key, GPGME_ATTR_KEYID ); i++ ) {
	if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_INVALID, NULL, i ) )	
	    continue; /* Don't use invalid keys */
		
	if( mode & KEYLIST_ALL ) {
	    uids = gpgme_key_count_items( key, GPGME_ATTR_USERID );
	    rc = do_addkey( lv, key, uids, i, 0 );
	    if( rc )
		return rc;
	}
	else if( mode & KEYLIST_LIST )	    
	    return do_addkey( lv, key, -1, i, 1 );
	else if (mode & KEYLIST_ENCRYPT) 
	{
	    if (gpgme_key_get_cability (key, GPGME_ATTR_CAN_ENCRYPT, i)
		&& gpgme_key_get_ulong_attr (key, GPGME_ATTR_KEY_USABLE, NULL, i))
	    {
		if (mode & KEYLIST_FLAG_FILE ) {
		    rc = do_addkey( lv, key, -1, i, -1 );
		    if( rc )
			return rc;
		}
		else {
		    for( uids = 0;  uids < gpgme_key_count_items( key, GPGME_ATTR_USERID ); uids++ ) {
			rc = do_addkey( lv, key, uids, i, -1 );
			if( rc )
			    return rc;
		    }
		}
	    }
	}
	else if (mode & KEYLIST_ENCRYPT_MIN)
	{
	    if( gpgme_key_get_cability (key, GPGME_ATTR_CAN_ENCRYPT, i)
		&& gpgme_key_get_ulong_attr (key, GPGME_ATTR_KEY_USABLE, NULL, i))
	    {
		rc = do_addkey (lv, key, -1, i, -1);
		return rc;
	    }
	}	
	else if( mode & KEYLIST_SIGN ) {
	    if ( gpgme_key_get_cability( key, GPGME_ATTR_CAN_SIGN, i )
		&& find_secret_key( key )
		&& gpgme_key_get_ulong_attr (key, GPGME_ATTR_KEY_USABLE, NULL, i))
	    {
		rc = do_addkey (lv, key, -1, i, -1);
		if( rc )
		    return rc;	
	    }
	}	
    }

    return rc;	
} /* keylist_add_key */


int
keylist_sort (listview_ctrl_t lv, int sortby)
{	
    return listview_sort_items( lv, sortby, keylist_cmp_cb );
} /* keylist_sort */


static int
key_check_validity (const char *validity) 
{    
    if (strstr (validity, "Unknown") 
	|| strstr (validity, "Undefined") 
	|| strstr (validity, "Never"))
	return 0;  
    return 1;
} /* key_check_validity */


gpgme_recipients_t
keylist_get_recipients (listview_ctrl_t lv, int *r_force_trust, int *r_count)
{
    int count = 0, force_trust = 0;
    int n, j, ka_pos = 0, rc = 0;
    char keyid[32], valid[32], id[100];
    key_array_s *ka = NULL;
    gpgme_error_t err;
    gpgme_recipients_t rset;

    err = gpgme_recipients_new( &rset );
    if( err )
	BUG( NULL );
    
    n = listview_count_items( lv, 0 );
    ka = key_array_new( n );
    if ( !ka )
	BUG( NULL );
	
    for( j = 0; j < n; j++ ) {
	if( listview_get_item_state( lv, j ) || n == 1 ) {
	    listview_get_item_text( lv, j, 1, keyid, sizeof keyid - 1 );			
	    listview_get_item_text( lv, j, 4, valid, sizeof valid -1 );
	    listview_get_item_text( lv, j, 0, id, sizeof id-1 );
	    if( !strncmp( keyid, "gpg_group_t", 5 ) ) {
		listview_get_item_text( lv, j, 0, id, sizeof id -1 );
		rc = km_groupdb_expand_recipients( id, rset );
		if( rc )
		    force_trust++;
	    }
	    else if( !key_check_validity( valid ) 
		    && !key_array_search( ka, ka_pos, keyid ) ) {
		char *warn = new char[512+strlen (id) + 1];
		if (!warn)
		    BUG (0);
		sprintf (warn,
		    _("It is NOT certain that the key belongs to the person\n"
		      "named in the user ID.  If you *really* know what you are\n"
		      "doing, you may answer the next question with yes\n"
		      "\n"
		      "Use \"%s\" anyway?"), id);
		if( reg_prefs.always_trust )
		    rc = IDYES;
		else
		    rc = msg_box (NULL, warn, _("Recipients"), MB_ERR_ASK);
		if (rc == IDYES)
		{
		    gpgme_recipients_add_name_with_validity (rset, keyid, GPGME_VALIDITY_FULL);
		    force_trust++;
		    ka[ka_pos].checked = 1;
		    strcpy (ka[ka_pos++].keyid, keyid);
		}
		free_if_alloc (warn);
	    }
	    else {
		listview_get_item_text( lv, j, 1, keyid, sizeof keyid -1 );
		gpgme_recipients_add_name( rset, keyid );
		count++;	
	    }
	}
    }
    key_array_release( ka );
    if( r_force_trust )
	*r_force_trust = force_trust;
    if( r_count )
	*r_count = count;
    return rset;
} /* keylist_get_recipients */


static int
keylist_get_keyflags (const char *buf, size_t buflen)
{
    int c = 0, flags = 0;

    if( *buf != '[' )
	return KEYFLAG_NONE;
    while (buf && c != ']') 
    {
	c = *buf++;
	if (c == 'R')
	    flags |= KEYFLAG_REVOKED;
	if (c == 'E')
	    flags |= KEYFLAG_EXPIRED;
	if (c == 'D')
	    flags |= KEYFLAG_DISABLED;
    }

    return flags;
} /* keylist_get_keyflags */


gpgme_recipients_t
keylist_enum_recipients (listview_ctrl_t lv,  int listype)
{
    gpgme_recipients_t rset;
    int i, n, id;
    char keyid[32], t[128], t2[128];

    if( gpgme_recipients_new( &rset ) )
	BUG( NULL );

    n = listview_count_items( lv, 0 );
    for( i = 0; i < n; i++ ) {
	if( !listview_get_item_state( lv, i ) )
	    continue;
	listview_get_item_text( lv, i, 1, keyid, sizeof keyid - 1 );
	switch( listype ) {
	case KEYLIST_LIST:
	    listview_get_item_text( lv, i, 5, t, sizeof t - 1 );
	    if( keylist_get_keyflags( t, strlen( t ) ) & KEYFLAG_REVOKED ) {
		_snprintf( t2, sizeof t2 -1, 
			    _("KeyID %s.\nDo you really want to export a revoked key?"), keyid );
		id = msg_box( lv->ctrl, t2, _("Recipients"), MB_INFO|MB_YESNO );
		if( id == IDNO )
		    continue;		
	    }
	    break;
	}
	gpgme_recipients_add_name( rset, keyid );
    }
    return rset;
} /* keylist_enum_recipients */


void
seclist_destroy (keylist_t * list)
{
    keylist_t l2;
    while (*list) {
	l2 = (*list)->next;
	safe_free (*list);
	*list = l2;	
    }
    list = NULL;
} /* seclist_destroy */


void
seclist_init (HWND dlg, int ctlid, int flags, keylist_t * ret_list)
{    
    gpgme_keycache_t kc = NULL;
    gpgme_key_t key = NULL;
    HWND kb;
    keylist_t list=NULL, l, l2;
    gpgme_attr_t name_attr = GPGME_ATTR_USERID;
    long pos = 0;

    SendDlgItemMessage (dlg, ctlid, CB_RESETCONTENT, 0, 0);
    kb = GetDlgItem (dlg, ctlid);
    kc = keycache_get_ctx (0);
    if (!kc)
	BUG (0);
    gpgme_keycache_rewind (kc);

    if (flags & KEYLIST_FLAG_SHORT)
	name_attr = GPGME_ATTR_NAME;
    while (!gpgme_keycache_next_key (kc, 1, &key)) {
	char * inf = NULL, * uid = NULL;
	const char * id;
	const char * keyid;
	int algo;
	size_t size = 0;
	
	id = gpgme_key_get_string_attr (key, name_attr, NULL, 0);
	keyid = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
	algo = gpgme_key_get_ulong_attr (key, GPGME_ATTR_ALGO, NULL, 0);
	if (!id || !keyid)
	    continue; /* fixme: error? */	
	if (!gpgme_key_get_ulong_attr (key, GPGME_ATTR_KEY_USABLE, NULL, 0))
	    continue;

	uid = utf8_to_wincp (id, strlen (id));
	size = strlen( uid ) + strlen( keyid ) + 32;
	inf = new char[size+1];
	if( !inf )
	    BUG( NULL );
	_snprintf(inf, size, _("%s (%s/0x%s)"), uid, 
		  gpgme_key_expand_attr (GPGME_ATTR_ALGO, algo), keyid + 8);
            
	combox_add_string (kb, inf);
	free_if_alloc (inf);
	free (uid);
	l = (struct keylist_s *)calloc (1, sizeof * l);
	if (!l)
	    BUG (0);
	l->key = key;
	if (!list)
	    list = l;
	else 
	{
	    for( l2 = list; l2->next; l2 = l2->next )
		;
	    l2->next = l;
	}
    }
    for( pos = 0, l2=list; pos < SendMessage( kb, CB_GETCOUNT, 0, 0 ); pos++, l2=l2->next )
	SendMessage( kb, CB_SETITEMDATA, pos, (LPARAM)(DWORD)l2->key );
    SendMessage( kb, CB_SETCURSEL, 0, 0 );
    *ret_list = list;
} /* seclist_init */
        

int
seclist_select_key (HWND dlg, int ctlid, gpgme_key_t * ret_key)
{
    int pos;
    DWORD k = 0;

    pos = SendDlgItemMessage( dlg, ctlid, CB_GETCURSEL, 0, 0 );
    if( pos == CB_ERR ) {
	msg_box( dlg, _("No key was selected."), _("Secret Key List"), MB_ERR );
	*ret_key = NULL;
    }
    else {
	k = SendDlgItemMessage( dlg, ctlid, CB_GETITEMDATA, pos, 0 );
	*ret_key = (gpgme_key_t)k;
    }
    return k? 0 : -1; 
} /* seclist_dlg_proc */