/* wptNLS.cpp - W32 Native Language Support
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *	Copyright (C) 1995-1999 Free Software Foundation, Inc.
 *
 * This code based on the simple-gettext.c code from the GnuPG 
 * by Ulrich Drepper.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
/* x-todo-status: OK */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <windows.h>

#include "wptTypes.h"
#include "wptNLS.h"


/* Missing W32 functions. */
static char *
stpcpy( char *a,const char *b )
{
    while( *b )
	*a++ = *b++;
    *a = 0;
    return (char*)a;
}

/* The magic number of the GNU message catalog format.	*/
#define MAGIC	      0x950412de
#define MAGIC_SWAPPED 0xde120495

/* Revision number of the currently used .mo (binary) file format.  */
#define MO_REVISION_NUMBER 0


/* Header for binary .mo file format.  */
struct mo_file_header {
  u32 magic; /* The magic number.	*/  
  u32 revision; /* The revision number of the file format.  */
  u32 nstrings; /* The number of strings pairs.  */  
  u32 orig_tab_offset; /* Offset of table with start offsets of original 
                          strings.  */
  u32 trans_tab_offset; /* Offset of table with start offsets of translation
                           strings.  */  
  u32 hash_tab_size; /* Size of hashing table.  */  
  u32 hash_tab_offset; /* Offset of first hashing entry.  */
};

struct string_desc {
  u32 length; /* Length of addressed string.  */  
  u32 offset; /* Offset of string in file.	*/
};

struct loaded_domain {
  char *data;
  int must_swap;
  u32 nstrings;
  char *mapped;
  struct string_desc *orig_tab;
  struct string_desc *trans_tab;
  u32 hash_size;
  u32 *hash_tab;
};

static struct loaded_domain *the_domain;

static u32
do_swap_u32( u32 i )
{
  return (i << 24) | ((i & 0xff00) << 8) | ((i >> 8) & 0xff00) | (i >> 24);
}

#define SWAPIT(flag, data) ((flag) ? do_swap_u32(data) : (data) )


/* We assume to have `unsigned long int' value with at least 32 bits.  */
#define HASHWORDBITS 32

/* The so called `hashpjw' function by P.J. Weinberger
   [see Aho/Sethi/Ullman, COMPILERS: Principles, Techniques and Tools,
   1986, 1987 Bell Telephone Laboratories, Inc.]  */

static u32
hash_string( const char *str_param )
{
    unsigned long int hval, g;
    const char *str = str_param;

    hval = 0;
    while (*str != '\0') {
	hval <<= 4;
	hval += (unsigned long int) *str++;	
	g = hval & ((unsigned long int) 0xf << (HASHWORDBITS - 4));
	if (g != 0) {
	    hval ^= g >> (HASHWORDBITS - 8);
	    hval ^= g;	
	}
    }
    return hval;
} /* hash_string */


static struct loaded_domain *
load_domain( const char *filename )
{
    FILE *fp;
    size_t size;
    struct stat st;
    struct mo_file_header *data = NULL;
    struct loaded_domain *domain = NULL;
    size_t to_read;
    char *read_ptr;

    fp = fopen( filename, "rb" );
    if( !fp )
       return NULL; /* can't open the file */
    /* we must know about the size of the file */
    if( fstat( fileno(fp ), &st )
	|| (size = (size_t)st.st_size) != st.st_size
	|| size < sizeof (struct mo_file_header) ) {
	fclose( fp );	
	return NULL;
    }

    data = (struct mo_file_header *) malloc( size );
    if( !data ) {
	fclose( fp );
	return NULL; /* out of memory */
    }

    to_read = size;
    read_ptr = (char *) data;
    do {
	long int nb = fread( read_ptr, 1, to_read, fp );
	if( nb < to_read ) {
	    fclose (fp);
	    free(data);	
	    return NULL; /* read error */
		
	}
	read_ptr += nb;	
	to_read -= nb;
    } while( to_read > 0 );
    fclose (fp);

    /* Using the magic number we can test whether it really is a message
     * catalog file.  */
    if( data->magic != MAGIC && data->magic != MAGIC_SWAPPED ) {
	/* The magic number is wrong: not a message catalog file.  */
	free( data );
	return NULL;
    }

    domain = (struct loaded_domain *)calloc( 1, sizeof *domain );
    if( !domain )  {
	free( data );	
	return NULL;
    }
    domain->data = (char *) data;
    domain->must_swap = data->magic != MAGIC;

    /* Fill in the information about the available tables.  */
    switch( SWAPIT(domain->must_swap, data->revision) ) {
      case 0:
	  domain->nstrings = SWAPIT(domain->must_swap, data->nstrings);
	  domain->orig_tab = (struct string_desc *)
	      ((char *) data + SWAPIT(domain->must_swap, data->orig_tab_offset));
	  domain->trans_tab = (struct string_desc *)
	      ((char *) data + SWAPIT(domain->must_swap, data->trans_tab_offset));
	  domain->hash_size = SWAPIT(domain->must_swap, data->hash_tab_size);	
	  domain->hash_tab = (u32 *)	
	      ((char *) data + SWAPIT(domain->must_swap, data->hash_tab_offset));	
	  break;

      default: /* This is an invalid revision.	*/
	  free( data );
	  free( domain );
	  return NULL;
    }

    /* allocate an array to keep track of code page mappings */
    domain->mapped = (char *)calloc( 1, domain->nstrings );
    if( !domain->mapped ) {
	free( data );
	free( domain );	
	return NULL;
    }

    return domain;
} /* load_domain */


/****************
 * Set the file used for translations.	Pass a NULL to disable
 * translation.  A new filename may be set at anytime.
 * WARNING: After changing the filename you shoudl not access any data
 *	    retrieved by gettext().
 */
int
set_gettext_file( const char *filename, const char *nls_dir )
{
    struct loaded_domain *domain = NULL;

    if( filename && *filename ) {
	if( filename[0] == '/'	   
	    || ( isalpha(filename[0])		 
	    && filename[1] == ':'	
	    && (filename[2] == '/' || filename[2] == '\\') )
	   ) {
	    /* absolute path - use it as is */	    
	    domain = load_domain( filename );
	}
	else { /* relative path - append ".mo" and get dir from the environment */
	    char *buf = NULL;
	    char *dir;
		
	    dir = strdup( nls_dir );
	    if( dir && (buf= (char *)malloc(strlen(dir)+strlen(filename)+1+3+1)) ) {
		strcpy(stpcpy(stpcpy(stpcpy( buf, dir),"/"), filename),".mo");
		domain = load_domain( buf );			
		free(buf);
	    }
	    free(dir);
	}
	if( !domain )
	    return -1;
    }

    if( the_domain ) {
	free( the_domain->data );
	free( the_domain->mapped );
	free( the_domain );	
	the_domain = NULL;
    }
    the_domain = domain;
    return 0;
} /* set_gettext_file */


static const char*
get_string( struct loaded_domain *domain, u32 idx )
{
    char *p = domain->data + SWAPIT(domain->must_swap,
				    domain->trans_tab[idx].offset);
    if( !domain->mapped[idx] ) {	
	domain->mapped[idx] = 1;		
    }
    return (const char*)p;
} /* get_string */

const char *
gettext( const char *msgid )
{
    struct loaded_domain *domain;
    size_t act = 0;
    size_t top, bottom;

    if( !(domain = the_domain) )	
	goto not_found;

    /* Locate the MSGID and its translation.  */
    if( domain->hash_size > 2 && domain->hash_tab ) {		
	/* Use the hashing table.  */		
	u32 len = strlen (msgid);	
	u32 hash_val = hash_string (msgid);	
	u32 idx = hash_val % domain->hash_size;	
	u32 incr = 1 + (hash_val % (domain->hash_size - 2));	
	u32 nstr = SWAPIT (domain->must_swap, domain->hash_tab[idx]);

	if ( !nstr ) /* Hash table entry is empty.  */	
	    goto not_found;

		
	if( SWAPIT(domain->must_swap,	
	    domain->orig_tab[nstr - 1].length) == len	
	    && !strcmp( msgid,	
	    domain->data + SWAPIT(domain->must_swap,
				    domain->orig_tab[nstr - 1].offset)) )
				    return get_string( domain, nstr - 1 );
	for(;;) {
	    if (idx >= domain->hash_size - incr)
		idx -= domain->hash_size - incr;
	    else
		idx += incr;
	    nstr = SWAPIT(domain->must_swap, domain->hash_tab[idx]);
	    if( !nstr )	
		goto not_found; /* Hash table entry is empty.  */

	    if ( SWAPIT(domain->must_swap,
				domain->orig_tab[nstr - 1].length) == len
				&& !strcmp (msgid,
				domain->data + SWAPIT(domain->must_swap,
					   domain->orig_tab[nstr - 1].offset)))
			
					   return get_string( domain, nstr-1 );
	} /* NOTREACHED */
    }

    /* Now we try the default method:  binary search in the sorted
       array of messages.  */
    bottom = 0;
    top = domain->nstrings;
    while( bottom < top ) {		
	int cmp_val;
	
	act = (bottom + top) / 2;	
	cmp_val = strcmp(msgid, domain->data + SWAPIT(domain->must_swap,
			domain->orig_tab[act].offset));
	if (cmp_val < 0)
	    top = act;
	else if (cmp_val > 0)
	    bottom = act + 1;
	else	
	    return get_string( domain, act );
    }

not_found:
    return msgid;
} /* gettext */
