/* wptPassphraseCB.cpp - GPGME Passphrase Callback
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <windows.h>
#include <ctype.h>

#include "../resource.h"
#include "wptNLS.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptDlgs.h"
#include "wptUTF8.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptAgent.h"
#include "wptRegistry.h"


#define item_ctrl_id( cmd ) \
    ((cmd) == GPG_CMD_DECRYPT? IDC_DECRYPT_PWD : IDC_DECRYPT_SIGN_PWD)


BOOL CALLBACK
passphrase_callback_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{    
    static passphrase_cb_s * c;
    static int hide = 1;
    gpgme_key_t key;
    const char * s, * id;
    char info[768] = {0};
    void * ctx = NULL, * item;
    unsigned int pkalgo;

    switch( msg )  {
    case WM_INITDIALOG:
	hide = 1;
	c = (passphrase_cb_s *)lparam;
	if (!c)
	    BUG (0);
	SetWindowText (dlg, c->title);
	if (c->gpg_cmd == GPG_CMD_DECRYPT) {
	    SetDlgItemText( dlg, IDC_DECRYPT_LISTINF,
		_("Encrypted with the following public key(s)") );
	    CheckDlgButton( dlg, IDC_DECRYPT_HIDE, BST_CHECKED );
	}
	else if( c->gpg_cmd == GPG_CMD_SIGN )
	    CheckDlgButton( dlg, IDC_DECRYPT_SIGN_HIDE, BST_CHECKED );
	if( c->enc_to && c->gpg_cmd == GPG_CMD_DECRYPT ) {
	    gpgme_recipients_enum_open( c->enc_to, &ctx );
	    while ( (s = gpgme_recipients_enum_read( c->enc_to, &ctx )) ) {
		pkalgo = *s; s++;
		get_pubkey( s, &key );
		if( key ) {
		    char * uid = NULL;
		    id = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
		    if( !id )
			id = _("Invalid User ID");
		    uid = utf8_to_wincp (id, strlen (id));		   
		    _snprintf( info, sizeof info - 1, "%s (%s, 0x%s)", uid,
				gpgme_key_expand_attr( GPGME_ATTR_ALGO, pkalgo ), s+8 );
		    free( uid );
		}
		else
		    _snprintf( info, sizeof info - 1, _("Unknown (key ID 0x%s)"),
			       s? s + 8 : "DEADBEEF" );
		listbox_add_string( GetDlgItem( dlg, IDC_DECRYPT_LIST ), info );
	    }
	    gpgme_recipients_enum_close( c->enc_to, &ctx );
	}
	SetDlgItemText( dlg, c->gpg_cmd == GPG_CMD_DECRYPT?
			IDC_DECRYPT_PWDINFO : IDC_DECRYPT_SIGN_PWDINFO,
			_("Please enter your passphrase") );
	if( c->gpg_cmd == GPG_CMD_DECRYPT ) {
	    SetFocus( GetDlgItem( dlg, IDC_DECRYPT_PWD ) );
	    SetDlgItemText( dlg, IDC_DECRYPT_MSG, c->info );
	}
	else {
	    SetFocus( GetDlgItem( dlg, IDC_DECRYPT_SIGN_PWD ) );
	    SetDlgItemText( dlg, IDC_DECRYPT_SIGN_MSG, c->info );
	}
	center_window( dlg );
	SetForegroundWindow( dlg );
	set_active_window( dlg );
	return FALSE;

	case WM_SYSCOMMAND:
	    if( LOWORD( wparam ) == SC_CLOSE ) {
		SetDlgItemText( dlg, item_ctrl_id( c->gpg_cmd ), "" );
		c->cancel = 1;
		EndDialog( dlg, TRUE );
	    }
	    return FALSE;

	case WM_COMMAND:
	    switch( HIWORD( wparam ) ) {
	    case BN_CLICKED:
		if ( LOWORD( wparam ) == IDC_DECRYPT_HIDE
		    || LOWORD( wparam ) == IDC_DECRYPT_SIGN_HIDE ) {
		    HWND hwnd;
		    hide ^= 1;
		    hwnd = GetDlgItem( dlg, item_ctrl_id( c->gpg_cmd ) );
		    SendMessage( hwnd, EM_SETPASSWORDCHAR, hide? '*' : 0, 0 );
		    SetFocus( hwnd );
		}
	    }

	    switch (LOWORD (wparam)) {
	    case IDOK:	
		/* fixme: the item is even cached when the passphrase is not
		          correct, which means that the user needs to delete all
			  cached entries to continue. */
		GetDlgItemText (dlg, item_ctrl_id (c->gpg_cmd), c->pwd, sizeof (c->pwd) -1);
		if (reg_prefs.cache_time > 0 && !c->is_card && !strstr (c->keyid, "missing")) {
		    if (agent_get_cache (c->keyid, &item))
			agent_unlock_cache_entry (&item);
		    else
			agent_put_cache (c->keyid, c->pwd, reg_prefs.cache_time);
		}
		EndDialog (dlg, TRUE);
		return TRUE;
		
	    case IDCANCEL:
		SetDlgItemText (dlg, item_ctrl_id (c->gpg_cmd), "" );
		c->cancel = 1;
		EndDialog (dlg, FALSE);
		return FALSE;
	    }
	    break;
    }
    
    return FALSE;
} /* passphrase_callback_proc */


static const char *
parse_gpg_keyid( const char * desc )
{
    static char keyid[16+1];
    char * p;
    
    p = strrchr( desc, '\n' );
    if( !p )
	return NULL;
    /* the format of the desc buffer looks like this:
       request_keyid[16] main_keyid[16] keytype[1] keylength[4]
       we use the main keyid to use only one cache entry. */
    strncpy( keyid, desc+(p-desc+1+17), 16 );
    return keyid;
} /* parse_gpg_keyid */


static void
parse_gpg_description( char * desc, int size )
{
    char * buffer = NULL, ch, uid[128] = {0}, * uid2 = NULL;
    char usedkey[16+1] = {0}, mainkey[16+1] = {0};
    const char * buf;
    int i, algo, try_again = 0;

    if( stristr( desc, "[User ID hint missing]" ) 
	|| stristr( desc, "[passphrase info missing]" ) ) {
	_snprintf( desc, size-1, 
		   _("You need a passphrase to unlock the secret key for\n"
		     "user: [UserID hint missing]\n"
		     "      [passphrase info missing]\n") );
	return;
    }

    buffer = new char[size+1];
    if( !buffer )
	BUG( NULL );
    strcpy( buffer, desc );
    buf = buffer;
    if( strstr( buf, "TRY_AGAIN" ) ) {
	buf += strlen( "TRY_AGAIN\n" );
	try_again = 1;
    }
    else if( strstr( buf, "ENTER_PASSPHRASE" ) )
	buf += strlen( "ENTER_PASSPHRASE\n" );
    else
	BUG( NULL );
    buf += 17;
    for( i = 0; i < sizeof uid-1; i++ ) {
	ch = *buf++;
	if( ch == '\n' ) {
	    uid[i] = '\0';
	    break;
	}
	uid[i] = ch;
    }
    memcpy( usedkey, buf, 16 );
    usedkey[16] = '\0';
    buf += 17;
    memcpy( mainkey, buf, 16 );
    mainkey[16] = '\0';
    buf += 17;
    if( !buf )
	BUG( NULL );
    algo = atol( buf );
    free_if_alloc( buffer );

    uid2 = utf8_to_wincp (uid, strlen (uid));

    if( strcmp( usedkey, mainkey ) )
	_snprintf( desc, size-1, 
		   _("You need a passphrase to unlock the secret key for\n"
		     "user: \"%s\"\n"
		     "%s key, ID %s (main key ID %s)\n"),
		   uid2, gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo ),
		   usedkey+8, mainkey+8 );
    else if( !strcmp( usedkey, mainkey ) )
	_snprintf( desc, size-1, 
		   _("You need a passphrase to unlock the secret key for\n"
		     "user: \"%s\"\n"
		     "%s key, ID %s\n"),
		     uid2, gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo ),
		     usedkey+8 );
    free( uid2 );
} /* parse_gpg_describtion */


static int inline
is_hexstring( const char * p )
{
    size_t i;
    for( i=0; i < strlen( p ); i++ ) {
	if( !isxdigit( p[i] ) )
	    return -1;
    }
    return 0;
}


const char *
passphrase_cb( void * opaque, const char * desc, void * r_hd )
{   
    passphrase_cb_s * c = (passphrase_cb_s *)opaque;
    void * item;
    const char * pass, * keyid;
    int rc = 0;

    if( !c )
	return NULL;

    if( desc ) {
	keyid = parse_gpg_keyid( desc );
	pass = agent_get_cache( keyid, &item );
	if( pass ) {
	    agent_unlock_cache_entry( &item );
	    c->pwd_init = 0;
	    return pass;
	}
    }

    if( c->pwd_init ) {
	c->keyid = keyid;
	/* if the desc has a length of 32 and only hex digits, we assume a 
	   smart card has been used. */	
	/*log_box( "", 0, "%s %d %d", desc,strlen( desc), is_hexstring( desc ) );*/
	if( desc && strlen( desc ) == 32 && !is_hexstring( desc ) ) {
	    char buf[16];
	    memset( buf, 0, sizeof buf );
	    strncpy( buf, desc+20, 8 );
	    _snprintf( c->info, sizeof c->info-1,
		    _("Please enter the PIN to unlock your secret card key\n"
		      "Card: %s"), buf );
	    c->is_card = 1;
	}
	else if( desc ) {
	    strcpy( c->info, desc );
	    parse_gpg_description( c->info, sizeof c->info - 1 );
	}
	if( c->gpg_cmd == GPG_CMD_DECRYPT ) {
	    rc = DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_DECRYPT,
				(HWND)c->hwnd, passphrase_callback_proc, 
				(LPARAM)c );
	}
	else if( c->gpg_cmd == GPG_CMD_SIGN ) {
	    rc = DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_DECRYPT_SIGN,
				(HWND)c->hwnd, passphrase_callback_proc, 
				(LPARAM)c );
	}
	if( rc == -1 )
	    return NULL;
	c->pwd_init = 0;
    }

    if( c->cancel )
	return NULL;

    return c->pwd;
} /* passphrase_cb */


void
set_gpg_passphrase_cb( gpgme_ctx_t c, passphrase_cb_s * ctx, int cmd, 
		       HWND hwnd, const char * title )
{
    ctx->gpg_cmd = cmd;
    ctx->is_card = 0;
    ctx->cancel = 0;
    ctx->hwnd = hwnd;
    ctx->pwd_init = 1;
    if( strlen( title ) > 256 )
	BUG( NULL ); /* check bounds */
    strcpy( ctx->title, title );
    gpgme_set_passphrase_cb( c, passphrase_cb, ctx );
} /* set_gpg_passphrase_cb */
