/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 * qpro-read.c: Read Quatro Pro files
 *
 * Copyright (C) 2002 Jody Goldberg (jody@gnome.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
#include <gnumeric-config.h>
#include <gnumeric.h>
#include "qpro.h"

#include <gutils.h>
#include <plugin-util.h>
#include <file.h>
#include <workbook-view.h>
#include <workbook.h>
#include <sheet.h>
#include <cell.h>
#include <value.h>
#include <expr.h>
#include <sheet-style.h>
#include <style-color.h>
#include <parse-util.h>
#include <module-plugin-defs.h>

#include <gsf/gsf-utils.h>
#include <gsf/gsf-input.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-infile-msole.h>
#include <libgnome/gnome-i18n.h>

GNUMERIC_MODULE_PLUGIN_INFO_DECL;

static gboolean
qpro_check_signature (GsfInput *input)
{
	guint8 const *header;
	guint16 version;

	if (gsf_input_seek (input, 0, GSF_SEEK_SET) ||
	    NULL == (header = gsf_input_read (input, 2+2+2, NULL)) ||
	    GSF_LE_GET_GUINT16 (header + 0) != 0 ||
	    GSF_LE_GET_GUINT16 (header + 2) != 2)
		return FALSE;
	version = GSF_LE_GET_GUINT16 (header + 4);
	return (version == 0x1001 || /* 'WB1' format, documented */
		version == 0x1002 || /* 'WB2' format, documented */
		version == 0x1006 ||  /* qpro 6.0 ?? */
		version == 0x1007);  /* qpro 7.0 ?? */
}

gboolean
qpro_file_probe (GnumFileOpener const *fo, GsfInput *input, FileProbeLevel pl)
{
	GsfInfile *ole;
	GsfInput *stream;
	gboolean res = FALSE;

	/* check for >= QPro 6.0 which is OLE based */
	ole = gsf_infile_msole_new (input, NULL);
	if (ole != NULL) {
		stream = gsf_infile_child_by_name (ole, "PerfectOffice_MAIN");
		if (stream != NULL) {
			res = qpro_check_signature (stream);
			g_object_unref (G_OBJECT (stream));
		}
		g_object_unref (G_OBJECT (ole));
	} else
		res = qpro_check_signature (input);

	return res;
}

typedef struct {
	GsfInput	*input;
	IOContext	*io_context;
	WorkbookView	*wbv;
	Workbook	*wb;
	Sheet		*cur_sheet;
} QProReadState;

static guint8 const *
qpro_get_record (QProReadState *state, guint16 *id, guint16 *len)
{
	guint8 const *data;

	if (NULL == (data = gsf_input_read (state->input, 4, NULL))) {
		g_warning ("read failure");
		return NULL;
	}
	*id  = GSF_LE_GET_GUINT16 (data + 0);
	*len = GSF_LE_GET_GUINT16 (data + 2);

#if 0
	printf ("%hd with %hd\n", *id, *len);
#endif

	if (*len == 0)
		return "";
	/* some sanity checking */
	g_return_val_if_fail (*len < 0x2000, NULL);
	data = gsf_input_read (state->input, *len, NULL);
	if (data == NULL)
		g_warning ("huh? failure reading %hd for type %hd", *len, *id);
	return data;
}

#define state(f)	case f : printf ("%s = %hd\n", #f, len); break
#define validate(f,expected) qpro_validate_len (state, #f, len, expected)

static gboolean
qpro_validate_len (QProReadState *state, char const *id, guint16 len, int expected_len)
{
	if (expected_len >= 0  && len != expected_len) {
		gnm_io_warning (state->io_context,
			_("Invalid '%s' record of length %hd instead of %d"),
			id, len, expected_len);
		return FALSE;
	}

	return TRUE;
}


static void
qpro_parse_formula (QProReadState *state, int col, int row,
		    guint8 const *data, guint8 const *end)
{
	int magic = GSF_LE_GET_GUINT16 (data + 6) & 0x7ff8;
	guint16 ref_offset = GSF_LE_GET_GUINT16 (data + 12);
	Value   *val;
	GSList  *stack = NULL;
	GnmExpr const *expr;
	guint8 const *refs, *fmla;
	int len;

	fmla = data + 14;
	refs = fmla + ref_offset;
	g_return_if_fail (refs <= end);

#if 0
	puts (cell_coord_name (col, row));
	gsf_mem_dump (fmla, refs-fmla);
	gsf_mem_dump (refs, end-fmla);
#endif

	while (fmla < refs && *fmla != QPRO_OP_EOF) {
		expr = NULL;
		len = 1;
		switch (*fmla) {
		case QPRO_OP_CONST_FLOAT:
			expr = gnm_expr_new_constant (value_new_float (
				gsf_le_get_double (fmla+1)));
			len = 9;
			break;

		case QPRO_OP_CELLREF: {
			CellRef ref;
			guint16 tmp = GSF_LE_GET_GUINT16 (refs + 4);
			ref.sheet = NULL;
			ref.col = *((gint8 *)(refs + 2));
			ref.col_relative = (tmp & 0x4000) ? TRUE : FALSE;
			ref.row_relative = (tmp & 0x2000) ? TRUE : FALSE;
			if (ref.row_relative)
				ref.row = ((gint16)(tmp & 0x1fff) << 3) >> 3;
			else
				ref.row = tmp & 0x1fff;
			expr = gnm_expr_new_cellref (&ref);
			refs += 6;
			break;
		}

		case QPRO_OP_RANGEREF: {
			CellRef a, b;

			expr = gnm_expr_new_constant (
				value_new_cellrange_unsafe (&a, &b));
			refs += 10;
			break;
		}
		case QPRO_OP_EOF:	break; /* exit */
		case QPRO_OP_PAREN:	break; /* ignore for now */

		case QPRO_OP_CONST_INT:
			expr = gnm_expr_new_constant (
				value_new_int ((gint16)GSF_LE_GET_GUINT16 (fmla+1)));
			len = 3;
			break;

		case QPRO_OP_CONST_STR:
			expr = gnm_expr_new_constant (value_new_string (fmla+1));
			len = 1 + strlen (fmla+1) + 1;
			break;

		case QPRO_OP_DEFAULT_ARG:
			expr = gnm_expr_new_constant (value_new_empty ());
			break;

		case QPRO_OP_ADD: case QPRO_OP_SUB:
		case QPRO_OP_MULT: case QPRO_OP_DIV:
		case QPRO_OP_EXP:
		case QPRO_OP_EQ: case QPRO_OP_NE:
		case QPRO_OP_LE: case QPRO_OP_GE:
		case QPRO_OP_LT: case QPRO_OP_GT:
		case QPRO_OP_CONCAT: {
			static GnmExprOp const binop_map[] = {
				GNM_EXPR_OP_ADD,	GNM_EXPR_OP_SUB,
				GNM_EXPR_OP_MULT,	GNM_EXPR_OP_DIV,
				GNM_EXPR_OP_EXP,
				GNM_EXPR_OP_EQUAL,	GNM_EXPR_OP_NOT_EQUAL,
				GNM_EXPR_OP_LTE,	GNM_EXPR_OP_GTE,
				GNM_EXPR_OP_LT,		GNM_EXPR_OP_GT,
				0, 0, 0, 0,
				GNM_EXPR_OP_CAT
			};
			GnmExpr const *l, *r;
			GSList *tmp = stack;

			g_return_if_fail (stack != NULL && stack->next != NULL);
			r = stack->data;
			l = stack->next->data;

			stack = stack->next->next;
			tmp->next->next = NULL;
			g_slist_free (tmp);
			expr = gnm_expr_new_binary (
				l, binop_map [*fmla - QPRO_OP_ADD], r);
			break;
		}

		case QPRO_OP_AND:
		case QPRO_OP_OR:
		case QPRO_OP_NOT:

		case QPRO_OP_UNARY_NEG:
		case QPRO_OP_UNARY_PLUS: {
			GSList *tmp = stack;

			g_return_if_fail (stack != NULL);
			expr = stack->data;

			stack = stack->next;
			tmp->next= NULL;
			g_slist_free (tmp);

			expr = gnm_expr_new_unary ((*fmla == QPRO_OP_UNARY_NEG)
				? GNM_EXPR_OP_UNARY_NEG : GNM_EXPR_OP_UNARY_PLUS,
				expr);
			break;
		}

		default:
			if (QPRO_OP_FIRST_FUNC <= *fmla && *fmla <= QPRO_OP_LAST_FUNC) {
			}
		}
		if (expr != NULL) {
			stack = g_slist_prepend (stack, (gpointer)expr);
		}
		fmla += len;
	}
	g_return_if_fail (fmla != refs);
	g_return_if_fail (stack != NULL);
	g_return_if_fail (stack->next == NULL);

	expr = stack->data;
	g_slist_free (stack);

	if (magic == 0x7ff0) {
		val = value_new_error (NULL,  gnumeric_err_VALUE);
	} else if (magic == 0x7ff8) {
		guint16 id, len;
		int new_row, new_col;

		data = qpro_get_record (state, &id, &len);

		g_return_if_fail (data != NULL);
		g_return_if_fail (id == QPRO_FORMULA_STRING);

		new_col = data [0]; 
		new_row = GSF_LE_GET_GUINT16 (data + 2);

		/* Be anal */
		g_return_if_fail (col == new_col);
		g_return_if_fail (row == new_row);

		val = value_new_string (data + 7);
	} else
		val = value_new_float (gsf_le_get_double (data));

	cell_set_expr_and_value (sheet_cell_fetch (state->cur_sheet, col, row),
		expr, val, TRUE);
	gnm_expr_unref (expr);
}

static MStyle *
qpro_get_style (QProReadState *state, guint8 const *data)
{
	unsigned attr_id = GSF_LE_GET_GUINT16 (data) >> 3;
	printf ("Get Attr %u\n", attr_id);
	return sheet_style_default (state->cur_sheet);
}

static void
qpro_read_sheet (QProReadState *state)
{
	guint16 id, len;
	guint8 const *data;

	/* We can use col_name as a quick proxy for the defaul q-pro sheet names */
	char const *def_name = col_name (workbook_sheet_count (state->wb));
	Sheet *sheet = sheet_new (state->wb, def_name);

	state->cur_sheet = sheet;
	workbook_sheet_attach (state->wb, sheet, NULL);
	sheet_flag_recompute_spans (sheet);
	printf ("----------> start %s\n", def_name);
	while (NULL != (data = qpro_get_record (state, &id, &len))) {
		switch (id) {
		case QPRO_BLANK_CELL:
			if (validate (QPRO_BLANK_CELL, 6))
				sheet_style_set_pos (sheet,
					data [0], GSF_LE_GET_GUINT16 (data + 2),
					qpro_get_style (state, data + 4));
			break;

		case QPRO_INTEGER_CELL:
			if (validate (QPRO_INTEGER_CELL, 8)) {
				int col = data [0];
				int row = GSF_LE_GET_GUINT16 (data + 2);
				sheet_style_set_pos (sheet, col, row,
					qpro_get_style (state, data + 4));
				cell_assign_value (sheet_cell_fetch (sheet, col, row),
					value_new_int (GSF_LE_GET_GUINT16 (data + 6)));
			}
			break;

		case QPRO_FLOATING_POINT_CELL:
			if (validate (QPRO_FLOATING_POINT_CELL, 14)) {
				int col = data [0];
				int row = GSF_LE_GET_GUINT16 (data + 2);
				sheet_style_set_pos (sheet, col, row,
					qpro_get_style (state, data + 4));
				cell_assign_value (sheet_cell_fetch (sheet, col, row),
					value_new_float (gsf_le_get_double (data + 6)));
			}
			break;

		case QPRO_LABEL_CELL:
			if (validate (QPRO_LABEL_CELL, -1)) {
				int col = data [0];
				int row = GSF_LE_GET_GUINT16 (data + 2);
				sheet_style_set_pos (sheet, col, row,
					qpro_get_style (state, data + 4));
				cell_assign_value (sheet_cell_fetch (sheet, col, row),
					value_new_string (data + 7));
			}
			break;

		case QPRO_FORMULA_CELL:
			if (validate (QPRO_FORMULA_CELL, -1)) {
				int col = data [0];
				int row = GSF_LE_GET_GUINT16 (data + 2);
				sheet_style_set_pos (sheet, col, row,
					qpro_get_style (state, data + 4));

				qpro_parse_formula (state, col, row,
					data + 6, data + len);
			}
			break;

		case QPRO_END_OF_PAGE:
			break;

		case QPRO_COLUMN_SIZE:
			/* ignore this, we auto generate this info */
			break;

		case QPRO_PROTECTION:
			if (validate (QPRO_PROTECTION, 1))
				sheet->is_protected = (data[0] == 0xff);
			break;

		case QPRO_PAGE_NAME:
			if (validate (QPRO_PAGE_NAME, -1)) {
#warning this is wrong, but the workbook interface is confused and needs a control.
				sheet_rename (sheet, data);
			}
			break;

		case QPRO_PAGE_ATTRIBUTE:
			if (validate (QPRO_PAGE_ATTRIBUTE, 30)) {
#warning TODO, mostly simple
			}
			break;

		case QPRO_DEFAULT_ROW_HEIGHT_PANE1:
		case QPRO_DEFAULT_ROW_HEIGHT_PANE2:
			if (validate (QPRO_DEFAULT_ROW_HEIGHT, 2)) {
			}
			break;

		case QPRO_DEFAULT_COL_WIDTH_PANE1:
		case QPRO_DEFAULT_COL_WIDTH_PANE2:
			if (validate (QPRO_DEFAULT_COL_WIDTH, 2)) {
			}
			break;

		case QPRO_MAX_FONT_PANE1:
		case QPRO_MAX_FONT_PANE2 :
			/* just ignore for now */
			break;

		case QPRO_PAGE_TAB_COLOR :
			if (validate (QPRO_PAGE_TAB_COLOR, 4))
				sheet_set_tab_color (sheet, style_color_new_i8 (
					data [0], data [1], data [2]), NULL);
			break;

		case QPRO_PAGE_ZOOM_FACTOR :
			if (validate (QPRO_PAGE_ZOOM_FACTOR, 4)) {
				guint16 low  = GSF_LE_GET_GUINT16 (data);
				guint16 high = GSF_LE_GET_GUINT16 (data + 2);

				if (low == 100) {
					if (high < 10 || high > 400)
						gnm_io_warning (state->io_context, 
							_("Invalid zoom %hd %%"), high);
					else
						sheet_set_zoom_factor (sheet, ((double)high) / 100.,
							FALSE, FALSE);
				}
			}
			break;
		}

		if (id == QPRO_END_OF_PAGE)
			break;
	}
	printf ("----------< end\n");
	state->cur_sheet = NULL;
}

static void
qpro_read_workbook (QProReadState *state, GsfInput *input)
{
	guint16 id, len;
	guint8 const *data;

	state->input = input;
	gsf_input_seek (input, 0, GSF_SEEK_SET);

	while (NULL != (data = qpro_get_record (state, &id, &len))) {
		switch (id) {
		case QPRO_BEGINNING_OF_FILE:
			if (validate (QPRO_BEGINNING_OF_FILE, 2)) {
				guint16 version;
				version = GSF_LE_GET_GUINT16 (data);
			}
			break;
		case QPRO_BEGINNING_OF_PAGE:
			qpro_read_sheet (state);
			break;

		default :
			if (id > QPRO_LAST_SANE_ID)
				gnm_io_warning (state->io_context,
					_("Invalid record %d of length %hd"),
					id, len);
		};
		if (id == QPRO_END_OF_FILE)
			break;
	}
}

void
qpro_file_open (GnumFileOpener const *fo, IOContext *context,
		WorkbookView *new_wb_view, GsfInput *input)
{
	QProReadState state;
	GsfInput *stream = NULL;
	GsfInfile *ole = gsf_infile_msole_new (input, NULL);

	state.io_context = context;
	state.wbv = new_wb_view;
	state.wb = wb_view_workbook (new_wb_view);
	state.cur_sheet = NULL;

	/* check for >= QPro 6.0 which is OLE based */
	ole = gsf_infile_msole_new (input, NULL);
	if (ole != NULL) {
		stream = gsf_infile_child_by_name (ole, "PerfectOffice_MAIN");
		if (stream != NULL) {
			qpro_read_workbook (&state, stream);
			g_object_unref (G_OBJECT (stream));
		}
		g_object_unref (G_OBJECT (ole));
	} else
		qpro_read_workbook (&state, input);
}
