/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * utils.c:  Various utility routines that do not depend on the GUI of Gnumeric
 *
 * Authors:
 *    Miguel de Icaza (miguel@gnu.org)
 *    Jukka-Pekka Iivonen (iivonen@iki.fi)
 *    Zbigniew Chyla (cyba@gnome.pl)
 */
#include <gnumeric-config.h>
#include <glib/gi18n.h>
#include "gnumeric.h"
#include "gutils.h"
#include "gnumeric-paths.h"

#include "sheet.h"
#include "ranges.h"
#include "mathfunc.h"

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <gsf/gsf-impl-utils.h>

#ifndef G_OS_WIN32
const char *gnumeric_lib_dir = GNUMERIC_LIBDIR;
const char *gnumeric_data_dir = GNUMERIC_DATADIR;
static const char gnumeric_icon_dir[] = GNUMERIC_ICONDIR;
static const char gnumeric_locale_dir[] = GNUMERIC_LOCALEDIR;
#else
const char *gnumeric_lib_dir;
const char *gnumeric_data_dir;
static const char *priv_lib_dir;
static const char *priv_data_dir;
static const char *gnumeric_icon_dir;
static const char *gnumeric_locale_dir;
#endif

static const char *gnumeric_usr_dir;

void
gutils_init (void)
{
	char const *home_dir;
#ifdef G_OS_WIN32
	gchar *dir = g_win32_get_package_installation_directory (NULL, NULL);
	priv_lib_dir = gnumeric_lib_dir = g_build_filename (dir,
		"lib", "gnumeric", GNUMERIC_VERSION, NULL);
	priv_data_dir = gnumeric_data_dir = g_build_filename (dir,
		"share", "gnumeric", GNUMERIC_VERSION, NULL);
	gnumeric_icon_dir = g_build_filename (dir,
		"share", "pixmaps", "gnumeric", NULL);
	gnumeric_locale_dir = g_build_filename (dir,
		"share", "locale", NULL);
	g_free (dir);
#endif
	home_dir = g_get_home_dir ();
	gnumeric_usr_dir = (home_dir == NULL ? NULL :
	   g_build_filename (home_dir, ".gnumeric", GNUMERIC_VERSION, NULL));
}

void
gutils_shutdown (void)
{
#ifdef G_OS_WIN32
	g_free (priv_lib_dir);
	g_free (priv_data_dir);
	g_free (gnumeric_icon_dir);
	g_free (gnumeric_locale_dir);
	g_free (gnumeric_usr_dir);
#endif
}

char const *
gnm_sys_lib_dir (void)
{
	return gnumeric_lib_dir;
}

char const *
gnm_sys_data_dir (void)
{
	return gnumeric_data_dir;
}

char const *
gnm_icon_dir (void)
{
	return gnumeric_icon_dir;
}

char const *
gnm_locale_dir (void)
{
	return gnumeric_locale_dir;
}

char const *
gnm_usr_dir (void)
{
	return gnumeric_usr_dir;
}

int
gnumeric_regcomp_XL (go_regex_t *preg, char const *pattern, int cflags)
{
	GString *res = g_string_new (NULL);
	int retval;

	while (*pattern) {
		switch (*pattern) {
		case '~':
			pattern++;
			if (*pattern == '*')
				g_string_append (res, "\\*");
			else
				g_string_append_c (res, *pattern);
			if (*pattern) pattern++;
			break;

		case '*':
			g_string_append (res, ".*");
			pattern++;
			break;

		case '?':
			g_string_append_c (res, '.');
			pattern++;
			break;

		default:
			pattern = go_regexp_quote1 (res, pattern);
		}
	}

	retval = go_regcomp (preg, res->str, cflags);
	g_string_free (res, TRUE);
	return retval;
}

#if 0
static const char *
color_to_string (PangoColor color)
{
	static char result[100];
	sprintf (result, "%04x:%04x:%04x", color.red, color.green, color.blue);
	return result;
}

static gboolean
cb_gnm_pango_attr_dump (PangoAttribute *attr, gpointer user_data)
{
	g_print ("  start=%u; end=%u\n", attr->start_index, attr->end_index);
	switch (attr->klass->type) {
	case PANGO_ATTR_FAMILY: g_print ("    family=\"%s\"\n", ((PangoAttrString *)attr)->value); break;
	case PANGO_ATTR_LANGUAGE: g_print ("    language=\"%s\"\n", pango_language_to_string (((PangoAttrLanguage *)attr)->value)); break;
	case PANGO_ATTR_STYLE: g_print ("    style=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_WEIGHT: g_print ("    weight=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_VARIANT: g_print ("    variant=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_STRETCH: g_print ("    stretch=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_UNDERLINE: g_print ("    underline=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_STRIKETHROUGH: g_print ("    strikethrough=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_RISE: g_print ("    rise=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_FALLBACK: g_print ("    fallback=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_LETTER_SPACING: g_print ("    letter_spacing=%d\n", ((PangoAttrInt *)attr)->value); break;
	case PANGO_ATTR_SIZE: g_print ("    size=%d%s\n",
				       ((PangoAttrSize *)attr)->size,
				       ((PangoAttrSize *)attr)->absolute ? " abs" : ""); break;
	case PANGO_ATTR_SCALE: g_print ("    scale=%g\n", ((PangoAttrFloat *)attr)->value); break;
	case PANGO_ATTR_FOREGROUND: g_print ("    foreground=%s\n", color_to_string (((PangoAttrColor *)attr)->color)); break;
	case PANGO_ATTR_BACKGROUND: g_print ("    background=%s\n", color_to_string (((PangoAttrColor *)attr)->color)); break;
	case PANGO_ATTR_UNDERLINE_COLOR: g_print ("    underline_color=%s\n", color_to_string (((PangoAttrColor *)attr)->color)); break;
	case PANGO_ATTR_STRIKETHROUGH_COLOR: g_print ("    strikethrough_color=%s\n", color_to_string (((PangoAttrColor *)attr)->color)); break;
	default: g_print ("    type=%d\n", attr->klass->type);
	}

	return FALSE;
}

void
gnm_pango_attr_dump (PangoAttrList *list)
{
	g_print ("PangoAttrList at %p\n", list);
	pango_attr_list_filter (list, cb_gnm_pango_attr_dump, NULL);
}
#endif

/* ------------------------------------------------------------------------- */
