package org.gnu.libebook;

import org.gnu.glib.Error;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.glib.JGException;
import org.gnu.libedataserver.EDSObject;
import org.gnu.libedataserver.ESource;
import org.gnu.libedataserver.ESourceList;

/**
 * Represents a complete Evolution Address Book object.
 *
 * Below are a list of Error Codes that can be retrieved if a JGException is thrown:<br/><br/>
 *
 * 0  - E_BOOK_ERROR_OK<br/>
 * 1  - E_BOOK_ERROR_INVALID_ARG,<br/>
 * 2  - E_BOOK_ERROR_BUSY,<br/>
 * 3  - E_BOOK_ERROR_REPOSITORY_OFFLINE,<br/>
 * 4  - E_BOOK_ERROR_NO_SUCH_BOOK,<br/>
 * 5  - E_BOOK_ERROR_NO_SELF_CONTACT,<br/>
 * 6  - E_BOOK_ERROR_SOURCE_NOT_LOADED,<br/>
 * 7  - E_BOOK_ERROR_SOURCE_ALREADY_LOADED,<br/>
 * 8  - E_BOOK_ERROR_PERMISSION_DENIED,<br/>
 * 9  - E_BOOK_ERROR_CONTACT_NOT_FOUND,<br/>
 * 10 - E_BOOK_ERROR_CONTACT_ID_ALREADY_EXISTS,<br/>
 * 11 - E_BOOK_ERROR_PROTOCOL_NOT_SUPPORTED,<br/>
 * 12 - E_BOOK_ERROR_CANCELLED,<br/>
 * 13 - E_BOOK_ERROR_COULD_NOT_CANCEL,<br/>
 * 14 - E_BOOK_ERROR_AUTHENTICATION_FAILED,<br/>
 * 15 - E_BOOK_ERROR_AUTHENTICATION_REQUIRED,<br/>
 * 16 - E_BOOK_ERROR_TLS_NOT_AVAILABLE,<br/>
 * 17 - E_BOOK_ERROR_CORBA_EXCEPTION,<br/>
 * 18 - E_BOOK_ERROR_NO_SUCH_SOURCE,<br/>
 * 19 - E_BOOK_ERROR_OFFLINE_UNAVAILABLE,<br/>
 * 20 - E_BOOK_ERROR_OTHER_ERROR,<br/>
 * 21 - E_BOOK_ERROR_INVALID_SERVER_VERSION<br/>
 */
public class EBook extends EDSObject
{
    /**
     * Construct a new Ebook using the handle returned by a call to the native layer.
     * @param handle
     *              The handle (which is retrieved from a native layer) which represents an EBook
     */
    public EBook(Handle handle)
    {
        super(handle);
    }

    protected static EBook getEBook(Handle handle)
    {
        if (handle == null)
            return null;

        EBook obj = (EBook) GObject.getGObjectFromHandle(handle);
        return obj == null ? new EBook(handle) : obj;
    }

    /**
     * Removes the backing data for this EBook. For example, with the file backend this deletes the database file.
     * You cannot get it back!
     *
     * @return true
     *              if the EBook was removed, false otherwise
     * @throws JGException
     */
    public boolean remove() throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_remove(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Gets a list of fields that can be stored for contacts in this book. Other fields may be discarded.
     * @return A String array of supported fields
     * @throws JGException
     */
    public String[] getSupportedFields() throws JGException
    {
        Handle error = getNullHandle();
        String[] result = e_book_get_supported_fields(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Queries the book for the list of authentication methods it supports.
     * @return A String arry of supported authentication methods
     * @throws JGException
     */
    public String[] getSupportedAuthMethods() throws JGException
    {
        Handle error = getNullHandle();
        String[] result = e_book_get_supported_auth_methods(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Authenticates user with password.
     * @param user
     *              The Username
     * @param password
     *              The Password
     * @param authMethod
     *              must be one of the authentication methods returned using getSupportedAuthMethods.
     * @return true if authenticated, false otherwise
     * @throws JGException
     */
    public boolean authenticateUser(String user, String password, String authMethod) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_authenticate_user(getHandle(), user, password, authMethod, error);
        checkError(error);
        return result;
    }

    /**
     * @param id
     *              The unique id of the {@link EContact} you wish to retrieve.
     * @return An EContact with the contents of the vcard in address book corresponding to id.
     * @throws JGException
     */
    public EContact getContact(String id) throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_get_contact(getHandle(), id, error);
        checkError(error);
        return EContact.getEContact(handle);
    }

    /**
     * Removes the contact with id from address book.
     * @param id
     *              The unique id of the {@link EContact} to remove from the address book
     * @return true if the contact was removed, false otherwise
     * @throws JGException
     */
    public boolean removeContact(String id) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_remove_contact(getHandle(), id, error);
        checkError(error);
        return result;
    }

    /**
     * Removes the contact from the address book.
     * @param contact
     *              The {@link EContact} to remove
     * @return true if the contcat was removed, false otherwise
     * @throws JGException
     */
    public boolean removeContact(EContact contact) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_remove_contact(getHandle(), contact.getString(EContactStringField.ID), error);
        checkError(error);
        return result;
    }

    /**
     * Removes contacts from the address book.
     * This is always more efficient than calling removeContact(String id) if you have more than one id to remove, as some backends can implement it as a batch request.
     * @param contacts
     *              The contacts to be removed
     * @return true if the contcts were removed, false otherwise
     * @throws JGException
     */
    public boolean removeContacts(EContact[] contacts) throws JGException
    {
        Handle error = getNullHandle();
        String[] contactIds = new String[contacts.length];
        for (int i = 0; i < contactIds.length; i++)
            contactIds[i] = contacts[i].getString(EContactStringField.ID);
        boolean result = e_book_remove_contacts(getHandle(), contactIds, error);
        checkError(error);
        return result;
    }

    /**
     * Adds {@link EContact} to the EBook.
     * @param contact
     *              The {@link EContact} to add
     * @return true if the contact was added, false otherwwise
     * @throws JGException
     */
    public boolean addContact(EContact contact) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_add_contact(getHandle(), contact.getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Applies the changes made to {@link EContact} to the stored version in book.
     * This basically updates a {@link EContact}.
     * @param contact
     *              The {@link EContact} to be updated
     * @return true if the commit/update was successful, false otherwise
     * @throws JGException
     */
    public boolean commitContact(EContact contact) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_commit_contact(getHandle(), contact.getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Query book with query, creating a {@link EBookView} with the fields specified by requestedFields and limited at maxResults records. On an error, a {@link JGException} is thrown with an appropriate internal error and FALSE returned.
     * @param query
     *              A Query to base the EBookView on. All Contacts in the EBookView will match query.
     * @param requestedFields
     *              An array of requested fields that the {@link EBookView} will be aware of
     * @param maxResults
     *              The maximum number of contacts that {@link EBookView} will be aware of
     * @return An {@link EBookView} of EContacts containing requestedFields mathcing query
     * @throws JGException
     */
    public EBookView getBookView(EBookQuery query, EContactField requestedFields[], int maxResults) throws JGException
    {
        Handle error = getNullHandle();
        String[] requestedFieldHandles = new String[requestedFields.length];
        for (int i = 0; i < requestedFieldHandles.length; i++)
            requestedFieldHandles[i] = requestedFields[i].getFieldName();
        Handle handle = e_book_get_book_view(getHandle(), query.getHandle(), requestedFieldHandles, maxResults, error);
        checkError(error);
        return EBookView.getEBookView(handle);
    }

    /**
     * Query book with query, returning a list of contacts which matched.
     * On failure, an exception will be thrown, an error will be set and FALSE returned.
     * @param query
     *              The query to perform on this EBook
     * @return All contacts matching query
     * @throws org.gnu.glib.JGException
     */
    public EContact[] getContacts(EBookQuery query) throws JGException
    {
        Handle error = getNullHandle();
        Handle[] handles = e_book_get_contacts(getHandle(), query.getHandle(), error);
        checkError(error);
        EContact[] result = new EContact[handles.length];
        for (int i = 0; i < handles.length; i++)
            result[i] = EContact.getEContact(handles[i]);
        return result;
    }

    /**
     * Get all the Contacts in an address book.
     * @return All the Contacts in this EBook
     * @throws JGException
     */
    public EContact[] getContacts() throws JGException
    {
        return getContacts(EBookQuery.createEmptyQuery());
    }

    /**
     * Get the URI that this book has loaded.
     * @return This ebooks URI
     */
    public String getUri()
    {
        return e_book_get_uri(getHandle());
    }

    /**
     * Get the list of capabilities which the backend for this address book supports.
     * @return comma seperated list of capabilities
     * @throws JGException
     */
    public String getStaticCapabilities() throws JGException
    {
        Handle error = getNullHandle();
        String result = e_book_get_static_capabilities(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Check to see if the backend for this address book supports the capability.
     * @param capability
     * @return Whether this address book has the capability. True is success, False if failed
     */
    public boolean checkStaticCapability(String capability)
    {
        return e_book_check_static_capability(getHandle(), capability);
    }

    /**
     * Check if this book is writable.
     * @return True if is writable, False if not
     */
    public boolean isWritable()
    {
        return e_book_is_writable(getHandle());
    }

    /**
     * Used to cancel an already running operation on book.
     * This function makes a synchronous CORBA to the backend telling it to cancel the operation.
     * If the operation wasn't cancellable (either transiently or permanently) or had already been completed on the server side, this function will return FALSE, and the operation will continue uncancelled.
     * If the operation could be cancelled, this function will return true.
     * @return True is success, False if failed
     * @throws JGException
     */
    public boolean cancel() throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_cancel(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Get the Contact referring to the user/owner of the address book.
     * @return Returns the {@link EContact} referring to the user of the address book.
     * @throws JGException
     */
    public EContact getSelf() throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_get_self(getHandle(), error);
        checkError(error);
        return EContact.getEContact(handle);
    }

    /**
     * Specify that contact residing in book is the {@link EContact} that refers to the user of the address book.
     * @param contact
     * @return True is success, False if failed
     * @throws JGException
     */
    public boolean setSelf(EContact contact) throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_set_self(getHandle(), contact.getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * All the ESourceGroups and ESources that are in Evolution can be retrieved from {@link ESourceList}
     * @return Return an ESourceList representing the list sources which have been added to Evolution.
     * @throws JGException
     */
    public static ESourceList getAddressBooks() throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_get_addressbooks(error);
        checkError(error);
        return ESourceList.getESourceList(handle);
    }

    /**
     * Open up the EBook. (Authors Note: This should probably be not called twice. Also this will create the appropriate directory struture and files for a Personal EBook if it doesn't already exist)
     * @return True is success, False if failed
     * @throws JGException
     */
    public boolean open() throws JGException
    {
        Handle error = getNullHandle();
        boolean result = e_book_open(getHandle(), error);
        checkError(error);
        return result;
    }

    /**
     * Create a new EBook from an existing or brand new absolute URI. The returned EBook must be opened.
     * @param uri
     * @return An Ebook
     * @throws JGException
     */
    public static EBook getNewFromUri(String uri) throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_new_from_uri(uri, error);
        checkError(error);
        return getEBook(handle);
    }


    /**
     * Creates a new EBook corresponding to the given source. There are only two operations that are valid on this book at this point: open(), and remove().
     * @param eSource
     * @return A new but unopened EBook.
     * @throws JGException
     *  */
    public static EBook createNew(ESource eSource) throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_new(eSource.getHandle(), error);
        checkError(error);
        return getEBook(handle);
    }

    /**
     * Creates an EBook representing the default evolution address book.
     * @return Returns Evolutions's default EBook which must be opened with open()
     * @throws JGException
     */
    public static EBook createNewDefaultAddressbook() throws JGException
    {
        Handle error = getNullHandle();
        Handle handle = e_book_new_default_addressbook(error);
        checkError(error);
        return getEBook(handle);
    }

    private static void checkError(Handle error) throws JGException
    {
        if (!error.isNull())
            throw new JGException(new Error(error));
    }


    native static private Handle e_book_new(Handle handle, Handle error);

    native static private Handle e_book_new_from_uri(String uri, Handle error);

    native static private boolean e_book_remove(Handle handle, Handle error);

    native static private String[] e_book_get_supported_fields(Handle handle, Handle error);

    //SC: maybe remove this or get to the bottom of it hanging / blocking.
    native static private String[] e_book_get_supported_auth_methods(Handle handle, Handle error);

    native static private boolean e_book_authenticate_user(Handle handle, String user, String password, String authMethod, Handle error);

    native static private Handle e_book_get_contact(Handle handle, String id, Handle error);

    native static private boolean e_book_remove_contact(Handle handle, String uid, Handle error);

    native static private boolean e_book_remove_contacts(Handle handle, String[] ids, Handle error);

    native static private boolean e_book_add_contact(Handle handle, Handle contact, Handle error);

    native static private boolean e_book_commit_contact(Handle handle, Handle contact, Handle error);

    native static private Handle e_book_get_book_view(Handle handle, Handle query, String[] requestedFields, int maxResults, Handle error);

    native static private Handle[] e_book_get_contacts(Handle handle, Handle query, Handle error);

    native static private String e_book_get_uri(Handle handle);

    native static private String e_book_get_static_capabilities(Handle handle, Handle error);

    native static private boolean e_book_check_static_capability(Handle handle, String capability);

    native static private boolean e_book_is_writable(Handle handle);

    native static private boolean e_book_cancel(Handle handle, Handle error);

    native static private Handle e_book_get_self(Handle handle, Handle error);

    native static private boolean e_book_set_self(Handle handle, Handle contact, Handle error);

    native static private Handle e_book_get_addressbooks(Handle error);

    native static private boolean e_book_open(Handle handle, Handle error);

    native static private Handle e_book_new_default_addressbook(Handle error);

//    native static private boolean e_book_is_self(Handle handle, Handle contact);
//    native static private void e_book_free_change_list(GList *change_list);
//    native static private Handle[] e_book_get_changes(Handle handle, String changeId, Handle error);
}
