/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.tsfile.fileSystem.fsFactory;

import org.apache.tsfile.utils.NoSyncBufferedInputStream;
import org.apache.tsfile.utils.NoSyncBufferedOutputStream;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.Files;

public class LocalFSFactory implements FSFactory {

  private static final Logger logger = LoggerFactory.getLogger(LocalFSFactory.class);

  @Override
  public File getFileWithParent(String pathname) {
    File res = new File(pathname);
    if (!res.exists()) {
      res.getParentFile().mkdirs();
    }
    return res;
  }

  @Override
  public File getFile(String pathname) {
    return new File(pathname);
  }

  @Override
  public File getFile(String parent, String child) {
    return new File(parent, child);
  }

  @Override
  public File getFile(File parent, String child) {
    return new File(parent, child);
  }

  @Override
  public File getFile(URI uri) {

    return new File(uri);
  }

  @Override
  public BufferedReader getBufferedReader(String filePath) {
    try {
      return new BufferedReader(new FileReader(filePath));
    } catch (IOException e) {
      logger.error("Failed to get buffered reader for {}. ", filePath, e);
      return null;
    }
  }

  @Override
  public BufferedWriter getBufferedWriter(String filePath, boolean append) {
    try {
      return new BufferedWriter(new FileWriter(filePath, append));
    } catch (IOException e) {
      logger.error("Failed to get buffered writer for {}. ", filePath, e);
      return null;
    }
  }

  @Override
  public InputStream getBufferedInputStream(String filePath) {
    try {
      return new NoSyncBufferedInputStream(new FileInputStream(filePath));
    } catch (IOException e) {
      logger.error("Failed to get buffered input stream for {}. ", filePath, e);
      return null;
    }
  }

  @Override
  public OutputStream getBufferedOutputStream(String filePath) {
    try {
      return new NoSyncBufferedOutputStream(new FileOutputStream(filePath));
    } catch (IOException e) {
      logger.error("Failed to get buffered output stream for {}. ", filePath, e);
      return null;
    }
  }

  @Override
  public void moveFile(File srcFile, File destFile) throws IOException {
    FileUtils.moveFile(srcFile, destFile);
  }

  @Override
  public void copyFile(File srcFile, File destFile) throws IOException {
    FileUtils.copyFile(srcFile, destFile);
  }

  @Override
  public File[] listFilesBySuffix(String fileFolder, String suffix) {
    return new File(fileFolder).listFiles(file -> file.getName().endsWith(suffix));
  }

  @Override
  public File[] listFilesByPrefix(String fileFolder, String prefix) {
    return new File(fileFolder).listFiles(file -> file.getName().startsWith(prefix));
  }

  @Override
  public boolean deleteIfExists(File file) throws IOException {
    return Files.deleteIfExists(file.toPath());
  }

  @Override
  public void deleteDirectory(String dir) throws IOException {
    FileUtils.deleteDirectory(new File(dir));
  }
}
