﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2Request.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/ListTenantResourcesFilterKey.h>

#include <utility>

namespace Aws {
namespace SESV2 {
namespace Model {

/**
 * <p>Represents a request to list resources associated with a specific
 * tenant.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/ListTenantResourcesRequest">AWS
 * API Reference</a></p>
 */
class ListTenantResourcesRequest : public SESV2Request {
 public:
  AWS_SESV2_API ListTenantResourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTenantResources"; }

  AWS_SESV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the tenant to list resources for.</p>
   */
  inline const Aws::String& GetTenantName() const { return m_tenantName; }
  inline bool TenantNameHasBeenSet() const { return m_tenantNameHasBeenSet; }
  template <typename TenantNameT = Aws::String>
  void SetTenantName(TenantNameT&& value) {
    m_tenantNameHasBeenSet = true;
    m_tenantName = std::forward<TenantNameT>(value);
  }
  template <typename TenantNameT = Aws::String>
  ListTenantResourcesRequest& WithTenantName(TenantNameT&& value) {
    SetTenantName(std::forward<TenantNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of filter keys and values for filtering the list of tenant resources.
   * Currently, the only supported filter key is <code>RESOURCE_TYPE</code>.</p>
   */
  inline const Aws::Map<ListTenantResourcesFilterKey, Aws::String>& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Aws::Map<ListTenantResourcesFilterKey, Aws::String>>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Aws::Map<ListTenantResourcesFilterKey, Aws::String>>
  ListTenantResourcesRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  inline ListTenantResourcesRequest& AddFilter(ListTenantResourcesFilterKey key, Aws::String value) {
    m_filterHasBeenSet = true;
    m_filter.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of results to show in a single call to
   * <code>ListTenantResources</code>. If the number of results is larger than the
   * number you specified in this parameter, then the response includes a
   * <code>NextToken</code> element, which you can use to obtain additional
   * results.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline ListTenantResourcesRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token returned from a previous call to <code>ListTenantResources</code> to
   * indicate the position in the list of tenant resources.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTenantResourcesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_tenantName;
  bool m_tenantNameHasBeenSet = false;

  Aws::Map<ListTenantResourcesFilterKey, Aws::String> m_filter;
  bool m_filterHasBeenSet = false;

  int m_pageSize{0};
  bool m_pageSizeHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
